#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "https.h"
#include "util.h"
#include "youtube.h"

#define OUT(s) (fputs((s), stdout))
#define OUTTEXT(s) gophertext(stdout, s, strlen(s));
#define OUTLINK(s) gophertext(stdout, s, strlen(s));

static const char *host = "127.0.0.1", *port = "70", *requestpath = "/";

void
line(int _type, const char *username, const char *selector)
{
	putchar(_type);
	OUTTEXT(username);
	putchar('\t');
	OUTLINK(selector);
	printf("\t%s\t%s\r\n", host, port);
}

void
error(const char *s)
{
	line('3', s, "");
}

void
info(const char *s)
{
	line('i', s, "");
}

void
header(void)
{
}

void
footer(void)
{
	fputs(".\r\n", stdout);
}

int
render_search(struct search_response *r)
{
	struct item *v;
	size_t i;

	if (pledge("stdio", NULL) == -1)
		exit(1);

	header();

	for (i = 0; i < r->nitems; i++) {
		v = &(r->items[i]);

		if (i > 0) {
			info("");
			info("");
		}
		if (v->id[0])
			putchar('h');
		else
			putchar('i');

		switch (v->linktype) {
		case Channel:
			OUT("[Channel] ");
			OUTTEXT(v->channeltitle);
			break;
		case Movie:
			OUT("[Movie] ");
			OUTTEXT(v->title);
			break;
		case Playlist:
			OUT("[Playlist] ");
			OUTTEXT(v->title);
			break;
		default:
			OUTTEXT(v->title);
			break;
		}

		OUT("\t");
		if (v->id[0]) {
			OUT("URL:https://www.youtube.com/embed/");
			OUTLINK(v->id);
		}
		printf("\t%s\t%s\r\n", host, port);

		if (v->id[0]) {
			printf("1Details\t%s?v=", requestpath);
			OUTLINK(v->id);
			printf("\t%s\t%s\r\n", host, port);
		}

		if (v->channelid[0]) {
			OUT("1");
			OUTTEXT(v->channeltitle);
			printf("\t%s?c=", requestpath);
			OUTLINK(v->channelid);
			printf("\t%s\t%s\r\n", host, port);
		} else if (v->userid[0]) {
			OUT("1");
			OUTTEXT(v->channeltitle);
			printf("\t%s?u=", requestpath);
			OUTLINK(v->userid);
			printf("\t%s\t%s\r\n", host, port);
		} else if (v->channeltitle[0]) {
			OUT("i");
			OUTTEXT(v->channeltitle);
			printf("\t%s\t%s\t%s\r\n", "", host, port);
		}

		if (v->channelid[0] || v->userid[0]) {
			OUT("hAtom feed of ");
			OUTTEXT(v->channeltitle);
			OUT("\t");
			OUTLINK("URL:https://www.youtube.com/feeds/videos.xml?");
			if (v->channelid[0]) {
				OUT("channel_id=");
				OUTLINK(v->channelid);
			} else if (v->userid[0]) {
				OUT("user=");
				OUTLINK(v->userid);
			}
			printf("\t%s\t%s\r\n", host, port);
		}
		if (v->duration[0]) {
			OUT("iDuration:      " );
			OUTTEXT(v->duration);
			printf("\t%s\t%s\t%s\r\n", "", host, port);
		}
		if (v->publishedat[0]) {
			OUT("iPublished:     ");
			OUTTEXT(v->publishedat);
			printf("\t%s\t%s\t%s\r\n", "", host, port);
		}
		if (v->viewcount[0]) {
			OUT("iViews:         ");
			printnumsep(v->viewcount);
			printf("\t%s\t%s\t%s\r\n", "", host, port);
		}
	}
	footer();

	return 0;
}
int
render_video(struct video_response *r)
{
	char buf[256];
	const char *s, *e;
	/* softwrap: try to wrap on word boundary, otherwise hardwrap */
	int i, sw = 72, hw = 9999;

	if (pledge("stdio", NULL) == -1)
		exit(1);

	header();

	info("");

	OUT("hTitle:     ");
	OUTTEXT(r->title);
	OUT("\tURL:https://www.youtube.com/embed/");
        OUTTEXT(r->id);
	printf("\t%s\t%s\r\n", host, port);

	OUT("iURL:       ");
	OUT("https://www.youtube.com/embed/");
        OUTTEXT(r->id);
	printf("\t%s\t%s\t%s\r\n", "", host, port);

	if (r->lengthseconds > 0) {
		OUT("iLength:    ");
		if (durationstr(r->lengthseconds, buf, sizeof(buf)) < sizeof(buf))
			OUTTEXT(buf);
		printf("\t%s\t%s\t%s\r\n", "", host, port);
	}

	OUT("hThumbnail: https://i.ytimg.com/vi/");
	OUTTEXT(r->id);
	OUT("/hqdefault.jpg\tURL:https://i.ytimg.com/vi/");
	OUTLINK(r->id);
	printf("/hqdefault.jpg\t%s\t%s\r\n", host, port);

	if (r->author[0]) {
		OUT("1Channel:   ");
		OUTTEXT(r->author);
		printf("\t%s?c=%s\t%s\t%s\r\n", requestpath, r->channelid, host, port);

		if (r->channelid[0]) {
			OUT("hAtom feed\tURL:https://www.youtube.com/feeds/videos.xml?channel_id=");
			OUTTEXT(r->channelid);
			printf("\t%s\t%s\r\n", host, port);
		}
	}

	OUT("iViews:     ");
	snprintf(buf, sizeof(buf), "%ld", r->viewcount);
	printnumsep(buf);
	printf("\t%s\t%s\t%s\r\n", "", host, port);

	if (r->publishdate[0]) {
		OUT("iPublished: ");
		OUTTEXT(r->publishdate);
		printf("\t%s\t%s\t%s\r\n", "", host, port);
	}

	if (r->uploaddate[0]) {
		OUT("iUploaded:  ");
		OUTTEXT(r->uploaddate);
		printf("\t%s\t%s\t%s\r\n", "", host, port);
	}

	if (r->shortdescription[0]) {
		info("");
		info("");

		/* multi-line text */
		i = 0;
		for (s = e = r->shortdescription; ; e++) {
			if (!i)
				putchar('i');
			if (*e == '\n' || *e == '\0' ||
			   i > hw || (i > sw && (*e == ' ' || *e == '\t'))) {
				i = 0;
				gophertext(stdout, s, e - s);
				printf("\t%s\t%s\t%s\r\n", "", host, port);
				if (*e == '\0')
					break;
				s = e + 1;
			} else {
				/* start of rune, wrongly assume 1 rune is 1 column for now */
				if (!((unsigned char)*e & 0x80))
					i++;
			}
		}
	}

	footer();

	return 0;
}

int
main(void)
{
	struct search_response *r = NULL;
	struct video_response *vr = NULL;
	const char *channelid = "", *userid = "", *videoid = "";
	const char *querystring = "", *rawsearch = "", *p;
	char search[1024] = "";

	if (pledge("stdio dns inet rpath unveil", NULL) == -1)
		exit(1);
	if (unveil(TLS_CA_CERT_FILE, "r") == -1 ||
	    unveil(NULL, NULL) == -1)
		exit(1);

	if ((p = getenv("SERVER_NAME")))
		host = p;
	if ((p = getenv("SERVER_PORT")))
		port = p;
	if ((p = getenv("PATH_INFO")))
		requestpath = p;
	if ((p = getenv("QUERY_STRING")))
		querystring = p;

	p = NULL;
	if (querystring[0] == '?')
		querystring++;
	if (querystring[0] == 'c' && querystring[1] == '=') {
		channelid = querystring + 2;
		p = querystring = "";
	} else if (querystring[0] == 'u' && querystring[1] == '=') {
		userid = querystring + 2;
		p = querystring = "";
	} else if (querystring[0] == 'v' && querystring[1] == '=') {
		videoid = querystring + 2;
		p = querystring = "";
	}

	if (querystring[0])
		p = querystring;
	if (!p)
		p = getenv("X_GOPHER_SEARCH"); /* geomyidae */
	if (!p)
		p = getenv("SEARCHREQUEST"); /* gophernicus */
	if (p)
		rawsearch = p;
	if (p && !uriencode(p, search, sizeof(search))) {
		error("Invalid search");
		footer();
		return 0;
	}

	OUT("7Search");
	if (rawsearch[0]) {
		OUT(": ");
		OUT(rawsearch);
	}
	printf("\t%s\t%s\t%s\r\n", requestpath, host, port);

	info("");
	line('1', "<- back to main directory", requestpath);

	if (search[0]) {
		r = youtube_search(search, "", "relevance");
	} else if (channelid[0]) {
		r = youtube_channel_videos(channelid);
	} else if (userid[0]) {
		r = youtube_user_videos(userid);
	} else if (videoid[0]) {
		vr = youtube_video(videoid);
		if (!vr || vr->isfound == 0) {
			error("No video found");
			footer();
			return 0;
		}
		render_video(vr);
		return 0;
	} else {
		footer();
		return 0;
	}
	info("");

	if (!r || r->nitems == 0) {
		error("No videos found");
		footer();
		return 0;
	}

	render_search(r);

	return 0;
}
