#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <wchar.h>

int
uriencode(const char *s, char *buf, size_t bufsiz)
{
	static char hex[] = "0123456789ABCDEF";
	char *d = buf, *e = buf + bufsiz;
	unsigned char c;

	if (!bufsiz)
		return 0;

	for (; *s; ++s) {
		c = (unsigned char)*s;
		if (d + 4 >= e)
			return 0;
		if (c == ' ' || c == '#' || c == '%' || c == '?' || c == '"' ||
		    c == '&' || c == '<' || c <= 0x1f || c >= 0x7f) {
			*d++ = '%';
			*d++ = hex[c >> 4];
			*d++ = hex[c & 0x0f];
		} else {
			*d++ = *s;
		}
	}
	*d = '\0';

	return 1;
}

int
hexdigit(int c)
{
	if (c >= '0' && c <= '9')
		return c - '0';
	else if (c >= 'A' && c <= 'F')
		return c - 'A' + 10;
	else if (c >= 'a' && c <= 'f')
		return c - 'a' + 10;

	return 0;
}

/* decode until NUL separator or end of "key". */
int
decodeparam(char *buf, size_t bufsiz, const char *s)
{
	size_t i;

	if (!bufsiz)
		return -1;

	for (i = 0; *s && *s != '&'; s++) {
		switch (*s) {
		case '%':
			if (i + 3 >= bufsiz)
				return -1;
			if (!isxdigit((unsigned char)*(s+1)) ||
			    !isxdigit((unsigned char)*(s+2)))
				return -1;
			buf[i++] = hexdigit(*(s+1)) * 16 + hexdigit(*(s+2));
			s += 2;
			break;
		case '+':
			if (i + 1 >= bufsiz)
				return -1;
			buf[i++] = ' ';
			break;
		default:
			if (i + 1 >= bufsiz)
				return -1;
			buf[i++] = *s;
			break;
		}
	}
	buf[i] = '\0';

	return i;
}

char *
getparam(const char *query, const char *s)
{
	const char *p, *last = NULL;
	size_t len;

	len = strlen(s);
	for (p = query; (p = strstr(p, s)); p += len) {
		if (p[len] == '=' && (p == query || p[-1] == '&' || p[-1] == '?'))
			last = p + len + 1;
	}

	return (char *)last;
}

int
friendlytime(time_t now, time_t t)
{
	long long d = now - t;

	if (d < 60) {
		printf("just now");
	} else if (d < 3600) {
		printf("%lld minutes ago", d / 60);
	} else if (d <= 24*3600) {
		printf("%lld hours ago", d / 3600);
	} else {
		return 0;
	}
	return 1;
}

/* Escape characters below as HTML 2.0 / XML 1.0. */
void
xmlencode(const char *s)
{
	for (; *s; s++) {
		switch(*s) {
		case '<':  fputs("&lt;", stdout);   break;
		case '>':  fputs("&gt;", stdout);   break;
		case '\'': fputs("&#39;", stdout);  break;
		case '&':  fputs("&amp;", stdout);  break;
		case '"':  fputs("&quot;", stdout); break;
		default:   putchar(*s);
		}
	}
}

/* format `len' columns of characters. If string is shorter pad the rest
 * with characters `pad`. */
int
utf8pad(char *buf, size_t bufsiz, const char *s, size_t len, int pad)
{
	wchar_t wc;
	size_t col = 0, i, slen, siz = 0;
	int rl, w;

	if (!len)
		return -1;

	slen = strlen(s);
	for (i = 0; i < slen; i += rl) {
		if ((rl = mbtowc(&wc, &s[i], slen - i < 4 ? slen - i : 4)) <= 0)
			break;
		if ((w = wcwidth(wc)) == -1)
			continue;
		if (col + w > len || (col + w == len && s[i + rl])) {
			if (siz + 4 >= bufsiz)
				return -1;
			memcpy(&buf[siz], "\xe2\x80\xa6", 3);
			siz += 3;
			if (col + w == len && w > 1)
				buf[siz++] = pad;
			buf[siz] = '\0';
			return 0;
		}
		if (siz + rl + 1 >= bufsiz)
			return -1;
		memcpy(&buf[siz], &s[i], rl);
		col += w;
		siz += rl;
		buf[siz] = '\0';
	}

	len -= col;
	if (siz + len + 1 >= bufsiz)
		return -1;
	memset(&buf[siz], pad, len);
	siz += len;
	buf[siz] = '\0';

	return 0;
}

/* Escape characters in gopher, CR and LF are ignored */
void
gophertext(FILE *fp, const char *s, size_t len)
{
	size_t i;

	for (i = 0; *s && i < len; s++, i++) {
		switch (*s) {
		case '\r': /* ignore CR */
		case '\n': /* ignore LF */
			break;
		case '\t':
			fputs("        ", fp);
			break;
		default:
			fputc(*s, fp);
			break;
		}
	}
}

/* seconds to duration string: "%H:%M:%S" or "%H:%M:%S" */
int
durationstr(long secs, char *buf, size_t bufsiz)
{
	int h, m, s, r;

	h = secs / 3600;
	m = secs / 60;
	s = secs;
	if (h <= 0)
		r = snprintf(buf, bufsiz, "%02d:%02d", m % 60, s % 60);
	else
		r = snprintf(buf, bufsiz, "%d:%02d:%02d", h, m % 60, s % 60);

	return r;
}

/* print views with thousand separators */
void
printnumsep(const char *s)
{
	const char *p;
	int ndigits = 0;

	/* first count all digits */
	for (p = s; *p; p++)
		if (*p >= '0' && *p <= '9')
			ndigits++;

	for (p = s; *p; p++) {
		if (!(*p >= '0' && *p <= '9'))
			continue;

		putchar(*p);
		ndigits--;

		/* show separator on every 3 digits and when there are
		   digits remaining */
		if ((ndigits % 3) == 0 && ndigits > 0)
			putchar(',');
	}
}
