#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "https.h"
#include "util.h"
#include "youtube.h"

#define OUT(s) fputs((s), stdout)
#define OUTESCAPE(s) printescape((s))

/* print: ignore control-characters */
void
printescape(const char *s)
{
	for (; *s; ++s)
		if (!iscntrl((unsigned char)*s))
			fputc(*s, stdout);
}

void
printescape_multiline(const char *s, const char *indent)
{
	int i = 0;

	for (; *s; ++s) {
		if (!i)
			fputs(indent, stdout);

		if (*s == '\n') {
			i = 0;
			fputc(*s, stdout);
		} else if (!iscntrl((unsigned char)*s)) {
			fputc(*s, stdout);
			i = 1;
		}
	}
}

int
render_search_tsv(struct search_response *r)
{
	struct item *v;
	size_t i;

	for (i = 0; i < r->nitems; i++) {
		v = &(r->items[i]);

		OUTESCAPE(v->id);
		OUT("\t");
		if (v->id[0]) {
			OUT("https://www.youtube.com/embed/");
                        OUTESCAPE(v->id);
		}
		OUT("\t");
		OUTESCAPE(v->title);
		OUT("\t");
		OUTESCAPE(v->publishedat);
		OUT("\t");
		OUTESCAPE(v->viewcount);
		OUT("\t");
		OUTESCAPE(v->duration);
		OUT("\t");
		switch (v->linktype) {
		case Channel:  OUT("channel"); break;
		case Movie:    OUT("movie"); break;
		case Playlist: OUT("playlist"); break;
		default:       break;
		}
		OUT("\t");
		OUTESCAPE(v->channelid);
		OUT("\t");
		OUTESCAPE(v->channeltitle);
		OUT("\t");
		OUTESCAPE(v->userid);
/*		OUT("\t");
		OUTESCAPE(v->shortdescription); */ /* TODO: escape newlines etc */
		OUT("\n");
	}

	return 0;
}

int
render_search(struct search_response *r)
{
	struct item *v;
	size_t i;

	for (i = 0; i < r->nitems; i++) {
		v = &(r->items[i]);

		switch (v->linktype) {
		case Channel:
			OUT("Channel:   ");
			OUTESCAPE(v->channeltitle);
			break;
		case Movie:
			OUT("Movie:     ");
			OUTESCAPE(v->title);
			break;
		case Playlist:
			OUT("Playlist:  ");
			OUTESCAPE(v->title);
			break;
		default:
			OUT("           ");
			OUTESCAPE(v->title);
			break;
		}
		if (v->duration[0]) {
			OUT(" [");
			OUTESCAPE(v->duration);
			OUT("]");
		}
		OUT("\n");

		if (v->id[0]) {
			OUT("URL:       https://www.youtube.com/embed/");
                        OUTESCAPE(v->id);
			OUT("\n");
		}

		if (v->channelid[0] || v->userid[0]) {
			OUT("Channel:   ");
			OUTESCAPE(v->channeltitle);
			OUT(": https://www.youtube.com/feeds/videos.xml?");
			if (v->channelid[0]) {
				OUT("channel_id=");
				OUTESCAPE(v->channelid);
			} else if (v->userid[0]) {
				OUT("user=");
				OUTESCAPE(v->userid);
			}
			OUT("\n");
		}
		if (v->publishedat[0]) {
			OUT("Published: ");
			OUTESCAPE(v->publishedat);
			OUT("\n");
		}
		if (v->viewcount[0]) {
			OUT("Views:     ");
			printnumsep(v->viewcount);
			OUT("\n");
		}
		OUT("\n");
	}

	return 0;
}

int
render_video(struct video_response *r)
{
	struct video_format *f;
	char buf[256];
	int i;

	OUT("URL:       ");
	OUT("https://www.youtube.com/embed/");
        OUTESCAPE(r->id);
	OUT("\n");

	OUT("Title:     ");
	OUTESCAPE(r->title);
	OUT("\n");

	if (r->lengthseconds > 0) {
		OUT("Length:    ");
		if (durationstr(r->lengthseconds, buf, sizeof(buf)) < sizeof(buf))
			OUTESCAPE(buf);
		OUT("\n");
	}

	OUT("Views:     ");
	snprintf(buf, sizeof(buf), "%ld", r->viewcount);
	printnumsep(buf);
	OUT("\n");

	if (r->publishdate[0]) {
		OUT("Published: ");
		OUTESCAPE(r->publishdate);
		OUT("\n");
	}

	if (r->uploaddate[0]) {
		OUT("Uploaded:  ");
		OUTESCAPE(r->uploaddate);
		OUT("\n");
	}

	if (r->author[0]) {
		OUT("Channel:   ");
		OUTESCAPE(r->author);
		if (r->channelid[0]) {
			OUT(": https://www.youtube.com/feeds/videos.xml?channel_id=");
			OUTESCAPE(r->channelid);
		}
		OUT("\n");
	}

	if (r->shortdescription[0]) {
		OUT("Description:\n\n");
		printescape_multiline(r->shortdescription, "");
		OUT("\n");
	}

	if (r->nformats == 0)
		return 0;

	OUT("\n\nFormats:\n\n");

	/* links expiration */
	if (r->expiresinseconds > 0) {
		OUT("Expires in ");
		printf("%ld", r->expiresinseconds);
		OUT(" seconds\n");
	}

	for (i = 0; i < r->nformats; i++) {
		f = &(r->formats[i]);

#if 0
		if (f->width < 1280 || f->height < 720)
			continue;
#endif

#if 0
		OUT("Last modified: ");
		OUTESCAPE(f->lastmodified);
		OUT("\n");
#endif

		if (f->url[0]) {
			OUT("URL:     ");
			OUTESCAPE(f->url);
			OUT("\n");
		}

		/* encrypted stream */
		if (f->signaturecipher[0]) {
			OUT("Signature: ");
			OUTESCAPE(f->signaturecipher);
			OUT("\n");
		}

		if (f->mimetype[0]) {
			OUT("Mime:    ");
			OUTESCAPE(f->mimetype);
			OUT("\n");
		}

		if (f->itag > 0) {
			OUT("itag:    ");
			printf("%ld\n", f->itag);
		}

		if (f->qualitylabel[0]) {
			OUT("Quality: ");
			OUTESCAPE(f->qualitylabel);
		} else if (f->quality[0]) {
			OUT("Quality: ");
			OUTESCAPE(f->quality);
		}

		if (f->width > 0) {
			OUT(", ");
			printf("%ld", f->width);
			OUT("x");
			printf("%ld", f->height);
			OUT("");
		}
		if (f->fps > 0) {
			OUT(", ");
			printf("%ld", f->fps);
			OUT(" FPS");
		}
		OUT("\n");

		if (f->bitrate > 0) {
			OUT("Bitrate: ");
			printf("%ld", f->bitrate);
			if (f->averagebitrate > 0)
				printf(", average: %ld", f->averagebitrate);
			OUT("\n");
		}

		if (f->contentlength > 0) {
			OUT("Size:    ");
			printf("%lld bytes (%.2f MB)\n", f->contentlength, f->contentlength / 1024.0 / 1024.0);
		}

		if (f->audiochannels > 0 || f->audiosamplerate) {
			OUT("Audio:   ");
			if (f->audiochannels > 0)
				printf("%ld channels", f->audiochannels);
			if (f->audiosamplerate > 0) {
				if (f->audiochannels > 0)
					OUT(", ");
				printf("%ld sample rate", f->audiosamplerate);
			}
			OUT("\n");
		}

		OUT("\n");
	}

	return 0;
}

static void
usage(const char *argv0)
{
	fprintf(stderr, "usage: %s [-t] <keyword> | <-c channelid> | <-u user> | <-i videoid> | [-o relevance|views|rating]\n", argv0);
	exit(1);
}

int
main(int argc, char *argv[])
{
	struct search_response *r = NULL;
	struct video_response *vr = NULL;
	char search[1024];
	const char *keywords = NULL, *channelid = NULL, *user = NULL, *videoid = NULL;
	const char *order = "relevance";
	int i, usetsv = 0;

	if (pledge("stdio dns inet rpath unveil", NULL) == -1) {
		fprintf(stderr, "pledge: %s\n", strerror(errno));
		exit(1);
	}

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
			case 'c':
				if (i + 1 >= argc)
					usage(argv[0]);
				channelid = argv[i + 1];
				i++;
				break;
			case 'i':
				if (i + 1 >= argc)
					usage(argv[0]);
				videoid = argv[i + 1];
				i++;
				break;
			case 'o':
				if (i + 1 >= argc)
					usage(argv[0]);
				order = argv[i + 1];
				i++;
				break;
			case 'u':
				if (i + 1 >= argc)
					usage(argv[0]);
				user = argv[i + 1];
				i++;
				break;
			case 't':
				usetsv = 1;
				break;
			default:
				usage(argv[0]);
			}
			continue;
		}
		keywords = argv[i];
	}

	if (unveil(TLS_CA_CERT_FILE, "r") == -1) {
		fprintf(stderr, "unveil: %s\n", strerror(errno));
		exit(1);
	}
	if (unveil(NULL, NULL) == -1) {
		fprintf(stderr, "unveil: %s\n", strerror(errno));
		exit(1);
	}

	if (argc < 2 || !argv[1][0])
		usage(argv[0]);

	/* check order options */
	if (strcmp(order, "relevance") &&
	    strcmp(order, "views") &&
	    strcmp(order, "rating"))
		usage(argv[0]);

	if (channelid) {
		r = youtube_channel_videos(channelid);
	} else if (user) {
		r = youtube_user_videos(user);
	} else if (videoid) {
		vr = youtube_video(videoid);
		if (!vr || vr->isfound == 0) {
			OUT("No video found\n");
			exit(1);
		}
		render_video(vr);
		return 0;
	} else if (keywords) {
		if (!uriencode(keywords, search, sizeof(search)))
			usage(argv[0]);
		r = youtube_search(search, "", order);
	}
	if (!r || r->nitems == 0) {
		OUT("No videos found\n");
		exit(1);
	}

	if (pledge("stdio", NULL) == -1) {
		fprintf(stderr, "pledge: %s\n", strerror(errno));
		exit(1);
	}

	if (usetsv)
		render_search_tsv(r);
	else
		render_search(r);

	return 0;
}
