#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "https.h"
#include "json.h"
#include "twitch.h"
#include "util.h"

#ifndef TWITCH_API_KEY
#error "make sure set a TWITCH_API_KEY in twitch.c"
#define TWITCH_API_KEY "API key here"
#endif
static const char *twitch_headers = "Client-ID: " TWITCH_API_KEY "\r\n";

static char *
twitch_request(const char *path)
{
	return request("api.twitch.tv", path, twitch_headers);
}

/* unmarshal JSON response, skip HTTP headers */
int
json_unmarshal(const char *data,
	void (*cb)(struct json_node *, size_t, const char *, size_t, void *),
	void *pp)
{
	const char *s;

	/* strip/skip header part */
	if (!(s = strstr(data, "\r\n\r\n"))) {
		fprintf(stderr, "error parsing HTTP response header\n");
		return -1; /* invalid response */
	}
	s += strlen("\r\n\r\n");

	/* parse */
	if (parsejson(s, strlen(s), cb, pp) < 0) {
		fprintf(stderr, "error parsing JSON\n");
		return -1;
	}

	return 0;
}

char *
twitch_games_bygameids_data(const char *param)
{
	char path[4096];
	int r;

	r = snprintf(path, sizeof(path), "/helix/games?%s", param);
	if (r < 0 || (size_t)r >= sizeof(path))
		return NULL;

	return twitch_request(path);
}

char *
twitch_users_byuserids_data(const char *param)
{
	char path[4096];
	int r;

	r = snprintf(path, sizeof(path), "/helix/users?%s", param);
	if (r < 0 || (size_t)r >= sizeof(path))
		return NULL;

	return twitch_request(path);
}

char *
twitch_users_bylogin_data(const char *login)
{
	char path[256];
	int r;

	r = snprintf(path, sizeof(path), "/helix/users?login=%s", login);
	if (r < 0 || (size_t)r >= sizeof(path))
		return NULL;

	return twitch_request(path);
}

char *
twitch_videos_byuserid_data(const char *user_id)
{
	char path[128];
	int r;

	r = snprintf(path, sizeof(path), "/helix/videos?first=100&user_id=%s",
	         user_id);
	if (r < 0 || (size_t)r >= sizeof(path))
		return NULL;

	return twitch_request(path);
}

char *
twitch_streams_data(void)
{
	return twitch_request("/helix/streams?first=100");
}

char *
twitch_streams_game_data(const char *game_id)
{
	char path[64];
	int r;

	r = snprintf(path, sizeof(path), "/helix/streams?first=100&game_id=%s",
	             game_id);
	if (r < 0 || (size_t)r >= sizeof(path))
		return NULL;

	return twitch_request(path);
}

char *
twitch_games_top_data(void)
{
	return twitch_request("/helix/games/top?first=100");
}

void
twitch_games_processnode(struct json_node *nodes, size_t depth, const char *value, size_t valuelen,
	void *pp)
{
	struct games_response *r = (struct games_response *)pp;
	struct game *item;

	if (r->nitems > MAX_ITEMS)
		return;

	/* new item */
	if (depth == 3 &&
	    nodes[0].type == TYPE_OBJECT &&
	    nodes[1].type == TYPE_ARRAY &&
	    nodes[2].type == TYPE_OBJECT &&
	    !strcmp(nodes[1].name, "data")) {
		r->nitems++;
		return;
	}

	if (r->nitems == 0)
		return;
	item = &(r->data[r->nitems - 1]);

	if (depth == 4 &&
	    nodes[0].type == TYPE_OBJECT &&
	    nodes[1].type == TYPE_ARRAY &&
	    nodes[2].type == TYPE_OBJECT &&
	    nodes[3].type == TYPE_STRING &&
	    !strcmp(nodes[1].name, "data")) {
		if (!strcmp(nodes[3].name, "id"))
			strlcpy(item->id, value, sizeof(item->id));
		else if (!strcmp(nodes[3].name, "name"))
			strlcpy(item->name, value, sizeof(item->name));
	}
}

struct games_response *
twitch_games_top(void)
{
	struct games_response *r;
	char *data;

	if ((data = twitch_games_top_data()) == NULL) {
		fprintf(stderr, "%s\n", __func__);
		return NULL;
	}

	if (!(r = calloc(1, sizeof(*r)))) {
		fprintf(stderr, "calloc\n");
		return NULL;
	}
	if (json_unmarshal(data, twitch_games_processnode, r) == -1) {
		free(r);
		r = NULL;
	}
	free(data);

	return r;
}

struct games_response *
twitch_games_bygameids(const char *param)
{
	struct games_response *r;
	char *data;

	if ((data = twitch_games_bygameids_data(param)) == NULL) {
		fprintf(stderr, "%s\n", __func__);
		return NULL;
	}

	if (!(r = calloc(1, sizeof(*r)))) {
		fprintf(stderr, "calloc\n");
		return NULL;
	}
	if (json_unmarshal(data, twitch_games_processnode, r) == -1) {
		free(r);
		r = NULL;
	}
	free(data);

	return r;
}

void
twitch_streams_processnode(struct json_node *nodes, size_t depth, const char *value, size_t valuelen,
	void *pp)
{
	struct streams_response *r = (struct streams_response *)pp;
	struct stream *item;

	if (r->nitems > MAX_ITEMS)
		return;
	item = &(r->data[r->nitems]);

	/* new item */
	if (depth == 3 &&
	    nodes[0].type == TYPE_OBJECT &&
	    nodes[1].type == TYPE_ARRAY &&
	    nodes[2].type == TYPE_OBJECT &&
	    !strcmp(nodes[1].name, "data")) {
		r->nitems++;
		return;
	}

	if (r->nitems == 0)
		return;
	item = &(r->data[r->nitems - 1]);

	if (depth == 4 &&
	    nodes[0].type == TYPE_OBJECT &&
	    nodes[1].type == TYPE_ARRAY &&
	    nodes[2].type == TYPE_OBJECT &&
	    !strcmp(nodes[1].name, "data")) {
		if (nodes[3].type == TYPE_STRING) {
			if (!strcmp(nodes[3].name, "id"))
				strlcpy(item->id, value, sizeof(item->id));
			else if (!strcmp(nodes[3].name, "title"))
				strlcpy(item->title, value, sizeof(item->title));
			else if (!strcmp(nodes[3].name, "user_id"))
				strlcpy(item->user_id, value, sizeof(item->user_id));
			else if (!strcmp(nodes[3].name, "user_name"))
				strlcpy(item->user_name, value, sizeof(item->user_name));
			else if (!strcmp(nodes[3].name, "game_id"))
				strlcpy(item->game_id, value, sizeof(item->game_id));
			else if (!strcmp(nodes[3].name, "language"))
				strlcpy(item->language, value, sizeof(item->language));
		} else if (nodes[3].type == TYPE_NUMBER) {
			/* TODO: check? */
			if (!strcmp(nodes[3].name, "viewer_count"))
				item->viewer_count = strtoll(value, NULL, 10);
		}
	}
}

struct streams_response *
twitch_streams_bygame(const char *game_id)
{
	struct streams_response *r;
	char *data;

	if (game_id[0])
		data = twitch_streams_game_data(game_id);
	else
		data = twitch_streams_data();

	if (!(r = calloc(1, sizeof(*r)))) {
		fprintf(stderr, "calloc\n");
		return NULL;
	}
	if (json_unmarshal(data, twitch_streams_processnode, r) == -1) {
		free(r);
		r = NULL;
	}
	free(data);

	return r;
}

struct streams_response *
twitch_streams(void)
{
	return twitch_streams_bygame("");
}

int
ids_cmp(const void *v1, const void *v2)
{
	const char *s1 = *((const char**)v1), *s2 = *((const char **)v2);

	return strcmp(s1, s2);
}

/* fill in games in the streams response */
struct games_response *
twitch_streams_games(struct streams_response *r)
{
	struct games_response *rg;
	char *game_ids[MAX_ITEMS];
	char game_ids_param[4096] = "";
	size_t i, j;

	/* create a list of game_ids, sort them and filter unique */
	for (i = 0; i < r->nitems; i++)
		game_ids[i] = r->data[i].game_id;

	qsort(game_ids, r->nitems, sizeof(*game_ids), ids_cmp);
	for (i = 0; i < r->nitems; i++) {
		if (!game_ids[i][0])
			continue;

		/* first or different than previous */
		if (i && !strcmp(game_ids[i], game_ids[i - 1]))
			continue;

		if (game_ids_param[0])
			strlcat(game_ids_param, "&", sizeof(game_ids_param));

		strlcat(game_ids_param, "id=", sizeof(game_ids_param));
		strlcat(game_ids_param, game_ids[i], sizeof(game_ids_param));
	}

	if ((rg = twitch_games_bygameids(game_ids_param))) {
		for (i = 0; i < r->nitems; i++) {
			for (j = 0; j < rg->nitems; j++) {
				/* match game on game_id */
				if (!strcmp(r->data[i].game_id, rg->data[j].id)) {
					r->data[i].game = &(rg->data[j]);
					break;
				}
			}
		}
	}
	return rg;
}

/* fill in users in the streams response */
struct users_response *
twitch_streams_users(struct streams_response *r)
{
	struct users_response *ru = NULL;
	char *user_ids[MAX_ITEMS];
	char user_ids_param[4096] = "";
	size_t i, j;

	/* create a list of user_ids, sort them and filter unique */
	for (i = 0; i < r->nitems; i++)
		user_ids[i] = r->data[i].user_id;

	qsort(user_ids, r->nitems, sizeof(*user_ids), ids_cmp);
	for (i = 0; i < r->nitems; i++) {
		if (!user_ids[i][0])
			continue;
		/* first or different than previous */
		if (i && !strcmp(user_ids[i], user_ids[i - 1]))
			continue;

		if (user_ids_param[0])
			strlcat(user_ids_param, "&", sizeof(user_ids_param));

		strlcat(user_ids_param, "id=", sizeof(user_ids_param));
		strlcat(user_ids_param, user_ids[i], sizeof(user_ids_param));
	}

	if ((ru = twitch_users_byuserids(user_ids_param))) {
		for (i = 0; i < r->nitems; i++) {
			for (j = 0; j < ru->nitems; j++) {
				/* match user on user_id */
				if (!strcmp(r->data[i].user_id, ru->data[j].id)) {
					r->data[i].user = &(ru->data[j]);
					break;
				}
			}
		}
	}
	return ru;
}

void
twitch_users_processnode(struct json_node *nodes, size_t depth, const char *value, size_t valuelen,
	void *pp)
{
	struct users_response *r = (struct users_response *)pp;
	struct user *item;

	if (r->nitems > MAX_ITEMS)
		return;
	item = &(r->data[r->nitems]);

	/* new item */
	if (depth == 3 &&
	    nodes[0].type == TYPE_OBJECT &&
	    nodes[1].type == TYPE_ARRAY &&
	    nodes[2].type == TYPE_OBJECT &&
	    !strcmp(nodes[1].name, "data")) {
		r->nitems++;
		return;
	}

	if (r->nitems == 0)
		return;
	item = &(r->data[r->nitems - 1]);

	if (depth == 4 &&
	    nodes[0].type == TYPE_OBJECT &&
	    nodes[1].type == TYPE_ARRAY &&
	    nodes[2].type == TYPE_OBJECT &&
	    !strcmp(nodes[1].name, "data")) {
		if (nodes[3].type == TYPE_STRING) {
			if (!strcmp(nodes[3].name, "id"))
				strlcpy(item->id, value, sizeof(item->id));
			else if (!strcmp(nodes[3].name, "login"))
				strlcpy(item->login, value, sizeof(item->login));
			else if (!strcmp(nodes[3].name, "display_name"))
				strlcpy(item->display_name, value, sizeof(item->display_name));
		} else if (nodes[3].type == TYPE_NUMBER) {
			/* TODO: check? */
			if (!strcmp(nodes[3].name, "view_count"))
				item->view_count = strtoll(value, NULL, 10);
		}
	}
}

struct users_response *
twitch_users_byuserids(const char *param)
{
	struct users_response *r;
	char *data;

	if ((data = twitch_users_byuserids_data(param)) == NULL) {
		fprintf(stderr, "%s\n", __func__);
		return NULL;
	}

	if (!(r = calloc(1, sizeof(*r)))) {
		fprintf(stderr, "calloc\n");
		return NULL;
	}
	if (json_unmarshal(data, twitch_users_processnode, r) == -1) {
		free(r);
		r = NULL;
	}
	free(data);

	return r;
}

struct users_response *
twitch_users_bylogin(const char *login)
{
	struct users_response *r;
	char *data;

	if ((data = twitch_users_bylogin_data(login)) == NULL) {
		fprintf(stderr, "%s\n", __func__);
		return NULL;
	}

	if (!(r = calloc(1, sizeof(*r)))) {
		fprintf(stderr, "calloc\n");
		return NULL;
	}
	if (json_unmarshal(data, twitch_users_processnode, r) == -1) {
		free(r);
		r = NULL;
	}
	free(data);

	return r;
}

void
twitch_videos_processnode(struct json_node *nodes, size_t depth, const char *value, size_t valuelen,
	void *pp)
{
	struct videos_response *r = (struct videos_response *)pp;
	struct video *item;

	if (r->nitems > MAX_ITEMS)
		return;
	item = &(r->data[r->nitems]);

	/* new item */
	if (depth == 3 &&
	    nodes[0].type == TYPE_OBJECT &&
	    nodes[1].type == TYPE_ARRAY &&
	    nodes[2].type == TYPE_OBJECT &&
	    !strcmp(nodes[1].name, "data")) {
		r->nitems++;
		return;
	}

	if (r->nitems == 0)
		return;
	item = &(r->data[r->nitems - 1]);

	if (depth == 4 &&
	    nodes[0].type == TYPE_OBJECT &&
	    nodes[1].type == TYPE_ARRAY &&
	    nodes[2].type == TYPE_OBJECT &&
	    !strcmp(nodes[1].name, "data")) {
		if (nodes[3].type == TYPE_STRING) {
			if (!strcmp(nodes[3].name, "id"))
				strlcpy(item->id, value, sizeof(item->id));
			else if (!strcmp(nodes[3].name, "user_id"))
				strlcpy(item->user_id, value, sizeof(item->user_id));
			else if (!strcmp(nodes[3].name, "user_name"))
				strlcpy(item->user_name, value, sizeof(item->user_name));
			else if (!strcmp(nodes[3].name, "title"))
				strlcpy(item->title, value, sizeof(item->title));
			else if (!strcmp(nodes[3].name, "created_at"))
				strlcpy(item->created_at, value, sizeof(item->created_at));
			else if (!strcmp(nodes[3].name, "url"))
				strlcpy(item->url, value, sizeof(item->url));
			else if (!strcmp(nodes[3].name, "duration"))
				strlcpy(item->duration, value, sizeof(item->duration));
		} else if (nodes[3].type == TYPE_NUMBER) {
			/* TODO: check? */
			if (!strcmp(nodes[3].name, "view_count"))
				item->view_count = strtoll(value, NULL, 10);
		}
	}
}

struct videos_response *
twitch_videos_byuserid(const char *user_id)
{
	struct videos_response *r;
	char *data;

	if ((data = twitch_videos_byuserid_data(user_id)) == NULL) {
		fprintf(stderr, "%s\n", __func__);
		return NULL;
	}

	if (!(r = calloc(1, sizeof(*r)))) {
		fprintf(stderr, "calloc\n");
		return NULL;
	}
	if (json_unmarshal(data, twitch_videos_processnode, r) == -1) {
		free(r);
		r = NULL;
	}
	free(data);

	return r;
}
