#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "reddit.h"

#define OUT(s) (fputs((s), stdout))

extern char **environ;

static struct list_response *response;
static time_t now;

/* CGI parameters */
static char rawsearch[4096], search[4096];
static char subreddit[1024], mode[16], slimit[32], order[16];
static long limit = 100;

void
parsecgi(void)
{
	char *query, *p;
	size_t len;

	if (!(query = getenv("QUERY_STRING")))
		query = "";

	/* subreddit: select subreddit */
	if ((p = getparam(query, "subreddit"))) {
		if (decodeparam(subreddit, sizeof(subreddit), p) == -1)
			subreddit[0] = '\0';
	}

	/* order */
	if ((p = getparam(query, "o"))) {
		if (decodeparam(order, sizeof(order), p) == -1 ||
			(strcmp(order, "hot") &&
			strcmp(order, "new") &&
			strcmp(order, "rising") &&
			strcmp(order, "controversial")))
			order[0] = '\0';
	}
	if (!order[0])
		snprintf(order, sizeof(order), "hot");

#if 0
	/* TODO */
	/* page */
	if ((p = getparam(query, "page"))) {
		if (decodeparam(page, sizeof(page), p) == -1)
			page[0] = '\0';
		/* check if it's a number >= 0 and < 100 */
		errno = 0;
		curpage = strtol(page, NULL, 10);
		if (errno || curpage < 0 || curpage > 100) {
			curpage = 1;
			page[0] = '\0';
		}
	}
#endif

	/* limit */
	if ((p = getparam(query, "limit"))) {
		if (decodeparam(slimit, sizeof(slimit), p) == -1)
			slimit[0] = '\0';
		/* check if it's a number > 0 and < 100 */
		errno = 0;
		limit = strtol(slimit, NULL, 10);
		if (errno || limit <= 0 || limit > 100) {
			limit = 100; /* default */
			slimit[0] = '\0';
		}
	}

	/* mode */
	if ((p = getparam(query, "m"))) {
		if (decodeparam(mode, sizeof(mode), p) != -1) {
			/* fixup first character (label) for matching */
			if (mode[0])
				mode[0] = tolower((unsigned char)mode[0]);
			/* allowed themes */
			if (strcmp(mode, "light") &&
			    strcmp(mode, "dark") &&
			    strcmp(mode, "pink") &&
			    strcmp(mode, "templeos"))
				mode[0] = '\0';
		}
	}
	if (!mode[0])
		snprintf(mode, sizeof(mode), "light");

	/* search */
	if ((p = getparam(query, "q"))) {
		if ((len = strcspn(p, "&")) && len + 1 < sizeof(rawsearch)) {
			memcpy(rawsearch, p, len);
			rawsearch[len] = '\0';
		}

		if (decodeparam(search, sizeof(search), p) == -1) {
			OUT("Status: 401 Bad Request\r\n\r\n");
			exit(1);
		}
	}
}

int
render(struct list_response *r)
{
	struct item *items = r->items, *item;
	char tmp[64], timebuf[32], baseurl[256];
	char *start, *end;
	int i;

#if 0
	if (pledge("stdio", NULL) == -1) {
		OUT("Status: 500 Internal Server Error\r\n\r\n");
		exit(1);
	}
#endif

	OUT(
		"Content-Type: text/html; charset=utf-8\r\n\r\n"
		"<!DOCTYPE html>\n<html>\n<head>\n"
		"<meta name=\"referrer\" content=\"no-referrer\" />\n"
		"<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\" />\n"
		"<title>");
		if (r->nitems && subreddit[0]) {
			OUT("r/");
			xmlencode(subreddit);
			OUT(" - Reddit");
		} else {
			OUT("Reddit");
		}
		OUT("</title>\n");
	OUT(
		"<link rel=\"stylesheet\" href=\"css/");
	xmlencode(mode);
	OUT(
		".css\" type=\"text/css\" media=\"screen\" />\n"
		"<link rel=\"icon\" type=\"image/png\" href=\"/favicon.png\" />\n"
		"<meta content=\"width=device-width\" name=\"viewport\" />\n"
		"</head>\n"
	        "<body class=\"search\">\n"
		"<form method=\"get\" action=\"\">\n");

	OUT("<input type=\"hidden\" name=\"m\" value=\"");
	xmlencode(mode);
	OUT("\" />\n");
	if (subreddit[0]) {
		OUT("<input type=\"hidden\" name=\"subreddit\" value=\"");
		xmlencode(subreddit);
		OUT("\" />\n");
	}
	OUT("<input type=\"hidden\" name=\"limit\" value=\"");
	xmlencode("100"); /* TODO maybe */
	OUT("\" />\n");
	OUT("<input type=\"hidden\" name=\"order\" value=\"");
	xmlencode("hot"); /* TODO maybe */
	OUT("\" />\n");

	OUT(
		"<table class=\"search\" width=\"100%\" border=\"0\" cellpadding=\"0\" cellspacing=\"0\">\n"
		"<tr>\n"
		"<td width=\"100%\" class=\"input\">\n"
		"	<input type=\"search\" name=\"q\" value=\"\" placeholder=\"Search...\" size=\"72\" autofocus=\"autofocus\" class=\"search\" accesskey=\"f\" />\n"
		"</td>\n"
		"<td nowrap class=\"nowrap\">\n"
		"	<input type=\"submit\" value=\"Search\" class=\"button\"/>\n"
		"	<select name=\"o\" title=\"Order by\" accesskey=\"o\">\n"
		"		<option value=\"hot\" selected=\"selected\">Hot</option>\n"
		"		<option value=\"new\">New</option>\n"
		"		<option value=\"rising\">Rising</option>\n"
		"		<option value=\"controversial\">Controversial</option>\n"
		"	</select>\n"
		"		<label for=\"m\">Style: </label>\n");

	if (!strcmp(mode, "light"))
		OUT("\t\t<input type=\"submit\" name=\"m\" value=\"Dark\" title=\"Dark mode\" id=\"m\" accesskey=\"s\"/>\n");
	else
		OUT("\t\t<input type=\"submit\" name=\"m\" value=\"Light\" title=\"Light mode\" id=\"m\" accesskey=\"s\"/>\n");

	OUT(
		"	</td>\n"
		"</tr>\n"
		"</table>\n"
		"</form>\n");

	if (r->nitems) {
		OUT(
			"<hr/>\n"
			"<table class=\"items\" width=\"100%\" border=\"0\" cellpadding=\"0\" cellspacing=\"0\">\n"
			"<tbody>\n");

		for (i = 0; i < r->nitems; i++) {
			item = r->items + i;

			OUT(
			"<tr>\n"
			"<td valign=\"middle\" align=\"center\" class=\"score\" width=\"100\">\n"
			"	\n");
			printf("%zu", item->ups); /* upvotes / score */
			OUT(
			"</td>\n"
			"<td valign=\"middle\" align=\"center\" class=\"thumb\" width=\"140\">\n");

			if (item->thumbnail[0] && !strncmp(item->thumbnail, "https://", 8)) {
				OUT("<a href=\"");

				/* link directly to dash url for video */
				if (item->is_video)
					xmlencode(item->dash_url);
				else
					xmlencode(item->url);

				OUT("\"><img src=\"");
				xmlencode(item->thumbnail);
				OUT("\" width=\"140\" alt=\"\" /></a>");
			}

			OUT(
			"</td>\n"
			"<td valign=\"top\">\n"
			"	<h2><a href=\"");
			/* link directly to dash url for video */
			if (item->is_video)
				xmlencode(item->dash_url);
			else
				xmlencode(item->url);
			OUT("\">");

			/* base url of url: somesite.org */
			baseurl[0] = '\0';
			if (!item->is_video && item->url[0]) {
				if ((start = strstr(item->url, "://"))) {
					start += strlen("://");
					if ((end = strstr(start, "/"))) {
						if (end - start + 1 < sizeof(baseurl)) {
							memcpy(baseurl, start, end - start);
							baseurl[end - start] = '\0';
						}
					}
				}
			}

			if (item->is_video)
				OUT("[video] ");
			xmlencode(item->title);

			strftime(timebuf, sizeof(timebuf), "%Y-%m-%d %H:%M", &(item->created_tm));

			OUT("</a></h2>\n");
			OUT(
			"	<time datetime=\"");
			OUT(timebuf);
			OUT("\">Submitted ");
			if (!friendlytime(item->created_utc)) {
				OUT("on ");
				OUT(timebuf);
			}
			OUT(" by</time>\n"
			"	<a href=\"\">");
			xmlencode(item->author);
			OUT("</a>\n");

			/* global view: show subreddit */
			if (!subreddit[0]) {
				/* TODO: link */
				OUT(" in r/");
				xmlencode(item->subreddit);
			}

			OUT(
			"	<br/>\n"
			"	<br/>\n"
			"	<a href=\"https://old.reddit.com/");
			xmlencode(item->permalink);
			OUT("\">");

			printf("%zu", item->num_comments);

			OUT(" comments</a>\n"
			"</td>\n"
			"<td valign=\"top\" align=\"right\" class=\"tags\">\n"
			"	<span class=\"base\">");
			OUT(baseurl);
			OUT(
				"</span>\n"
			"	<br/>\n"
			"	<span class=\"tag\">");

			/* TODO: + flair color */
			xmlencode(item->link_flair_text);
			OUT("	</span>\n"
			"</td>\n"
			"</tr>\n");

			OUT("<tr><td colspan=\"4\"><hr/></td></tr>\n");
		}
		OUT("</tbody>\n");

		/* pagination does not work for user/channel search */
		if (r->before[0] || r->after[0]) {
			OUT(
				"<tfoot>\n"
				"<tr>\n"
				"\t<td align=\"left\" class=\"nowrap\" nowrap>\n");
			if (r->before[0]) {
				OUT("\t\t<a href=\"?q=");
				xmlencode(""); // TODO: remove q param later
				OUT("&amp;before=");
				xmlencode(r->before);
				OUT("&amp;m=");
				xmlencode(mode);
				OUT("\" rel=\"prev\" accesskey=\"p\">&larr; prev</a>\n");
			}
			if (r->after[0]) {
				OUT(
					"\t</td>\n\t<td colspan=\"2\"></td>\n"
					"\t<td align=\"right\" class=\"a-r nowrap\" nowrap>\n");
				OUT("\t\t<a href=\"?q="); // TODO: remove q param later.
				xmlencode(""); //
				OUT("&amp;after=");
				xmlencode(r->after);
				OUT("&amp;m=");
				xmlencode(mode);
				OUT("\" rel=\"next\" accesskey=\"n\">next &rarr;</a>\n");
			}

			OUT(
				"\t</td>\n"
				"</tr>\n"
				"</tfoot>\n");
		}
		OUT("</table>\n");
	}

	OUT("</body>\n</html>\n");

	return 0;
}

int
main(void)
{
	struct list_response *r;

#if 0
	if (pledge("stdio dns inet rpath unveil", NULL) == -1 ||
	    unveil(TLS_CA_CERT_FILE, "r") == -1 ||
	    unveil(NULL, NULL) == -1) {
		OUT("Status: 500 Internal Server Error\r\n\r\n");
		exit(1);
	}
#endif

	parsecgi();

#if 0
	if (!rawsearch[0] && !chan[0] && !user[0])
		goto show;
#endif

	r = reddit_list(subreddit, ""); // TODO: page
	if (!r || r->nitems <= 0) {
		OUT("Status: 500 Internal Server Error\r\n\r\n");
		exit(1);
	}

show:
	now = time(NULL);
	render(r);

	return 0;
}
