#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "https.h"
#include "json.h"
#include "util.h"
#include "youtube.h"

static char *
youtube_request(const char *path)
{
	return request("www.youtube.com", path, "");
}

static char *
request_search(const char *s, const char *page, const char *order)
{
	char path[4096];

	snprintf(path, sizeof(path), "/results?search_query=%s", s);

	/* NOTE: pagination doesn't work at the moment:
	   this parameter is not supported anymore by Youtube */
	if (page[0]) {
		strlcat(path, "&page=", sizeof(path));
		strlcat(path, page, sizeof(path));
	}

	if (order[0] && strcmp(order, "relevance")) {
		strlcat(path, "&sp=", sizeof(path));
		if (!strcmp(order, "date"))
			strlcat(path, "CAI%3D", sizeof(path));
		else if (!strcmp(order, "views"))
			strlcat(path, "CAM%3D", sizeof(path));
		else if (!strcmp(order, "rating"))
			strlcat(path, "CAE%3D", sizeof(path));
	}

	/* check if request is too long (truncation) */
	if (strlen(path) >= sizeof(path) - 1)
		return NULL;

	return youtube_request(path);
}

int
extractjson(const char *s, char **start, char **end)
{
	*start = strstr(s, "window[\"ytInitialData\"] = ");
	if (*start) {
		(*start) += sizeof("window[\"ytInitialData\"] = ") - 1;
	} else {
		*start = strstr(s, "var ytInitialData = ");
		if (*start)
			(*start) += sizeof("var ytInitialData = ") - 1;
	}
	if (!*start)
		return -1;
	*end = strstr(*start, "};\n");
	if (!*end)
		*end = strstr(*start, "}; \n");
	if (!*end)
		*end = strstr(*start, "};<");
	if (!*end)
		return -1;
	(*end)++;

	return 0;
}

void
processnode(struct json_node *nodes, size_t depth, const char *value,
	void *pp)
{
	struct search_response *r = (struct search_response *)pp;
	static struct item *item;

	if (r->nitems > MAX_VIDEOS)
		return;

	/* new item, structures can be very deep, just check the end for:
	   (items|contents)[].videoRenderer objects */
	if (depth >= 3 &&
	    nodes[depth - 3].type == JSON_TYPE_ARRAY &&
	    nodes[depth - 2].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 1].type == JSON_TYPE_OBJECT &&
	    (!strcmp(nodes[depth - 3].name, "items") ||
	     !strcmp(nodes[depth - 3].name, "contents")) &&
	    !strcmp(nodes[depth - 1].name, "videoRenderer")) {
		r->nitems++;
		return;
	}

	if (r->nitems == 0)
		return;
	item = &(r->items[r->nitems - 1]);

	if (depth >= 4 &&
	    nodes[depth - 4].type == JSON_TYPE_ARRAY &&
	    nodes[depth - 3].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 2].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 1].type == JSON_TYPE_STRING &&
	    (!strcmp(nodes[depth - 4].name, "items") ||
	     !strcmp(nodes[depth - 4].name, "contents")) &&
	    !strcmp(nodes[depth - 2].name, "videoRenderer") &&
	    !strcmp(nodes[depth - 1].name, "videoId")) {
		strlcpy(item->id, value, sizeof(item->id));
	}

	if (depth >= 7 &&
	    nodes[depth - 7].type == JSON_TYPE_ARRAY &&
	    nodes[depth - 6].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 5].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 4].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 3].type == JSON_TYPE_ARRAY &&
	    nodes[depth - 2].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 1].type == JSON_TYPE_STRING &&
	    (!strcmp(nodes[depth - 7].name, "items") ||
	     !strcmp(nodes[depth - 7].name, "contents")) &&
	    !strcmp(nodes[depth - 5].name, "videoRenderer") &&
	    !strcmp(nodes[depth - 4].name, "title") &&
	    !strcmp(nodes[depth - 3].name, "runs") &&
	    !strcmp(nodes[depth - 1].name, "text") &&
		!item->title[0]) {
		strlcpy(item->title, value, sizeof(item->title));
	}

	if (depth >= 5 &&
	    nodes[depth - 5].type == JSON_TYPE_ARRAY &&
	    nodes[depth - 4].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 3].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 2].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 1].type == JSON_TYPE_STRING &&
	    (!strcmp(nodes[depth - 5].name, "items") ||
	     !strcmp(nodes[depth - 5].name, "contents")) &&
	    !strcmp(nodes[depth - 3].name, "videoRenderer") &&
	    !strcmp(nodes[depth - 1].name, "simpleText")) {
		if (!strcmp(nodes[depth - 2].name, "viewCountText") &&
		    !item->viewcount[0]) {
			strlcpy(item->viewcount, value, sizeof(item->viewcount));
		} else if (!strcmp(nodes[depth - 2].name, "lengthText") &&
		    !item->duration[0]) {
			strlcpy(item->duration, value, sizeof(item->duration));
		} else if (!strcmp(nodes[depth - 2].name, "publishedTimeText") &&
		    !item->publishedat[0]) {
			strlcpy(item->publishedat, value, sizeof(item->publishedat));
		}
	}

	if (depth >= 9 &&
	    nodes[depth - 9].type == JSON_TYPE_ARRAY &&
	    nodes[depth - 8].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 7].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 6].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 5].type == JSON_TYPE_ARRAY &&
	    nodes[depth - 4].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 3].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 2].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 1].type == JSON_TYPE_STRING &&
	    (!strcmp(nodes[depth - 9].name, "items") ||
	     !strcmp(nodes[depth - 9].name, "contents")) &&
	    !strcmp(nodes[depth - 7].name, "videoRenderer") &&
	    !strcmp(nodes[depth - 6].name, "longBylineText") &&
	    !strcmp(nodes[depth - 5].name, "runs") &&
	    !strcmp(nodes[depth - 3].name, "navigationEndpoint") &&
	    !strcmp(nodes[depth - 2].name, "browseEndpoint")) {
		if (!strcmp(nodes[depth - 1].name, "browseId")) {
			strlcpy(item->channelid, value, sizeof(item->channelid));
		}
	}

	if (depth >= 7 &&
	    nodes[depth - 7].type == JSON_TYPE_ARRAY &&
	    nodes[depth - 6].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 5].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 4].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 3].type == JSON_TYPE_ARRAY &&
	    nodes[depth - 2].type == JSON_TYPE_OBJECT &&
	    nodes[depth - 1].type == JSON_TYPE_STRING &&
	    (!strcmp(nodes[depth - 7].name, "items") ||
	     !strcmp(nodes[depth - 7].name, "contents")) &&
	    !strcmp(nodes[depth - 5].name, "videoRenderer") &&
	    !strcmp(nodes[depth - 4].name, "longBylineText") &&
	    !strcmp(nodes[depth - 3].name, "runs")) {
		if (!strcmp(nodes[depth - 1].name, "text") &&
		    !item->channeltitle[0]) {
			strlcpy(item->channeltitle, value, sizeof(item->channeltitle));
		}
	}
}

struct search_response *
youtube_search(const char *rawsearch, const char *page, const char *order)
{
	struct search_response *r;
	char *data, *s, *start, *end;
	int ret;

	if (!(data = request_search(rawsearch, page, order)))
		return NULL;

	if (!(s = strstr(data, "\r\n\r\n")))
		return NULL; /* invalid response */
	/* skip header */
	s += strlen("\r\n\r\n");

	if (!(r = calloc(1, sizeof(*r))))
		return NULL;

	if (extractjson(s, &start, &end) == -1) {
//		fprintf(stderr, "error extracting JSON");
		free(r);
		return NULL;
	}

	ret = parsejson(start, end - start, processnode, r);
	if (ret < 0) {
//		fprintf(stderr, "error parsing JSON");
		free(r);
		return NULL;
	}

	return r;
}
