#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "https.h"
#include "util.h"
#include "youtube.h"

#define OUT(s) (fputs((s), stdout))

extern char **environ;

static int curpage = 1;

/* CGI parameters */
static char rawsearch[4096], search[4096], order[16], page[64];

void
parsecgi(void)
{
	char *query, *p;
	size_t len;

	if (!(query = getenv("QUERY_STRING")))
		query = "";

	/* order */
	if ((p = getparam(query, "o"))) {
		if (decodeparam(order, sizeof(order), p) == -1 ||
			(strcmp(order, "date") &&
			strcmp(order, "relevance") &&
			strcmp(order, "views") &&
			strcmp(order, "rating")))
			order[0] = '\0';
	}
	if (!order[0])
		snprintf(order, sizeof(order), "relevance");

	/* page */
	if ((p = getparam(query, "page"))) {
		if (decodeparam(page, sizeof(page), p) == -1)
			page[0] = '\0';
		/* check if it's a number > 0 and < 100 */
		errno = 0;
		curpage = strtol(page, NULL, 10);
		if (errno || curpage < 0 || curpage > 100) {
			curpage = 1;
			page[0] = '\0';
		}
	}

	/* search */
	if ((p = getparam(query, "q"))) {
		if ((len = strcspn(p, "&")) && len + 1 < sizeof(rawsearch)) {
			memcpy(rawsearch, p, len);
			rawsearch[len] = '\0';
		}

		if (decodeparam(search, sizeof(search), p) == -1) {
			OUT("Status: 401 Bad Request\r\n\r\n");
			exit(1);
		}
	}
}

int
render(struct search_response *r)
{
	struct item *videos = r ? r->items : NULL;
	char tmp[64];
	size_t i;

	if (pledge("stdio", NULL) == -1) {
		OUT("Status: 500 Internal Server Error\r\n\r\n");
		exit(1);
	}

	OUT(
		"Content-Type: text/html; charset=utf-8\r\n\r\n"
		"<!DOCTYPE html>\n<html>\n<head>\n"
		"<meta name=\"referrer\" content=\"no-referrer\" />\n"
		"<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\" />\n"
		"<title>Search: \"");
		xmlencode(search);
		printf("\" sorted by %s</title>\n", order);
	OUT(
		"<link rel=\"stylesheet\" href=\"css/style.css\" type=\"text/css\" media=\"screen\" />\n"
		"<link rel=\"icon\" type=\"image/png\" href=\"/favicon.png\" />\n"
		"<meta content=\"width=device-width\" name=\"viewport\" />\n"
		"</head>\n"
	        "<body class=\"search\">\n"
		"<form method=\"get\" action=\"\">\n");

	OUT(
		"<table class=\"search\" width=\"100%\" border=\"0\" cellpadding=\"0\" cellspacing=\"0\">\n"
		"<tr>\n"
		"	<td width=\"100%\" class=\"input\">\n"
		"		<input type=\"search\" name=\"q\" value=\"");
	xmlencode(search);
	OUT(
		"\" placeholder=\"Search...\" size=\"72\" autofocus=\"autofocus\" class=\"search\" accesskey=\"f\" />\n"
		"	</td>\n"
		"	<td nowrap class=\"nowrap\">\n"
		"		<input type=\"submit\" value=\"Search\" class=\"button\"/>\n"
		"		<select name=\"o\" title=\"Order by\" accesskey=\"o\">\n");
	printf("			<option value=\"date\"%s>Creation date</option>\n", !strcmp(order, "date") ? " selected=\"selected\"" : "");
	printf("			<option value=\"relevance\"%s>Relevance</option>\n", !strcmp(order, "relevance") ? " selected=\"selected\"" : "");
	printf("			<option value=\"views\"%s>Views</option>\n", !strcmp(order, "views") ? " selected=\"selected\"" : "");
	printf("			<option value=\"rating\"%s>Rating</option>\n", !strcmp(order, "rating") ? " selected=\"selected\"" : "");
	OUT(
		"		</select>\n"
		"	</td>\n"
		"</tr>\n"
		"</table>\n"
		"</form>\n");

	if (r && r->nitems) {
		OUT(
			"<hr/>\n"
			"<table class=\"videos\" width=\"100%\" border=\"0\" cellpadding=\"0\" cellspacing=\"0\">\n"
			"<tbody>\n");

		for (i = 0; i < r->nitems; i++) {
			OUT("<tr class=\"v\">\n"
			"	<td class=\"thumb\" width=\"120\" align=\"center\">\n");

			if (videos[i].id[0]) {
				OUT("		<a href=\"https://www.youtube.com/embed/");
				xmlencode(videos[i].id);
				OUT("\"><img src=\"https://i.ytimg.com/vi/");
				xmlencode(videos[i].id);
				OUT("/default.jpg\" alt=\"\" height=\"90\" border=\"0\" /></a>\n");
			} else {
				/* placeholder image */
				OUT("		<img src=\"https://i.ytimg.com/vi/\" alt=\"\" height=\"90\" border=\"0\" />\n");
			}
			OUT("	</td>\n"
				"	<td>\n"
				"		<span class=\"title\">");

			if (videos[i].id[0]) {
				OUT("<a href=\"https://www.youtube.com/embed/");
				xmlencode(videos[i].id);
				printf("\" accesskey=\"%zu\">", i);
			}

			switch (videos[i].linktype) {
			case Channel:
				OUT("[Channel] ");
				xmlencode(videos[i].channeltitle);
				break;
			case Movie:
				OUT("[Movie] ");
				xmlencode(videos[i].title);
				break;
			case Playlist:
				OUT("[Playlist] ");
				xmlencode(videos[i].title);
				break;
			default:
				xmlencode(videos[i].title);
				break;
			}

			if (videos[i].id[0])
				OUT("</a>");

			OUT(
				"</span><br/>\n"
				"\t\t<span class=\"channel\">");

			OUT("<a href=\"?q=");
			xmlencode(videos[i].channeltitle);
			OUT("&amp;o=");
			xmlencode(order);
			OUT("\">");
			xmlencode(videos[i].channeltitle);
			OUT("</a>");
			if (videos[i].channelid[0] || videos[i].userid[0]) {
				OUT(" | <a title=\"");
				xmlencode(videos[i].channeltitle);
				OUT(" Atom feed\" href=\"https://www.youtube.com/feeds/videos.xml?");
				if (videos[i].channelid[0]) {
					OUT("channel_id=");
					xmlencode(videos[i].channelid);
				} else if (videos[i].userid[0]) {
					OUT("user=");
					xmlencode(videos[i].userid);
				}
				OUT("\">Atom feed</a>");
			}
			OUT("</span><br/>\n");
			if (videos[i].publishedat[0]) {
				OUT("		<span class=\"publishedat\">Published: ");
				OUT(videos[i].publishedat);
				OUT("</span><br/>\n");
			}
			OUT("		<span class=\"stats\">");
			OUT(videos[i].viewcount);
			OUT(
				"</span><br/>\n"
				"	</td>\n"
				"	<td align=\"right\" class=\"a-r\">\n"
				"		<span class=\"duration\">");
			OUT(videos[i].duration);
			OUT(
				"</span>\n"
				"	</td>\n"
				"</tr>\n"
				"<tr class=\"hr\">\n"
				"	<td colspan=\"3\"><hr/></td>\n"
				"</tr>\n");
		}
		OUT("</tbody>\n");

		OUT(
			"<tfoot>\n"
			"<tr>\n"
			"\t<td align=\"left\" class=\"nowrap\" nowrap>\n");
		if (curpage > 1) {
			OUT("\t\t<a href=\"?q=");
			xmlencode(search);
			OUT("&amp;page=");
			snprintf(tmp, sizeof(tmp), "%d", curpage - 1);
			xmlencode(tmp);
			OUT("&amp;o=");
			xmlencode(order);
			OUT("\" rel=\"prev\" accesskey=\"p\">&larr; prev</a>\n");
		}
		OUT(
			"\t</td>\n\t<td></td>\n"
			"\t<td align=\"right\" class=\"a-r nowrap\" nowrap>\n");

		OUT("\t\t<a href=\"?q=");
		xmlencode(search);
		OUT("&amp;page=");
		snprintf(tmp, sizeof(tmp), "%d", curpage + 1);
		xmlencode(tmp);
		OUT("&amp;o=");
		xmlencode(order);
		OUT("\" rel=\"next\" accesskey=\"n\">next &rarr;</a>\n"
		    "\t</td>\n"
		    "</tr>\n"
		    "</tfoot>\n"
		    "</table>\n");
	}

	OUT("</body>\n</html>\n");

	return 0;
}

int
main(void)
{
	struct search_response *r = NULL;

	if (pledge("stdio dns inet rpath unveil", NULL) == -1 ||
	    unveil(TLS_CA_CERT_FILE, "r") == -1 ||
	    unveil(NULL, NULL) == -1) {
		OUT("Status: 500 Internal Server Error\r\n\r\n");
		exit(1);
	}

	parsecgi();

	if (!rawsearch[0])
		goto show;

	r = youtube_search(rawsearch, page, order);
	if (!r || r->nitems == 0) {
		OUT("Status: 500 Internal Server Error\r\n\r\n");
		exit(1);
	}

show:
	render(r);

	return 0;
}
