#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "https.h"
#include "reddit.h"
#include "util.h"

#define OUT(s) (fputs((s), stdout))
#define OUTESCAPE(s) (printescape(s))

/* print: ignore control-characters */
void
printescape(const char *s)
{
	for (; *s; ++s)
		if (!iscntrl((unsigned char)*s))
			fputc(*s, stdout);
}

void
printitem(struct item *item)
{
	if (!item || !item->title[0])
		return;

	printf("title:           %s\n", item->title);
	printf("url:             %s\n", item->url);
	printf("subreddit:       %s\n", item->subreddit);
	printf("author:          %s\n", item->author);
	printf("thumbnail:       %s\n", item->thumbnail);
	printf("ups:             %ld\n", item->ups);
	printf("downs:           %ld\n", item->downs);
	printf("num_comments:    %ld\n", item->num_comments);

	struct tm *tm = gmtime(&(item->created_utc));
	printf("created_utc:     %lld\n", item->created_utc);
	printf("created_utc:     %s", asctime(tm));

	if (item->is_video) {
		printf("is_video:        %d\n", item->is_video);
		printf("duration:        %ld\n", item->duration);
		printf("dash_url:        %s\n", item->dash_url);
	}

	printf("===\n");
}

int
render(struct list_response *r)
{
	size_t i;

	if (pledge("stdio", NULL) == -1) {
		fprintf(stderr, "pledge: %s\n", strerror(errno));
		exit(1);
	}

	for (i = 0; i < r->nitems; i++)
		printitem(&(r->items[i]));

	if (r->before[0])
		printf("before pagination token: %s\n", r->before);
	if (r->after[0])
		printf("after pagination token: %s\n", r->after);

	return 0;
}

static void
usage(const char *argv0)
{
	fprintf(stderr, "usage: %s [subreddit]\n", argv0);
	exit(1);
}

int
main(int argc, char *argv[])
{
	struct list_response *r;
	char subreddit[1024] = "";

	if (pledge("stdio dns inet rpath unveil", NULL) == -1) {
		fprintf(stderr, "pledge: %s\n", strerror(errno));
		exit(1);
	}
	if (unveil(TLS_CA_CERT_FILE, "r") == -1) {
		fprintf(stderr, "unveil: %s\n", strerror(errno));
		exit(1);
	}
	if (unveil(NULL, NULL) == -1) {
		fprintf(stderr, "unveil: %s\n", strerror(errno));
		exit(1);
	}

	if (argc > 1) {
		if (!uriencode(argv[1], subreddit, sizeof(subreddit)))
			usage(argv[0]);
	}

	r = reddit_list(subreddit, 100, "", ""); /* TODO: pagination */
	if (!r || r->nitems == 0) {
		OUT("No items found\n");
		exit(1);
	}

	render(r);

	return 0;
}
