#include <sys/types.h>

#include <ctype.h>
#include <err.h>
#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <wchar.h>

#include "duckduckgo.h"
#include "https.h"
#include "util.h"
#include "xml.h"

static XMLParser x;

static struct duckduckgo_results *results;
static struct duckduckgo_result result;
static int istitle, isdescription, isurl, isresult;

void
sanitize(char *s, size_t len)
{
	size_t i;

	/* trim trailing whitespace */
	for (i = strlen(s); i > 0; i--) {
		if (!isspace((unsigned char)s[i - 1]))
			break;
	}
	s[i] = '\0';

	/* trim leading whitespace */
	for (i = 0; s[i]; i++) { // TODO: wrong
		if (!isspace((unsigned char)s[i]))
			break;
	}
	memmove(s, s + i, len - i + 1);

	for (i = 0; s[i]; i++) {
		if (iscntrl((unsigned char)s[i]))
			s[i] = ' ';
	}
}

void
xmlattr(XMLParser *x, const char *t, size_t tl, const char *a, size_t al,
        const char *v, size_t vl)
{
	if (!strcmp(t, "div") && !strcmp(a, "class") && strstr(v, "results_links"))
		isresult = 1;

	if (!isresult)
		return;

	/* clear fix is use in the end of a result */
	if (!strcmp(t, "div") && !strcmp(a, "style") && strstr(v, "clear: both")) {
		isresult = 0;

		if (!result.title[0] || !result.url[0])
			return;

		/* add result */
		if (results->nitems <= MAX_ITEMS) {
			memcpy(&(results->items[results->nitems]),
			       &result, sizeof(result));
			results->nitems++;
		}
		memset(&result, 0, sizeof(result));
		return;
	}

	if (!strcmp(t, "h2") && !strcmp(a, "class") && strstr(v, "result__title"))
		istitle = 1;
	if (!strcmp(t, "a") && !strcmp(a, "class") && strstr(v, "result__snippet"))
		isdescription = 1;
	if (!strcmp(t, "a") && !strcmp(a, "class") && strstr(v, "result__url"))
		isurl = 1;
	if (isurl && !strcmp(t, "a") && !strcmp(a, "href"))
		strlcpy(result.url, v, sizeof(result.url));
}

void
xmlattrentity(XMLParser *x, const char *t, size_t tl, const char *a, size_t al,
              const char *v, size_t vl)
{
	char buf[16];
	ssize_t len;

	if (!isresult || !istitle || !isdescription || !isurl)
		return;

	if ((len = xml_entitytostr(v, buf, sizeof(buf))) > 0)
		xmlattr(x, t, tl, a, al, buf, (size_t)len);
	else
		xmlattr(x, t, tl, a, al, v, vl);
}

void
xmldata(XMLParser *x, const char *d, size_t dl)
{
	if (istitle)
		strlcat(result.title, d, sizeof(result.title));
	if (isdescription)
		strlcat(result.description, d, sizeof(result.description));
}

void
xmlcdata(XMLParser *x, const char *d, size_t dl)
{
	xmldata(x, d, dl);
}

void
xmldataentity(XMLParser *x, const char *d, size_t dl)
{
	char buf[16];
	ssize_t len;

	if (!isresult || !istitle || !isdescription || !isurl)
		return;

	if ((len = xml_entitytostr(d, buf, sizeof(buf))) > 0)
		xmldata(x, buf, (size_t)len);
	else
		xmldata(x, d, dl);
}

void
xmltagend(XMLParser *x, const char *t, size_t tl, int isshort)
{
	if (!isresult)
		return;

	if (isdescription) {
		/* highlight */
		if (!strcmp(t, "b"))
			strlcat(result.description, "*", sizeof(result.description));
	}

	if (istitle && !strcmp(t, "h2"))
		istitle = 0;
	if (isdescription && !strcmp(t, "a"))
		isdescription = 0;
	if (isurl && !strcmp(t, "a"))
		isurl = 0;
	if (!strcmp(t, "div")) {
		/* decode url and remove "tracking"/usage part via DDG */
		if (!strncmp(result.url, "uddg=", sizeof("uddg=") - 1)) {
			if (decodeparam(result.urldecoded, sizeof(result.urldecoded),
			    result.url + sizeof("uddg=") - 1) == -1)
				result.urldecoded[0] = '\0';
		}

		sanitize(result.title, strlen(result.title));
		sanitize(result.urldecoded, strlen(result.urldecoded));
		sanitize(result.description, strlen(result.description));

		istitle = isdescription = isurl = 0;
	}
}

void
xmltagstart(XMLParser *x, const char *t, size_t tl)
{
	/* highlight */
	if (isdescription && !strcmp(t, "b"))
		strlcat(result.description, "*", sizeof(result.description));

}

char *
duckduckgo_search_data(const char *s)
{
	char path[4096];
	int r;

	r = snprintf(path, sizeof(path), "/html/?q=%s", s);
	if (r < 0 || (size_t)r >= sizeof(path))
		return NULL;

	return request("html.duckduckgo.com", path, "");
}

struct duckduckgo_results *
duckduckgo_search(const char *s)
{
	struct duckduckgo_results *r;
	char *data;

	results = NULL; /* global */

	if (!(r = calloc(1, sizeof(*r))))
		return NULL;

	/* TODO: encodeuri s */
	if (!(data = duckduckgo_search_data(s))) {
		free(r);
		results = NULL;
		return NULL;
	}

	// TODO: xmlparser, parse data into struct duckduckgo_results.

	x.xmlattr = xmlattr;
	x.xmlattrentity = xmlattrentity;
	x.xmlcdata = xmlcdata;
	x.xmldata = xmldata;
	x.xmldataentity = xmldataentity;
	x.xmltagend = xmltagend;
	x.xmltagstart = xmltagstart;

	results = r; /* global: store */
	setxmldata(data, strlen(data));
	xml_parse(&x);

	free(data);

	return r;
}
