#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "https.h"
#include "util.h"
#include "youtube.h"

#define OUT(s) (fputs((s), stdout))
#define OUTTEXT(s) gophertext(stdout, s, strlen(s));
#define OUTLINK(s) gophertext(stdout, s, strlen(s));

static const char *host = "127.0.0.1", *port = "70";

void
line(int _type, const char *username, const char *selector)
{
	putchar(_type);
	OUTTEXT(username);
	putchar('\t');
	OUTLINK(selector);
	printf("\t%s\t%s\r\n", host, port);
}

void
error(const char *s)
{
	line('3', s, "");
}

void
info(const char *s)
{
	line('i', s, "");
}

void
header(void)
{
}

void
footer(void)
{
	printf(".\r\n");
}

int
render(struct search_response *r)
{
	struct item *videos = r->items;
	size_t i;

	if (pledge("stdio", NULL) == -1)
		exit(1);

	header();
	for (i = 0; i < r->nitems; i++) {
		if (videos[i].id[0])
			putchar('h');
		else
			putchar('i');

		switch (videos[i].linktype) {
		case Channel:
			OUT("[Channel] ");
			OUTTEXT(videos[i].channeltitle);
			break;
		case Movie:
			OUT("[Movie] ");
			OUTTEXT(videos[i].title);
			break;
		case Playlist:
			OUT("[Playlist] ");
			OUTTEXT(videos[i].title);
			break;
		default:
			OUTTEXT(videos[i].title);
			break;
		}

		OUT("\t");
		if (videos[i].id[0]) {
			OUT("URL:https://www.youtube.com/embed/");
			OUTLINK(videos[i].id);
		}
		printf("\t%s\t%s\r\n", host, port);

		if (videos[i].channelid[0] || videos[i].userid[0]) {
			OUT("hAtom feed of ");
			OUTTEXT(videos[i].channeltitle);
			OUT("\t");
			OUTLINK("URL:https://www.youtube.com/feeds/videos.xml?");
			if (videos[i].channelid[0]) {
				OUT("channel_id=");
				OUTLINK(videos[i].channelid);
			} else if (videos[i].userid[0]) {
				OUT("user=");
				OUTLINK(videos[i].userid);
			}
			printf("\t%s\t%s\r\n", host, port);
		}
		if (videos[i].duration[0]) {
			OUT("iDuration:      " );
			OUTTEXT(videos[i].duration);
			printf("\t%s\t%s\t%s\r\n", "", host, port);
		}
		if (videos[i].publishedat[0]) {
			OUT("iPublished:     ");
			OUTTEXT(videos[i].publishedat);
			printf("\t%s\t%s\t%s\r\n", "", host, port);
		}
		if (videos[i].viewcount[0]) {
			OUT("iViews:         ");
			OUTTEXT(videos[i].viewcount);
			printf("\t%s\t%s\t%s\r\n", "", host, port);
		}
		info("");
		info("");
	}
	footer();

	return 0;
}

int
main(void)
{
	struct search_response *r;
	char *p, search[1024];

	if (pledge("stdio dns inet rpath unveil", NULL) == -1)
		exit(1);
	if (unveil(TLS_CA_CERT_FILE, "r") == -1 ||
	    unveil(NULL, NULL) == -1)
		exit(1);

	if ((p = getenv("SERVER_NAME")))
		host = p;
	if ((p = getenv("SERVER_PORT")))
		port = p;

	if (!(p = getenv("X_GOPHER_SEARCH"))) /* geomyidae */
		p = getenv("SEARCHREQUEST"); /* gophernicus */

	if (p && !uriencode(p, search, sizeof(search))) {
		error("Invalid search");
		printf(".\r\n");
		exit(1);
	}

	r = youtube_search(search, "", "", "", "relevance");
	if (!r || r->nitems == 0) {
		error("No videos found");
		printf(".\r\n");
		exit(1);
	}

	render(r);

	return 0;
}
