#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "https.h"
#include "util.h"
#include "youtube.h"

#define OUT(s) fputs((s), stdout)
#define OUTESCAPE(s) printescape((s))

/* print: ignore control-characters */
void
printescape(const char *s)
{
	for (; *s; ++s)
		if (!iscntrl((unsigned char)*s))
			fputc(*s, stdout);
}

int
render(struct search_response *r)
{
	struct item *videos = r->items;
	size_t i;

	if (pledge("stdio", NULL) == -1) {
		fprintf(stderr, "pledge: %s\n", strerror(errno));
		exit(1);
	}

	for (i = 0; i < r->nitems; i++) {
		/* TODO: better printing of other types */
		switch (videos[i].linktype) {
		case Channel:
			OUT("[Channel] ");
			OUTESCAPE(videos[i].channeltitle);
			break;
		case Movie:
			OUT("[Movie] ");
			OUTESCAPE(videos[i].title);
			break;
		case Playlist:
			OUT("[Playlist] ");
			OUTESCAPE(videos[i].title);
			break;
		default:
			OUTESCAPE(videos[i].title);
			break;
		}
		OUT("\n");

		if (videos[i].id[0]) {
			OUT("URL:           https://www.youtube.com/embed/");
                        OUTESCAPE(videos[i].id);
			OUT("\n");
		}

		if (videos[i].channelid[0] || videos[i].userid[0]) {
			OUT("Atom feed:     https://www.youtube.com/feeds/videos.xml?");
			if (videos[i].channelid[0]) {
				OUT("channel_id=");
				OUTESCAPE(videos[i].channelid);
			} else if (videos[i].userid[0]) {
				OUT("user=");
				OUTESCAPE(videos[i].userid);
			}
			OUT("\n");
		}

		if (videos[i].channelid[0] || videos[i].userid[0]) {
			OUT("Channel title: ");
			OUTESCAPE(videos[i].channeltitle);
			OUT("\n");
			if (videos[i].channelid[0]) {
				OUT("Channelid:     ");
				OUTESCAPE(videos[i].channelid);
				OUT("\n");
			} else if (videos[i].userid[0]) {
				OUT("Userid:        ");
				OUTESCAPE(videos[i].userid);
				OUT("\n");
			}
		}
		if (videos[i].publishedat[0]) {
			OUT("Published:     ");
			OUTESCAPE(videos[i].publishedat);
			OUT("\n");
		}
		if (videos[i].viewcount[0]) {
			OUT("Viewcount:     ");
			OUTESCAPE(videos[i].viewcount);
			OUT("\n");
		}
		if (videos[i].duration[0]) {
			OUT("Duration:      " );
			OUTESCAPE(videos[i].duration);
			OUT("\n");
		}
		OUT("===\n");
	}

	return 0;
}

static void
usage(const char *argv0)
{
	fprintf(stderr, "usage: %s <keywords>\n", argv0);
	exit(1);
}

int
main(int argc, char *argv[])
{
	struct search_response *r;
	char search[1024];

	if (pledge("stdio dns inet rpath unveil", NULL) == -1) {
		fprintf(stderr, "pledge: %s\n", strerror(errno));
		exit(1);
	}
	if (unveil(TLS_CA_CERT_FILE, "r") == -1) {
		fprintf(stderr, "unveil: %s\n", strerror(errno));
		exit(1);
	}
	if (unveil(NULL, NULL) == -1) {
		fprintf(stderr, "unveil: %s\n", strerror(errno));
		exit(1);
	}

	if (argc < 2 || !argv[1][0])
		usage(argv[0]);
	if (!uriencode(argv[1], search, sizeof(search)))
		usage(argv[0]);

	r = youtube_search(search, "", "", "", "relevance");
	if (!r || r->nitems == 0) {
		OUT("No videos found\n");
		exit(1);
	}

	render(r);

	return 0;
}
