#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "https.h"
#include "json.h"
#include "twitch.h"
#include "util.h"

#define OUT(s) (fputs((s), stdout))

extern char **environ;

/* page variables */
static char *title = "", *pagetitle = "", *classname = "";

static int
gamecmp_name(const void *v1, const void *v2)
{
	struct game *g1 = (struct game *)v1;
	struct game *g2 = (struct game *)v2;

	return strcmp(g1->name, g2->name);
}

void
header(void)
{
	OUT(
	"<!DOCTYPE html>\n"
	"<html>\n"
	"<head>\n"
	"<title>");
	if (title[0]) {
		xmlencode(title);
		OUT(" - ");
	}
	if (pagetitle[0]) {
		xmlencode(pagetitle);
		OUT(" - ");
	}
	OUT("Twitch.tv</title>\n"
	"	<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\" />\n"
	"	<meta http-equiv=\"Content-Language\" content=\"en\" />\n"
	"	<link href=\"twitch.css\" rel=\"stylesheet\" type=\"text/css\" />\n"
	"</head>\n"
	"<body class=\"");
	xmlencode(classname);
	OUT(
	"\">\n"
	"<div id=\"menuwrap\">\n"
	"<div id=\"menu\">\n"
	"<span id=\"links\">\n"
	"	<a href=\"featured\">Featured</a> | \n"
	"	<a href=\"games\">Games</a> | \n"
	"	<a href=\"vods\">VODS</a> |\n"
	"	<a href=\"https://git.codemadness.org/frontends/\">Source-code</a> | \n"
	"	<a href=\"links\">Links</a>\n"
	"</span>\n"
	"</div></div>\n"
	"	<hr class=\"hidden\" />\n"
	"	<div id=\"mainwrap\">\n"
	"	<div id=\"main\">\n");

	if (pagetitle[0] || title[0]) {
		OUT("<h1><a href=\"\">");
		if (title[0])
			xmlencode(title);
		if (pagetitle[0]) {
			if (title[0])
				OUT(" - ");
			xmlencode(pagetitle);
		}
		OUT("</a></h1>\n");
	}
}

void
footer(void)
{
	OUT("</div></div></body></html>\n");
}

void
render_links(void)
{
	OUT("Content-Type: text/html; charset=utf-8\r\n\r\n");

	header();
	OUT("<ul>\n"
	"<li><a href=\"https://mpv.io/installation/\">mpv player</a></li>\n"
	"<li><a href=\"https://github.com/ytdl-org/youtube-dl\">youtube-dl</a></li>\n"
	"<li><a href=\"https://www.videolan.org/\">VLC</a></li>\n"
	"<li><a href=\"https://dev.twitch.tv/docs\">Twitch.tv API</a></li>\n"
	"</ul>\n");
	footer();
}

void
render_games_top(struct games_response *r)
{
	struct game *game;
	size_t i;

	OUT("Content-Type: text/html; charset=utf-8\r\n\r\n");

	header();
	OUT("<table class=\"table\" border=\"0\">");
	OUT("<thead>\n<tr>");
	OUT("<th align=\"left\" class=\"name\">Name</th>");
	OUT("</tr>\n</thead>\n<tbody>\n");
	for (i = 0; i < r->nitems; i++) {
		game = &(r->data[i]);

		OUT("<tr><td><a href=\"streams?game_id=");
		xmlencode(game->id);
		OUT("\">");
		xmlencode(game->name);
		OUT("</a></td></tr>\n");
	}
	OUT("</tbody></table>");
	footer();
}

void
render_streams(struct streams_response *r, const char *game_id)
{
	struct stream *stream;
	size_t i;

	OUT("Content-Type: text/html; charset=utf-8\r\n\r\n");

	header();
	OUT("<table class=\"table\" border=\"0\">");
	OUT("<thead>\n<tr>");
	if (!game_id[0])
		OUT("<th align=\"left\" class=\"game\">Game</th>");

	OUT("<th align=\"left\" class=\"name\">Name</th>");
	OUT("<th align=\"left\" class=\"title\">Title</th>");
	OUT("<th align=\"right\" class=\"viewers\">Viewers</th>");
	OUT("</tr>\n</thead>\n<tbody>\n");
	for (i = 0; i < r->nitems; i++) {
		stream = &(r->data[i]);

		OUT("<tr>");

		if (!game_id[0]) {
			OUT("<td>");
			if (r->data[i].game) {
				OUT("<a href=\"streams?game_id=");
				xmlencode(r->data[i].game->id);
				OUT("\">");
				xmlencode(r->data[i].game->name);
				OUT("</a>");
			}
			OUT("</td>");
		}

		OUT("<td class=\"name\">");
		if (stream->user) {
			OUT("<a href=\"https://www.twitch.tv/");
			xmlencode(stream->user->login);
			OUT("\">");
			xmlencode(stream->user_name);
			OUT("</a>");
		} else {
			xmlencode(stream->user_name);
		}
		OUT("</td>");

		OUT("<td class=\"title\">");
		if (stream->user) {
			OUT("<a href=\"https://www.twitch.tv/");
			xmlencode(stream->user->login);
			OUT("\">");
		}
		if (stream->language[0]) {
			OUT("[");
			xmlencode(stream->language);
			OUT("] ");
		}
		xmlencode(stream->title);
		if (stream->user)
			OUT("</a>");

		OUT("</td>");

		OUT("<td align=\"right\" class=\"viewers\">");
		printf("%lld", stream->viewer_count);
		OUT("</td>");

		OUT("</tr>\n");
	}
	OUT("</tbody></table>");
	footer();
}

void
render_videos_atom(struct videos_response *r, const char *login)
{
	struct video *video;
	size_t i;

	OUT("Content-Type: text/xml; charset=utf-8\r\n\r\n");

	OUT("<feed xmlns=\"http://www.w3.org/2005/Atom\" xml:lang=\"en\">\n");
	for (i = 0; i < r->nitems; i++) {
		video = &(r->data[i]);

		OUT("<entry>\n");
		OUT("\t<title type=\"text\">");
		xmlencode(video->title);
		OUT("</title>\n");
		OUT("\t<link rel=\"alternate\" type=\"text/html\" href=\"");
		xmlencode(video->url);
		OUT("\" />\n");
		OUT("\t<id>");
		xmlencode(video->url);
		OUT("</id>\n");
		OUT("\t<updated>");
		xmlencode(video->created_at);
		OUT("</updated>\n");
		OUT("\t<published>");
		xmlencode(video->created_at);
		OUT("</published>\n");
		OUT("</entry>\n");
	}
	OUT("</feed>\n");
}

void
render_videos(struct videos_response *r, const char *login)
{
	struct video *video;
	size_t i;

	OUT("Content-Type: text/html; charset=utf-8\r\n\r\n");

	header();

	OUT("<form method=\"get\" action=\"\">\n");
	OUT("\t<input type=\"search\" name=\"login\" value=\"\" placeholder=\"Login name...\" autofocus=\"autofocus\" />\n");
	OUT("\t<input type=\"submit\" name=\"list\" value=\"List vods\" />\n");
	OUT("</form>\n<hr/>\n");

	/* no results or no user_id parameter: quick exit */
	if (r == NULL) {
		footer();
		return;
	}

	/* link to atom format. */
	if (login[0]) {
		OUT("<p><a href=\"?login=");
		xmlencode(login);
		OUT("&amp;format=atom\">Atom feed</a></p>\n");
	}

	OUT("<table class=\"table\" border=\"0\">");
	OUT("<thead>\n<tr>");
	OUT("<th align=\"left\" class=\"created_at\">Created</th>");
	OUT("<th align=\"left\" class=\"title\">Title</th>");
	OUT("<th align=\"right\" class=\"duration\">Duration</th>");
	OUT("<th align=\"right\" class=\"views\">Views</th>");
	OUT("</tr>\n</thead>\n<tbody>\n");
	for (i = 0; i < r->nitems; i++) {
		video = &(r->data[i]);

		OUT("<tr>");

		OUT("<td>");
		xmlencode(video->created_at);
		OUT("</td>");

		OUT("<td class=\"wrap\"><a href=\"");
		xmlencode(video->url);
		OUT("\">");
		xmlencode(video->title);
		OUT("</a></td>");

		OUT("<td align=\"right\"><a href=\"");
		xmlencode(video->url);
		OUT("\">");
		xmlencode(video->duration);
		OUT("</a></td>");

		OUT("<td align=\"right\">");
		printf("%lld", video->view_count);
		OUT("</td>");

		OUT("</tr>\n");
	}
	OUT("</tbody></table>");
	footer();
}

void
handle_streams(void)
{
	struct streams_response *r;
	struct users_response *ru = NULL;
	struct games_response *rg = NULL;
	char game_id[32] = "";
	char *p, *querystring;

	pagetitle = "Streams";
	classname = "streams";

	/* parse "game_id" parameter */
	if ((querystring = getenv("QUERY_STRING"))) {
		if ((p = getparam(querystring, "game_id"))) {
			if (decodeparam(game_id, sizeof(game_id), p) == -1)
				game_id[0] = '\0';
		}
	}

	if (game_id[0])
		r = twitch_streams_bygame(game_id);
	else
		r = twitch_streams();

	if (r == NULL)
		return;

	/* find detailed games data with streams */
	if (!game_id[0])
		rg = twitch_streams_games(r);

	/* find detailed user data with streams */
	ru = twitch_streams_users(r);

	if (pledge("stdio", NULL) == -1) {
		OUT("Status: 500 Internal Server Error\r\n\r\n");
		exit(1);
	}

	render_streams(r, game_id);

	free(r);
	free(rg);
	free(ru);
}

void
handle_videos(void)
{
	struct videos_response *r = NULL;
	struct users_response *ru = NULL;
	char user_id[32] = "", login[64] = "", format[6] = "";
	char *p, *querystring;

	pagetitle = "Videos";
	classname = "videos";

	/* parse "user_id" or "login" parameter */
	if ((querystring = getenv("QUERY_STRING"))) {
		if ((p = getparam(querystring, "user_id"))) {
			if (decodeparam(user_id, sizeof(user_id), p) == -1)
				user_id[0] = '\0';
		}
		if ((p = getparam(querystring, "login"))) {
			if (decodeparam(login, sizeof(login), p) == -1)
				login[0] = '\0';
		}
		if ((p = getparam(querystring, "format"))) {
			if (decodeparam(format, sizeof(format), p) == -1)
				format[0] = '\0';
		}
	}

	/* no parameter given, show form */
	if (!user_id[0] && !login[0]) {
		if (pledge("stdio", NULL) == -1) {
			OUT("Status: 500 Internal Server Error\r\n\r\n");
			exit(1);
		}

		render_videos(r, "");
		return;
	}

	if (user_id[0]) {
		r = twitch_videos_byuserid(user_id);
	} else {
		ru = twitch_users_bylogin(login);
		if (ru && ru->nitems > 0)
			r = twitch_videos_byuserid(ru->data[0].id);
	}

	if (pledge("stdio", NULL) == -1) {
		OUT("Status: 500 Internal Server Error\r\n\r\n");
		exit(1);
	}

	if (r && r->nitems > 0)
		title = r->data[0].user_name;

	if (!strcmp(format, "atom"))
		render_videos_atom(r, login);
	else
		render_videos(r, login);

	free(ru);
	free(r);
}

void
handle_games_top(void)
{
	struct games_response *r;

	pagetitle = "Top 100 games";
	classname = "topgames";

	if (!(r = twitch_games_top()))
		return;

	if (pledge("stdio", NULL) == -1) {
		OUT("Status: 500 Internal Server Error\r\n\r\n");
		exit(1);
	}

	/* sort by name alphabetically, NOTE: the results are the top 100
	   sorted by viewcount. View counts are not visible in the new
	   Helix API data). */
	qsort(r->data, r->nitems, sizeof(r->data[0]), gamecmp_name);

	render_games_top(r);
	free(r);
}

void
handle_links(void)
{
	if (pledge("stdio", NULL) == -1) {
		OUT("Status: 500 Internal Server Error\r\n\r\n");
		exit(1);
	}

	pagetitle = "Links";
	classname = "links";

	render_links();
}

int
main(void)
{
	char *path, *pathinfo;

	if (pledge("stdio dns inet rpath unveil", NULL) == -1 ||
	    unveil(TLS_CA_CERT_FILE, "r") == -1 ||
	    unveil(NULL, NULL) == -1) {
		OUT("Status: 500 Internal Server Error\r\n\r\n");
		exit(1);
	}

	if (!(pathinfo = getenv("PATH_INFO")))
		pathinfo = "/";

	path = pathinfo;
	if (path[0] == '/')
		path++;

	if (!strcmp(path, "") ||
	    !strcmp(path, "featured") ||
	    !strcmp(path, "streams")) {
		/* featured / by game id */
		handle_streams();
	} else if (!strcmp(path, "topgames") ||
	           !strcmp(path, "games")) {
		handle_games_top();
	} else if (!strcmp(path, "videos") ||
	           !strcmp(path, "vods")) {
		handle_videos();
	} else if (!strcmp(path, "links")) {
		handle_links();
	} else {
		OUT("Status: 404 Not Found\r\n\r\n");
		exit(1);
	}

	return 0;
}
