#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <tls.h>

#define READ_BUF_SIZ        16384  /* read buffer in bytes */
#define MAX_RESPONSETIMEOUT 10     /* timeout in seconds */
#define MAX_RESPONSESIZ     4194304 /* max download size in bytes */

static void
die(const char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);

	exit(1);
}

/* TODO: use die or rename die to fatal */
void
fatal(const char *s)
{
	fputs(s, stderr);
	exit(1);
}

char *
readtls(struct tls *t)
{
	char *buf;
	size_t len = 0, size = 0;
	ssize_t r;

	/* always allocate an empty buffer */
	if (!(buf = calloc(1, size + 1)))
		die("calloc: %s\n", strerror(errno));

	while (1) {
		if (len + READ_BUF_SIZ + 1 > size) {
			/* allocate size: common case is small textfiles */
			size += READ_BUF_SIZ;
			if (!(buf = realloc(buf, size + 1)))
				die("realloc: %s\n", strerror(errno));
		}
		r = tls_read(t, &buf[len], READ_BUF_SIZ);
		if (r == TLS_WANT_POLLIN || r == TLS_WANT_POLLOUT) {
			continue;
		} else if (r <= 0) {
			break;
		}
		len += r;
		buf[len] = '\0';
		if (len > MAX_RESPONSESIZ)
			die("response is too big: > %zu bytes\n", MAX_RESPONSESIZ);
	}
	if (r == -1)
		die("tls_read: %s\n", tls_error(t));

	return buf;
}

int
edial(const char *host, const char *port)
{
	struct addrinfo hints, *res, *res0;
	int error, save_errno, s;
	const char *cause = NULL;
	struct timeval timeout;

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_flags = AI_NUMERICSERV; /* numeric port only */
	if ((error = getaddrinfo(host, port, &hints, &res0)))
		die("%s: %s: %s:%s\n", __func__, gai_strerror(error), host, port);
	s = -1;
	for (res = res0; res; res = res->ai_next) {
		s = socket(res->ai_family, res->ai_socktype,
		           res->ai_protocol);
		if (s == -1) {
			cause = "socket";
			continue;
		}

		timeout.tv_sec = MAX_RESPONSETIMEOUT;
		timeout.tv_usec = 0;
		if (setsockopt(s, SOL_SOCKET, SO_SNDTIMEO, &timeout, sizeof(timeout)) == -1)
			die("%s: setsockopt: %s\n", __func__, strerror(errno));

		timeout.tv_sec = MAX_RESPONSETIMEOUT;
		timeout.tv_usec = 0;
		if (setsockopt(s, SOL_SOCKET, SO_RCVTIMEO, &timeout, sizeof(timeout)) == -1)
			die("%s: setsockopt: %s\n", __func__, strerror(errno));

		if (connect(s, res->ai_addr, res->ai_addrlen) == -1) {
			cause = "connect";
			save_errno = errno;
			close(s);
			errno = save_errno;
			s = -1;
			continue;
		}
		break;
	}
	if (s == -1)
		die("%s: %s: %s:%s\n", __func__, cause, host, port);
	freeaddrinfo(res0);

	return s;
}

char *
request(const char *host, const char *path, const char *headers)
{
	struct tls *t;
	char request[4096];
	char *data;
	size_t len;
	ssize_t w;
	int fd;

	/* use HTTP/1.0, don't use HTTP/1.1 using ugly chunked-encoding */
	snprintf(request, sizeof(request),
		"GET %s HTTP/1.0\r\n"
		"Host: %s\r\n"
		"Accept-Language: en-US,en;q=0.5\r\n"
		"Connection: close\r\n"
		"%s"
		"\r\n", path, host, headers);

	if (tls_init() == -1)
		die("tls_init\n");

	if (!(t = tls_client()))
		die("tls_client: %s\n", tls_error(t));

	fd = edial(host, "443");

	if (tls_connect_socket(t, fd, host) == -1)
		die("tls_connect: %s\n", tls_error(t));

	data = request;
	len = strlen(data);
	while (len > 0) {
		w = tls_write(t, data, len);
		if (w == TLS_WANT_POLLIN || w == TLS_WANT_POLLOUT)
			continue;
		else if (w == -1)
			die("tls_write: %s\n", tls_error(t));
		data += w;
		len -= w;
	}

	data = readtls(t);

	tls_close(t);
	tls_free(t);

	return data;
}

/* DEBUG */
char *
readfile(const char *file)
{
	FILE *fp;
	char *buf;
	size_t n, len = 0, size = 0;

	fp = fopen(file, "rb");
	if (!fp)
		die("fopen");
	buf = calloc(1, size + 1); /* always allocate an empty buffer */
	if (!buf)
		die("calloc");
	while (!feof(fp)) {
		if (len + READ_BUF_SIZ + 1 > size) {
			/* allocate size: common case is small textfiles */
			size += READ_BUF_SIZ;
			if (!(buf = realloc(buf, size + 1))) {
				fprintf(stderr, "realloc: %s\n", strerror(errno));
				exit(1);
			}
		}
		if (!(n = fread(&buf[len], 1, READ_BUF_SIZ, fp)))
			break;
		len += n;
		buf[len] = '\0';
		if (n != READ_BUF_SIZ)
			break;
	}
	if (ferror(fp)) {
		fprintf(stderr, "fread: file: %s: %s\n", file, strerror(errno));
		exit(1);
	}
	fclose(fp);

	return buf;
}
