#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <locale.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <wchar.h>

#include "https.h"
#include "json.h"
#include "twitch.h"
#include "util.h"

#define OUT(s)       (fputs((s), stdout))
#define OUTTEXT(s)   gophertext(stdout, s, strlen(s))
#define OUTLINK(s)   gophertext(stdout, s, strlen(s))

extern char **environ;

static const char *baserel = "/twitch.cgi";
static const char *host = "127.0.0.1", *port = "70";

/* page variables */
static char *title = "", *pagetitle = "";

void
line(int _type, const char *username, const char *selector)
{
	putchar(_type);
	OUTTEXT(username);
	putchar('\t');
	OUTLINK(selector);
	printf("\t%s\t%s\r\n", host, port);
}

void
error(const char *s)
{
	line('3', s, "");
}

void
info(const char *s)
{
	line('i', s, "");
}

void
dir(const char *username, const char *selector)
{
	line('1', username, selector);
}

void
html(const char *username, const char *selector)
{
	line('h', username, selector);
}

void
page(int _type, const char *username, const char *page)
{
	putchar(_type);
	OUTTEXT(username);
	putchar('\t');
	printf("%s?p=%s", baserel, page);
	printf("\t%s\t%s\r\n", host, port);
}

static int
gamecmp_name(const void *v1, const void *v2)
{
	struct game *g1 = (struct game *)v1;
	struct game *g2 = (struct game *)v2;

	return strcmp(g1->name, g2->name);
}

void
header(void)
{
	putchar('i');
	if (title[0]) {
		OUTTEXT(title);
		OUT(" - ");
	}
	if (pagetitle[0]) {
		OUTTEXT(pagetitle);
		OUT(" - ");
	}
	printf("Twitch.tv\t%s\t%s\t%s\r\n", "", host, port);
	info("---");
	page('1', "Featured", "featured");
	page('1', "Games", "games");
	page('1', "VODS", "vods");
	dir("Source-code", "/git/frontends");
	page('1', "Links", "links");
	info("---");
}

void
footer(void)
{
	printf(".\r\n");
}

void
render_links(void)
{
	header();
	info("");
	html("mpv player",    "URL:https://mpv.io/installation/");
	html("youtube-dl",    "URL:https://github.com/ytdl-org/youtube-dl");
	html("VLC",           "URL:https://www.videolan.org/");
	html("Twitch.tv API", "URL:https://dev.twitch.tv/docs");
	footer();
}

void
render_games_top(struct games_response *r)
{
	struct game *game;
	size_t i;

	header();
	info("Name");
	for (i = 0; i < r->nitems; i++) {
		game = &(r->data[i]);

		putchar('1');
		OUTTEXT(game->name);
		printf("\t%s?p=streams&game_id=", baserel);
		OUTLINK(game->id);
		printf("\t%s\t%s\r\n", host, port);
	}
	footer();
}

void
render_streams(struct streams_response *r, const char *game_id)
{
	struct stream *stream;
	char buf[256], title[256];
	size_t i;

	header();
	if (!game_id[0])
		printf("i%-20s %-20s %-50s %7s\t%s\t%s\t%s\r\n",
		        "Game", "Name", "Title", "Viewers", "", host, port);
	else
		printf("i%-20s %-50s %7s\t%s\t%s\t%s\r\n",
		       "Name", "Title", "Viewers", "", host, port);

	for (i = 0; i < r->nitems; i++) {
		stream = &(r->data[i]);

		if (stream->user)
			putchar('h');
		else
			putchar('i');

		if (!game_id[0]) {
			if (stream->game) {
				if (utf8pad(buf, sizeof(buf), stream->game->name, 20, ' ') != -1)
					OUTTEXT(buf);
			} else {
				printf("%20s", "");
			}
			OUT(" ");
		}

		if (utf8pad(buf, sizeof(buf), stream->user_name, 20, ' ') != -1)
			OUTTEXT(buf);
		OUT(" ");

		if (stream->language[0])
			snprintf(title, sizeof(title), "[%s] %s", stream->language, stream->title);
			if (utf8pad(buf, sizeof(buf), title, 50, ' ') != -1)
				OUTTEXT(buf);
		else {
			if (utf8pad(buf, sizeof(buf), stream->title, 50, ' ') != -1)
				OUTTEXT(buf);
		}

		printf(" %7lld\t", stream->viewer_count);

		if (stream->user) {
			OUT("URL:https://www.twitch.tv/");
			OUTLINK(stream->user->login);
		}
		printf("\t%s\t%s\r\n", host, port);
	}
	footer();
}

void
render_videos_atom(struct videos_response *r, const char *login)
{
	struct video *video;
	size_t i;

	OUT("<feed xmlns=\"http://www.w3.org/2005/Atom\" xml:lang=\"en\">\n");
	for (i = 0; i < r->nitems; i++) {
		video = &(r->data[i]);

		OUT("<entry>\n");
		OUT("\t<title type=\"text\">");
		xmlencode(video->title);
		OUT("</title>\n");
		OUT("\t<link rel=\"alternate\" type=\"text/html\" href=\"");
		xmlencode(video->url);
		OUT("\" />\n");
		OUT("\t<id>");
		xmlencode(video->url);
		OUT("</id>\n");
		OUT("\t<updated>");
		xmlencode(video->created_at);
		OUT("</updated>\n");
		OUT("\t<published>");
		xmlencode(video->created_at);
		OUT("</published>\n");
		OUT("</entry>\n");
	}
	OUT("</feed>\n");
}

void
render_videos(struct videos_response *r, const char *login)
{
	struct video *video;
	char buf[256];
	size_t i;

	header();

	page('7', "Submit Twitch login name to list VODs", "vods");
	info("");

	/* no results or no user_id parameter: quick exit */
	if (r == NULL) {
		footer();
		return;
	}

	/* link to Atom format (text). */
	if (login[0]) {
		OUT("0Atom feed for ");
		OUTLINK(login);
		printf("\t%s?p=vods&format=atom&login=", baserel);
		OUTLINK(login);
		printf("\t%s\t%s\r\n", host, port);
		info("");
	}

	printf("i%-20s %-50s %-10s %7s\t%s\t%s\t%s\r\n",
	       "Created", "Title", "Duration", "Views", "", host, port);

	for (i = 0; i < r->nitems; i++) {
		video = &(r->data[i]);

		putchar('h');
		if (utf8pad(buf, sizeof(buf), video->created_at, 20, ' ') != -1)
			OUTLINK(buf);
		OUT(" ");
		if (utf8pad(buf, sizeof(buf), video->title, 50, ' ') != -1)
			OUTLINK(buf);
		OUT(" ");
		if (utf8pad(buf, sizeof(buf), video->duration, 10, ' ') != -1)
			OUTLINK(buf);

		printf(" %7lld\t", video->view_count);
		OUT("URL:");
		OUTLINK(video->url);
		printf("\t%s\t%s\r\n", host, port);
	}
	footer();
}

void
handle_streams(void)
{
	struct streams_response *r;
	struct users_response *ru = NULL;
	struct games_response *rg = NULL;
	char game_id[32] = "";
	char *p, *querystring;

	pagetitle = "Streams";

	/* parse "game_id" parameter */
	if ((querystring = getenv("QUERY_STRING"))) {
		if ((p = getparam(querystring, "game_id"))) {
			if (decodeparam(game_id, sizeof(game_id), p) == -1)
				game_id[0] = '\0';
		}
	}

	if (game_id[0])
		r = twitch_streams_bygame(game_id);
	else
		r = twitch_streams();

	if (r == NULL)
		return;

	/* find detailed games data with streams */
	if (!game_id[0])
		rg = twitch_streams_games(r);

	/* find detailed user data with streams */
	ru = twitch_streams_users(r);

	if (pledge("stdio", NULL) == -1)
		exit(1);

	render_streams(r, game_id);

	free(r);
	free(rg);
	free(ru);
}

void
handle_videos(void)
{
	struct videos_response *r = NULL;
	struct users_response *ru = NULL;
	char user_id[32] = "", login[64] = "", format[6] = "";
	char *p, *querystring;

	pagetitle = "Videos";

	/* parse "user_id" or "login" parameter */
	if ((querystring = getenv("QUERY_STRING"))) {
		if ((p = getparam(querystring, "user_id"))) {
			if (decodeparam(user_id, sizeof(user_id), p) == -1)
				user_id[0] = '\0';
		}
		if ((p = getparam(querystring, "login"))) {
			if (decodeparam(login, sizeof(login), p) == -1)
				login[0] = '\0';
		}
		if ((p = getparam(querystring, "format"))) {
			if (decodeparam(format, sizeof(format), p) == -1)
				format[0] = '\0';
		}
	}

	/* login: if not set as query string parameter then use gopher search
	   parameter */
	if (login[0] == '\0') {
		if (!(p = getenv("X_GOPHER_SEARCH"))) /* geomyidae */
			p = getenv("SEARCHREQUEST"); /* gophernicus */
		if (p && decodeparam(login, sizeof(login), p) == -1)
			login[0] = '\0';
	}

	/* no parameter given, show form */
	if (!user_id[0] && !login[0]) {
		if (pledge("stdio", NULL) == -1)
			exit(1);

		render_videos(r, "");
		return;
	}

	if (user_id[0]) {
		r = twitch_videos_byuserid(user_id);
	} else {
		ru = twitch_users_bylogin(login);
		if (ru && ru->nitems > 0)
			r = twitch_videos_byuserid(ru->data[0].id);
	}

	if (pledge("stdio", NULL) == -1)
		exit(1);

	if (r && r->nitems > 0)
		title = r->data[0].user_name;

	if (!strcmp(format, "atom"))
		render_videos_atom(r, login);
	else
		render_videos(r, login);

	free(ru);
	free(r);
}

void
handle_games_top(void)
{
	struct games_response *r;

	pagetitle = "Top 100 games";

	if (!(r = twitch_games_top()))
		return;

	if (pledge("stdio", NULL) == -1)
		exit(1);

	/* sort by name alphabetically, NOTE: the results are the top 100
	   sorted by viewcount. View counts are not visible in the new
	   Helix API data). */
	qsort(r->data, r->nitems, sizeof(r->data[0]), gamecmp_name);

	render_games_top(r);

	free(r);
}

void
handle_links(void)
{
	if (pledge("stdio", NULL) == -1)
		exit(1);

	pagetitle = "Links";

	render_links();
}

int
main(void)
{
	char *p, path[256] = "", *querystring;

	setlocale(LC_CTYPE, "");

	if (pledge("stdio dns inet rpath unveil", NULL) == -1 ||
	    unveil(TLS_CA_CERT_FILE, "r") == -1 ||
	    unveil(NULL, NULL) == -1) {
		exit(1);
	}

	if ((p = getenv("SERVER_NAME")))
		host = p;
	if ((p = getenv("SERVER_PORT")))
		port = p;

	if (!(querystring = getenv("QUERY_STRING")))
		querystring = "";

	if ((p = getparam(querystring, "p"))) {
		if (decodeparam(path, sizeof(path), p) == -1)
			path[0] = '\0';
	}

	if (!strcmp(path, "") ||
	    !strcmp(path, "featured") ||
	    !strcmp(path, "streams")) {
		/* featured / by game id */
		handle_streams();
	} else if (!strcmp(path, "topgames") ||
	           !strcmp(path, "games")) {
		handle_games_top();
	} else if (!strcmp(path, "videos") ||
	           !strcmp(path, "vods")) {
		handle_videos();
	} else if (!strcmp(path, "links")) {
		handle_links();
	} else {
		error("Not Found");
		printf(".\r\n");
		exit(1);
	}

	return 0;
}
