#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "https.h"
#include "reddit.h"
#include "util.h"

#define OUT(s) (fputs((s), stdout))
#define OUTLINK(s) gophertext(stdout, s, strlen(s))
#define OUTTEXT(s) gophertext(stdout, s, strlen(s))

static const char *baserel = "/reddit.cgi";
static const char *host = "127.0.0.1", *port = "70";

static char before[16], after[16], subreddit[1024];

void
line(int _type, const char *username, const char *selector)
{
	putchar(_type);
	OUTTEXT(username);
	putchar('\t');
	OUTLINK(selector);
	printf("\t%s\t%s\r\n", host, port);
}

void
error(const char *s)
{
	line('3', s, "");
}

void
info(const char *s)
{
	line('i', s, "");
}

void
dir(const char *username, const char *selector)
{
	line('1', username, selector);
}

void
html(const char *username, const char *selector)
{
	line('h', username, selector);
}

void
page(int _type, const char *username, const char *page)
{
	putchar(_type);
	OUTTEXT(username);
	putchar('\t');
	printf("%s?p=%s", baserel, page);
	printf("\t%s\t%s\r\n", host, port);
}

void
header(void)
{
}

void footer(void)
{
	printf(".\r\n");
}

void
printitem(struct item *item)
{
	if (!item || !item->title[0])
		return;

	struct tm *tm = gmtime(&(item->created_utc));

	putchar('h');
	OUTTEXT(item->title);
	OUT(" by ");
	OUTTEXT(item->author);
	OUT(" at ");
	printf("%04d-%02d-%02d %02d:%02d",
		tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday,
		tm->tm_hour, tm->tm_min);
	OUT("\tURL:");
	OUTLINK(item->url);
	printf("\t%s\t%s\r\n", host, port);

	/* if global (no subreddit), show where it was posted */
	if (!subreddit[0]) {
		printf("1Posted in r/");
		OUTTEXT(item->subreddit);
		OUT("\t");
		OUTTEXT(baserel);
		OUT("?subreddit=");
		OUTTEXT(item->subreddit);
		printf("\t%s\t%s\r\n", host, port);
	}

	printf("hUpvotes: %ld, downvotes %ld, comments: %ld",
	       item->ups, item->downs, item->num_comments);
	OUT("\tURL:https://old.reddit.com/r/");
	OUTTEXT(item->subreddit);
	OUTTEXT("/comments/");
	if (reddit_isvalidlink(item->name) && !strncmp(item->name, "t3_", 3))
		OUTTEXT(item->name + 3);
	printf("/\t%s\t%s\r\n", host, port);

	/* TODO: also noticed "spoiler" as value, ignore? */
	if (item->thumbnail[0] &&
	    strcmp(item->thumbnail, "self") &&
	    strcmp(item->thumbnail, "default")) {
		putchar('h');
		OUT("Thumbnail");
		OUT("\tURL:");
		OUTTEXT(item->thumbnail);
		printf("\t%s\t%s\r\n", host, port);
	}

	if (item->is_video) {
		putchar('h');
		OUT("Video, duration: ");
		printf("%ld, url: ", item->duration);
		OUTTEXT(item->dash_url);
		OUT("\tURL:");
		OUTTEXT(item->dash_url);
		printf("\t%s\t%s\r\n", host, port);
	}

	info("");
}

int
render_pagination(struct list_response *r)
{
	int i = 0;

	if (r->before[0]) {
		printf("1Previous page");
		OUT("\t");
		OUTTEXT(baserel);
		OUT("?subreddit=");
		OUTTEXT(subreddit);
		OUT("&before=");
		OUTTEXT(r->before);
		printf("\t%s\t%s\r\n", host, port);
		i++;
	}
	if (r->after[0]) {
		printf("1Next page");
		OUT("\t");
		OUTTEXT(baserel);
		OUT("?subreddit=");
		OUTTEXT(subreddit);
		OUT("&after=");
		OUTTEXT(r->after);
		printf("\t%s\t%s\r\n", host, port);
		i++;
	}
	return i;
}

int
render(struct list_response *r)
{
	size_t i;

	header();

	if (pledge("stdio", NULL) == -1)
		exit(1);

	if (render_pagination(r))
		info("");

	for (i = 0; i < r->nitems; i++)
		printitem(&(r->items[i]));

	render_pagination(r);

	footer();

	return 0;
}

static void
usage(void)
{
	printf("3Specify a subreddit\t\t%s\t%s\r\n", host, port);
	printf(".\r\n");
	exit(1);
}

int
main(int argc, char *argv[])
{
	struct list_response *r;
	char *querystring, *p, *search;

	if (pledge("stdio dns inet rpath unveil", NULL) == -1)
		exit(1);
	if (unveil(TLS_CA_CERT_FILE, "r") == -1 ||
	    unveil(NULL, NULL) == -1)
		exit(1);

	if ((p = getenv("SERVER_NAME")))
		host = p;
	if ((p = getenv("SERVER_PORT")))
		port = p;

	if (!(querystring = getenv("QUERY_STRING")))
		querystring = "";

	if ((p = getparam(querystring, "subreddit"))) {
		if (decodeparam(subreddit, sizeof(subreddit), p) == -1)
			subreddit[0] = '\0';
	} else {
		if (!(search = getenv("X_GOPHER_SEARCH"))) /* geomyidae */
			search = getenv("SEARCHREQUEST"); /* gophernicus */

		if (search && !uriencode(search, subreddit, sizeof(subreddit)))
			usage();
	}

	if ((p = getparam(querystring, "before"))) {
		if (decodeparam(before, sizeof(before), p) == -1)
			before[0] = '\0';
		if (!reddit_isvalidlink(before))
			before[0] = '\0';
	}

	if ((p = getparam(querystring, "after"))) {
		if (decodeparam(after, sizeof(after), p) == -1)
			after[0] = '\0';
		if (!reddit_isvalidlink(after))
			after[0] = '\0';
	}

	r = reddit_list(subreddit, 100, before, after);
	if (!r || r->nitems == 0) {
		error("No items found");
		printf(".\r\n");
		exit(1);
	}

	render(r);

	return 0;
}
