/*
*
* gramscii: a simple editor for ASCII box-and-arrow charts
*
* Copyright (c) 2019 Vincenzo "KatolaZ" Nicosia <katolaz@freaknet.org>
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program. Please see the attached file COPYING. 
* Otherwise, please visit <https://www.gnu.org/licenses/>.
*
*/

#define _POSIX_C_SOURCE 2 

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#include "arg.h"
#include "gramscii.h"


char *argv0;


void cleanup(int s){
	
	if (!silent)
		printf("\033[;H\033[2J");
	else
		dump_lines(screen, stdout);
	tcsetattr(0, TCSANOW, &t1);
	fflush(stdout);
	exit(s);
}

/*** Initialisation ***/

void init(){

	signal(SIGHUP, cleanup);
	signal(SIGINT, cleanup);
	signal(SIGTERM, cleanup);
	signal(SIGQUIT, cleanup);

	tcgetattr(0, &t1);
	t2 = t1;
	t2.c_lflag &= ~(ICANON | ECHO);
	tcsetattr(0, TCSANOW, &t2);

	init_screen();
	x = 0;
	y = 0;
	step = 1;
	modified = 0;
	fname[0] = '\0';
	redraw();
}


/*** Commands ***/

void commands(FILE *fc){

	char c;
	while((c=fgetc(fc))!=EOF){
		if (!change_style(c) && !move_around(c, fc, 1)){
			switch(c){
				case 'i':
					mode = TEXT;
					get_text(fc);
					break;
				case 'R':
					redraw();
					break;
				case 'b':
					mode = BOX;
					get_box(fc);
					break;
				case 'A': autoend=1;
				case 'a':
					mode = ARROW;
					get_arrow(fc);
					autoend = 0;
					break;
				case 'W':
					force_new = 1;/** FALLTHROUGH **/
				case 'w':
					write_file(fc);
					break;
				case 'e':
					check_modified(fc);/** FALLTHROUGH **/
				case 'E':
					load_file(fc);
					break;
				case 'N':
					new_file(fc);
					break;
				case 'x':
					mode = DEL;
					erase(fc);
					break;
				case 'v':
					mode = VIS;
					visual_box(fc);
					break;
				case 'C':
					crop_to_nonblank();
					break;
				case 'p':
					paste();
					break;
				case 'u':
					undo_change();
					break;
				case 'U':
					redo_change();
					break;
				case 'r':
					read_file_at(fc, x, y);
					break;
				case 'q':
					check_modified(fc);/** FALLTHROUGH **/
				case 'Q':
					cleanup(0);
					break;
			}
		}
		check_bound(&x, &y);
		status_bar();
		show_cursor();
		step = 1;
		force_new = 0;
	}

}

void usage(){
	fprintf(stderr, "Usage: %s [-s] [-h] [file ...]\n", argv0);
	cleanup(1);
}


int main(int argc, char *argv[]){
	FILE *fc;

	ARGBEGIN {
		case 's':
			silent = 1;
			break;
		case 'h': /* FALLTHROUGH */
		default:
			usage();
	} ARGEND;	

	init();
	while (argc){
		fc = fopen(argv[0], "r");
		if (fc == NULL){
			fprintf(stderr, "Error opening file %s\n", argv[0]);
		}
		else {
			commands(fc);
			fclose(fc);
			redraw();
		}
		argv++;
		argc--;
	}
	commands(stdin);
	return 0;
}
