/* Top-level block layer implementation */
#include <sys/types.h>
#include <sys/stat.h>

#include <fcntl.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sodium.h>

#include "block.h"
#include "config.h"
#include "misc.h"

static int
bhash(void *buf, size_t n, unsigned char *md)
{
	return crypto_generichash(md, MDSIZE, buf, n, NULL, 0);
}

int
bcreat(char *path, int mode, struct bctx **bctx)
{
	if (path == NULL || bctx == NULL) {
		seterr("invalid params");
		return -1;
	}

	if (sodium_init() < 0) {
		seterr("sodium_init: failed");
		return -1;
	}

	*bctx = calloc(1, sizeof(**bctx));
	if (*bctx == NULL) {
		seterr("calloc: out of memory");
		return -1;
	}

	if (bcompressops()->creat(*bctx, path, mode) < 0) {
		free(*bctx);
		return -1;
	}
	return 0;
}

int
bopen(char *path, int flags, int mode, struct bctx **bctx)
{
	if (path == NULL || bctx == NULL) {
		seterr("invalid params");
		return -1;
	}

	if (sodium_init() < 0) {
		seterr("sodium_init: failed");
		return -1;
	}

	*bctx = calloc(1, sizeof(**bctx));
	if (*bctx == NULL) {
		seterr("calloc: out of memory");
		return -1;
	}

	if (bcompressops()->open(*bctx, path, flags, mode) < 0) {
		free(*bctx);
		return -1;
	}
	return 0;
}

int
bput(struct bctx *bctx, void *buf, size_t n, unsigned char *md)
{
	if (bctx == NULL || buf == NULL || n == 0 || md == NULL) {
		seterr("invalid params");
		return -1;
	}

	if (bhash(buf, n, md) < 0) {
		seterr("bhash: failed");
		return -1;
	}

	return bcompressops()->put(bctx, buf, n, md);
}

int
bget(struct bctx *bctx, unsigned char *md, void *buf, size_t *n)
{
	if (bctx == NULL || md == NULL || buf == NULL || n == NULL) {
		seterr("invalid params");
		return -1;
	}

	return bcompressops()->get(bctx, md, buf, n);
}

int
brm(struct bctx *bctx, unsigned char *md)
{
	if (bctx == NULL || md == NULL) {
		seterr("invalid params");
		return -1;
	}

	return bcompressops()->rm(bctx, md);
}

int
bgc(struct bctx *bctx)
{
	if (bctx == NULL) {
		seterr("invalid params");
		return -1;
	}

	return bcompressops()->gc(bctx);
}

/*
 * Lookup the block given the hash and rehash it.
 * Check that the hashes match.  It returns -1
 * on error, 0 on success and 1 if a block hash
 * mismatch is detected.
 */
int
bcheck(struct bctx *bctx, unsigned char *md)
{
	unsigned char tmp[MDSIZE];
	void *buf;
	size_t n;

	if (bctx == NULL || md == NULL) {
		seterr("invalid params");
		return -1;
	}

	buf = malloc(BSIZEMAX);
	if (buf == NULL) {
		seterr("malloc: out of memory");
		return -1;
	}
	n = BSIZEMAX;

	if (bcompressops()->get(bctx, md, buf, &n) < 0) {
		free(buf);
		return -1;
	}

	if (bhash(buf, n, tmp) < 0) {
		free(buf);
		seterr("bhash: failed");
		return -1;
	}

	free(buf);

	if (memcmp(tmp, md, MDSIZE) != 0)
		return 1;
	return 0;
}

int
bsync(struct bctx *bctx)
{
	if (bctx == NULL) {
		seterr("invalid params");
		return -1;
	}

	return bcompressops()->sync(bctx);
}

int
bclose(struct bctx *bctx)
{
	int r;

	if (bctx == NULL) {
		seterr("invalid params");
		return -1;
	}

	if (bsync(bctx) < 0)
		return -1;
	r = bcompressops()->close(bctx);
	free(bctx);
	return r;
}
