#include <sys/types.h>

#include <assert.h>
#include <err.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>

#include "blake2.h"
#include "dedup.h"

void
read_snap_hdr(int fd, struct snap_hdr *hdr)
{
	uint8_t buf[SNAP_HDR_SIZE];
	int n;

	if (xread(fd, buf, sizeof(buf)) == 0)
		errx(1, "%s: unexpected EOF", __func__);

	n = unpack(buf, "qqq",
	           &hdr->flags,
	           &hdr->size,
	           &hdr->nr_snaps);

	n += unpack(&buf[n], "qqqqqq",
	            &hdr->st.orig_size,
	            &hdr->st.compr_size,
	            &hdr->st.dedup_size,
	            &hdr->st.min_blk_size,
	            &hdr->st.max_blk_size,
	            &hdr->st.nr_blks);

	n += unpack(&buf[n], "qqqq",
	            &hdr->st.reserved[0],
	            &hdr->st.reserved[1],
	            &hdr->st.reserved[2],
	            &hdr->st.reserved[3]);

	assert(n == SNAP_HDR_SIZE);
}

void
write_snap_hdr(int fd, struct snap_hdr *hdr)
{
	uint8_t buf[SNAP_HDR_SIZE];
	int n;

	n = pack(buf, "qqq",
	         hdr->flags,
	         hdr->size,
	         hdr->nr_snaps);

	n += pack(&buf[n], "qqqqqq",
	          hdr->st.orig_size,
	          hdr->st.compr_size,
	          hdr->st.dedup_size,
	          hdr->st.min_blk_size,
	          hdr->st.max_blk_size,
	          hdr->st.nr_blks);

	n += pack(&buf[n], "qqqq",
	          hdr->st.reserved[0],
	          hdr->st.reserved[1],
	          hdr->st.reserved[2],
	          hdr->st.reserved[3]);

	assert(n == SNAP_HDR_SIZE);
	xwrite(fd, buf, n);
}

void
read_blk_hdr(int fd, struct blk_hdr *hdr)
{
	uint8_t buf[BLK_HDR_SIZE];
	int n;

	if (xread(fd, buf, sizeof(buf)) == 0)
		errx(1, "%s: unexpected EOF", __func__);

	n = unpack(buf, "qq",
	           &hdr->flags,
	           &hdr->size);

	assert(n == BLK_HDR_SIZE);
}

void
write_blk_hdr(int fd, struct blk_hdr *hdr)
{
	uint8_t buf[BLK_HDR_SIZE];
	int n;

	n = pack(buf, "qq",
	         hdr->flags,
	         hdr->size);

	assert(n == BLK_HDR_SIZE);
	xwrite(fd, buf, n);
}

void
read_blk_desc(int fd, struct blk_desc *desc)
{
	uint8_t buf[BLK_DESC_SIZE];
	char fmt[BUFSIZ];
	int n;

	if (xread(fd, buf, sizeof(buf)) == 0)
		errx(1, "%s: unexpected EOF", __func__);

	snprintf(fmt, sizeof(fmt), "'%dqq", MD_SIZE);
	n = unpack(buf, fmt,
	           desc->md,
	           &desc->offset,
	           &desc->size);

	assert(n == BLK_DESC_SIZE);
}

void
write_blk_desc(int fd, struct blk_desc *desc)
{
	uint8_t buf[BLK_DESC_SIZE];
	char fmt[BUFSIZ];
	int n;

	snprintf(fmt, sizeof(fmt), "'%dqq", MD_SIZE);
	n = pack(buf, fmt,
	         desc->md,
	         desc->offset,
	         desc->size);

	assert(n == BLK_DESC_SIZE);
	xwrite(fd, buf, n);
}

void
read_snap(int fd, struct snap *snap)
{
	uint8_t buf[SNAPSHOT_SIZE];
	char fmt[BUFSIZ];
	int n;

	if (xread(fd, buf, sizeof(buf)) == 0)
		errx(1, "%s: unexpected EOF", __func__);

	snprintf(fmt, sizeof(fmt), "q'%d'%dq", MSG_SIZE, MD_SIZE);
	n = unpack(buf, fmt,
	           &snap->size,
	           snap->msg,
	           snap->md,
	           &snap->nr_blk_descs);

	assert(n == SNAPSHOT_SIZE);
};

void
read_snap_descs(int fd, struct snap *snap)
{
	uint64_t i;

	for (i = 0; i < snap->nr_blk_descs; i++)
		read_blk_desc(fd, &snap->blk_desc[i]);
}

void
write_snap(int fd, struct snap *snap)
{
	uint8_t buf[SNAPSHOT_SIZE];
	char fmt[BUFSIZ];
	int n;

	snprintf(fmt, sizeof(fmt), "q'%d'%dq", MSG_SIZE, MD_SIZE);
	n = pack(buf, fmt,
	         snap->size,
	         snap->msg,
	         snap->md,
	         snap->nr_blk_descs);

	assert(n == SNAPSHOT_SIZE);
	xwrite(fd, buf, n);
}

void
write_snap_blk_descs(int fd, struct snap *snap)
{
	uint64_t i;

	for (i = 0; i < snap->nr_blk_descs; i++)
		write_blk_desc(fd, &snap->blk_desc[i]);
}
