#include <sys/types.h>
#include <sys/stat.h>

#include <err.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "dedup.h"
#include "tree.h"

struct cache_node {
	struct cache_entry ent;
	RB_ENTRY(cache_node) e;
};
RB_HEAD(cache_head, cache_node);

struct cache {
	struct cache_head nodes;
};

static int
cache_node_cmp(struct cache_node *e1, struct cache_node *e2)
{
	int r;

 	r = memcmp(e1->ent.md, e2->ent.md, sizeof(e1->ent.md));
	if (r > 0)
		return 1;
	else if (r < 0)
		return -1;
	return 0;
}
static RB_PROTOTYPE(cache_head, cache_node, e, cache_node_cmp);
static RB_GENERATE(cache_head, cache_node, e, cache_node_cmp);

static struct cache_node *
alloc_cache_node(struct cache_entry *ent)
{
	struct cache_node *node;

	node = calloc(1, sizeof(*node));
	if (node == NULL)
		err(1, "calloc");
	node->ent = *ent;
	return node;
}

static void
free_cache_node(struct cache_node *node)
{
	free(node);
}

struct cache *
alloc_cache(void)
{
	struct cache *cache;

	cache = calloc(1, sizeof(*cache));
	if (cache == NULL)
		err(1, "calloc");
	RB_INIT(&cache->nodes);
	return cache;
}

void
free_cache(struct cache *cache)
{
	struct cache_node *node, *tmp;

	RB_FOREACH_SAFE(node, cache_head, &cache->nodes, tmp) {
		RB_REMOVE(cache_head, &cache->nodes, node);
		free_cache_node(node);
	}
	free(cache);
}

void
add_cache_entry(struct cache *cache, struct cache_entry *ent)
{
	struct cache_node *node;

	node = alloc_cache_node(ent);
	RB_INSERT(cache_head, &cache->nodes, node);
}

int
lookup_cache_entry(struct cache *cache, struct cache_entry *ent)
{
	struct cache_node *node, key;

	key.ent = *ent;
	node = RB_FIND(cache_head, &cache->nodes, &key);
	if (node != NULL) {
		*ent = node->ent;
		return 0;
	}
	return -1;
}

void
walk_cache(struct cache *cache, int (*fn)(struct cache_entry *))
{
	struct cache_node *node;

	RB_FOREACH(node, cache_head, &cache->nodes)
		(*fn)(&node->ent);
}
