#include <err.h>
#include <locale.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <wchar.h>

#include <git2.h>

#define PAD_TRUNCATE_SYMBOL    "\xe2\x80\xa6" /* symbol: "ellipsis" */
#define UTF_INVALID_SYMBOL     "\xef\xbf\xbd" /* symbol: "replacement" */

static git_repository *repo;

static const char *relpath = "";

static char description[255] = "Repositories";
static char *name = "";

/* Format `len' columns of characters. If string is shorter pad the rest
 * with characters `pad`. */
int
utf8pad(char *buf, size_t bufsiz, const char *s, size_t len, int pad)
{
	wchar_t wc;
	size_t col = 0, i, slen, siz = 0;
	int inc, rl, w;

	if (!bufsiz)
		return -1;
	if (!len) {
		buf[0] = '\0';
		return 0;
	}

	slen = strlen(s);
	for (i = 0; i < slen; i += inc) {
		inc = 1; /* next byte */
		if ((unsigned char)s[i] < 32)
			continue;

		rl = mbtowc(&wc, &s[i], slen - i < 4 ? slen - i : 4);
		inc = rl;
		if (rl < 0) {
			mbtowc(NULL, NULL, 0); /* reset state */
			inc = 1; /* invalid, seek next byte */
			w = 1; /* replacement char is one width */
		} else if ((w = wcwidth(wc)) == -1) {
			continue;
		}

		if (col + w > len || (col + w == len && s[i + inc])) {
			if (siz + 4 >= bufsiz)
				return -1;
			memcpy(&buf[siz], PAD_TRUNCATE_SYMBOL, sizeof(PAD_TRUNCATE_SYMBOL) - 1);
			siz += sizeof(PAD_TRUNCATE_SYMBOL) - 1;
			buf[siz] = '\0';
			col++;
			break;
		} else if (rl < 0) {
			if (siz + 4 >= bufsiz)
				return -1;
			memcpy(&buf[siz], UTF_INVALID_SYMBOL, sizeof(UTF_INVALID_SYMBOL) - 1);
			siz += sizeof(UTF_INVALID_SYMBOL) - 1;
			buf[siz] = '\0';
			col++;
			continue;
		}
		if (siz + inc + 1 >= bufsiz)
			return -1;
		memcpy(&buf[siz], &s[i], inc);
		siz += inc;
		buf[siz] = '\0';
		col += w;
	}

	len -= col;
	if (siz + len + 1 >= bufsiz)
		return -1;
	memset(&buf[siz], pad, len);
	siz += len;
	buf[siz] = '\0';

	return 0;
}

/* Escape characters in text in geomyidae .gph format,
   newlines are ignored */
void
gphtext(FILE *fp, const char *s, size_t len)
{
	size_t i;

	for (i = 0; *s && i < len; s++, i++) {
		switch (*s) {
		case '\r': /* ignore CR */
		case '\n': /* ignore LF */
			break;
		case '\t':
			fputs("        ", fp);
			break;
		default:
			putc(*s, fp);
			break;
		}
	}
}

/* Escape characters in links in geomyidae .gph format */
void
gphlink(FILE *fp, const char *s, size_t len)
{
	size_t i;

	for (i = 0; *s && i < len; s++, i++) {
		switch (*s) {
		case '\r': /* ignore CR */
		case '\n': /* ignore LF */
			break;
		case '\t':
			fputs("        ", fp);
			break;
		case '|': /* escape separators */
			fputs("\\|", fp);
			break;
		default:
			putc(*s, fp);
			break;
		}
	}
}

void
joinpath(char *buf, size_t bufsiz, const char *path, const char *path2)
{
	int r;

	r = snprintf(buf, bufsiz, "%s%s%s",
		path, path[0] && path[strlen(path) - 1] != '/' ? "/" : "", path2);
	if (r < 0 || (size_t)r >= bufsiz)
		errx(1, "path truncated: '%s%s%s'",
			path, path[0] && path[strlen(path) - 1] != '/' ? "/" : "", path2);
}

void
printtimeshort(FILE *fp, const git_time *intime)
{
	struct tm *intm;
	time_t t;
	char out[32];

	t = (time_t)intime->time;
	if (!(intm = gmtime(&t)))
		return;
	strftime(out, sizeof(out), "%Y-%m-%d %H:%M", intm);
	fputs(out, fp);
}

void
writeheader(FILE *fp)
{
	if (description[0]) {
		putchar('t');
		gphtext(fp, description, strlen(description));
		fputs("\n\n", fp);
	}

	fprintf(fp, "%-20.20s  ", "Name");
	fprintf(fp, "%-39.39s  ", "Description");
	fprintf(fp, "%s\n", "Last commit");
}

int
writelog(FILE *fp)
{
	git_commit *commit = NULL;
	const git_signature *author;
	git_revwalk *w = NULL;
	git_oid id;
	char *stripped_name = NULL, *p;
	char buf[1024];
	int ret = 0;

	git_revwalk_new(&w, repo);
	git_revwalk_push_head(w);
	git_revwalk_simplify_first_parent(w);

	if (git_revwalk_next(&id, w) ||
	    git_commit_lookup(&commit, repo, &id)) {
		ret = -1;
		goto err;
	}

	author = git_commit_author(commit);

	/* strip .git suffix */
	if (!(stripped_name = strdup(name)))
		err(1, "strdup");
	if ((p = strrchr(stripped_name, '.')))
		if (!strcmp(p, ".git"))
			*p = '\0';

	fputs("[1|", fp);
	utf8pad(buf, sizeof(buf), stripped_name, 20, ' ');
	gphlink(fp, buf, strlen(buf));
	fputs("  ", fp);
	utf8pad(buf, sizeof(buf), description, 39, ' ');
	gphlink(fp, buf, strlen(buf));
	fputs("  ", fp);
	if (author)
		printtimeshort(fp, &(author->when));
	fprintf(fp, "|%s/%s/log.gph|server|port]\n", relpath, stripped_name);

	git_commit_free(commit);
err:
	git_revwalk_free(w);
	free(stripped_name);

	return ret;
}

void
usage(const char *argv0)
{
	fprintf(stderr, "%s [-b baseprefix] [repodir...]\n", argv0);
	exit(1);
}

int
main(int argc, char *argv[])
{
	FILE *fp;
	char path[PATH_MAX], repodirabs[PATH_MAX + 1];
	const char *repodir = NULL;
	int i, r, ret = 0;

	setlocale(LC_CTYPE, "");

	git_libgit2_init();

#ifdef __OpenBSD__
	if (pledge("stdio rpath", NULL) == -1)
		err(1, "pledge");
#endif

	for (i = 1, r = 0; i < argc; i++) {
		if (argv[i][0] == '-') {
			if (argv[i][1] != 'b' || i + 1 >= argc)
				usage(argv[0]);
			relpath = argv[++i];
			continue;
		}

		if (r++ == 0)
			writeheader(stdout);

		repodir = argv[i];
		if (!realpath(repodir, repodirabs))
			err(1, "realpath");

		if (git_repository_open_ext(&repo, repodir,
		    GIT_REPOSITORY_OPEN_NO_SEARCH, NULL)) {
			fprintf(stderr, "%s: cannot open repository\n", argv[0]);
			ret = 1;
			continue;
		}

		/* use directory name as name */
		if ((name = strrchr(repodirabs, '/')))
			name++;
		else
			name = "";

		/* read description or .git/description */
		joinpath(path, sizeof(path), repodir, "description");
		if (!(fp = fopen(path, "r"))) {
			joinpath(path, sizeof(path), repodir, ".git/description");
			fp = fopen(path, "r");
		}
		description[0] = '\0';
		if (fp) {
			if (fgets(description, sizeof(description), fp))
				description[strcspn(description, "\t\r\n")] = '\0';
			else
				description[0] = '\0';
			fclose(fp);
		}

		writelog(stdout);
	}
	if (!repodir) {
		fprintf(stderr, "%s [-b baseprefix] [repodir...]\n", argv[0]);
		return 1;
	}

	/* cleanup */
	git_repository_free(repo);
	git_libgit2_shutdown();

	return ret;
}
