#include <sys/stat.h>

#include <err.h>
#include <errno.h>
#include <inttypes.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <wchar.h>

#include <git2.h>

#include "compat.h"

static git_repository *repo;

static const char *relpath = "";

static char description[255] = "Repositories";
static char *name = "";

#ifndef USE_PLEDGE
#define pledge(p1,p2) 0
#endif

/* print `len' columns of characters. If string is shorter pad the rest
 * with characters `pad`. */
void
printutf8pad(FILE *fp, const char *s, size_t len, int pad)
{
	wchar_t w;
	size_t col = 0, i, slen;
	int rl, wc;

	if (!len)
		return;

	slen = strlen(s);
	for (i = 0; i < slen && col < len + 1; i += rl) {
		if ((rl = mbtowc(&w, &s[i], slen - i < 4 ? slen - i : 4)) <= 0)
			break;
		if ((wc = wcwidth(w)) == -1)
			wc = 1;
		col += (size_t)wc;
		if (col >= len && s[i + rl]) {
			fputs("\xe2\x80\xa6", fp);
			break;
		}
		fwrite(&s[i], 1, rl, fp);
	}
	for (; col < len; col++)
		putc(pad, fp);
}

void
trim(char *buf, size_t bufsiz, const char *src)
{
	size_t d = 0, i, len, s;

	len = strlen(src);
	for (s = 0; s < len && d < bufsiz - 1; s++) {
		switch (src[s]) {
		case '\t':
			if (d + 8 >= bufsiz - 1)
				goto end;
			for (i = 0; i < 8; i++)
				buf[d++] = ' ';
			break;
		case '|':
		case '\n':
		case '\r':
			buf[d++] = ' ';
			break;
		default:
			buf[d++] = src[s];
			break;
		}
	}
end:
	buf[d] = '\0';
}

void
joinpath(char *buf, size_t bufsiz, const char *path, const char *path2)
{
	int r;

	r = snprintf(buf, bufsiz, "%s%s%s",
		path, path[0] && path[strlen(path) - 1] != '/' ? "/" : "", path2);
	if (r == -1 || (size_t)r >= bufsiz)
		errx(1, "path truncated: '%s%s%s'",
			path, path[0] && path[strlen(path) - 1] != '/' ? "/" : "", path2);
}

void
printtimeshort(FILE *fp, const git_time *intime)
{
	struct tm *intm;
	time_t t;
	char out[32];

	t = (time_t)intime->time;
	if (!(intm = gmtime(&t)))
		return;
	strftime(out, sizeof(out), "%Y-%m-%d %H:%M", intm);
	fputs(out, fp);
}

void
writeheader(FILE *fp)
{
	char buf[256];

	trim(buf, sizeof(buf), description);
	if (buf[0] == 't' || buf[0] == '[')
		fputc('t', fp);
	fprintf(fp, "%s\n\n", buf);

	fprintf(fp, "%-20.20s  ", "Name");
	fprintf(fp, "%-50.50s  ", "Description");
	fprintf(fp, "%-16.16s\n", "Last commit");
}

int
writelog(FILE *fp)
{
	git_commit *commit = NULL;
	const git_signature *author;
	git_revwalk *w = NULL;
	git_oid id;
	char *stripped_name = NULL, *p;
	char buf[1024];
	int ret = 0;

	git_revwalk_new(&w, repo);
	git_revwalk_push_head(w);
	git_revwalk_sorting(w, GIT_SORT_TIME);
	git_revwalk_simplify_first_parent(w);

	if (git_revwalk_next(&id, w) ||
	    git_commit_lookup(&commit, repo, &id)) {
		ret = -1;
		goto err;
	}

	author = git_commit_author(commit);

	/* strip .git suffix */
	if (!(stripped_name = strdup(name)))
		err(1, "strdup");
	if ((p = strrchr(stripped_name, '.')))
		if (!strcmp(p, ".git"))
			*p = '\0';

	fputs("[1|", fp);
	trim(buf, sizeof(buf), stripped_name);
	printutf8pad(fp, buf, 20, ' ');
	fputs("  ", fp);
	trim(buf, sizeof(buf), description);
	printutf8pad(fp, buf, 50, ' ');
	fputs("  ", fp);
	if (author)
		printtimeshort(fp, &(author->when));
	trim(buf, sizeof(buf), stripped_name);
	fprintf(fp, "|%s/%s/log.gph|server|port]\n", relpath, buf);

	git_commit_free(commit);
err:
	git_revwalk_free(w);
	free(stripped_name);

	return ret;
}

void
usage(const char *argv0)
{
	fprintf(stderr, "%s [-b baseprefix] [repodir...]\n", argv0);
	exit(1);
}

int
main(int argc, char *argv[])
{
	const git_error *e = NULL;
	FILE *fp;
	char path[PATH_MAX], repodirabs[PATH_MAX + 1];
	const char *repodir = NULL;
	int i, ret = 0;

	if (pledge("stdio rpath", NULL) == -1)
		err(1, "pledge");

	git_libgit2_init();

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			if (argv[i][1] != 'b' || i + 1 >= argc)
				usage(argv[0]);
			relpath = argv[++i];
			continue;
		}

		if (i == 1)
			writeheader(stdout);

		repodir = argv[i];
		if (!realpath(repodir, repodirabs))
			err(1, "realpath");

		if (git_repository_open_ext(&repo, repodir,
		    GIT_REPOSITORY_OPEN_NO_SEARCH, NULL)) {
			e = giterr_last();
			fprintf(stderr, "%s: %s\n", argv[0], e->message);
			ret = 1;
			continue;
		}

		/* use directory name as name */
		if ((name = strrchr(repodirabs, '/')))
			name++;
		else
			name = "";

		/* read description or .git/description */
		joinpath(path, sizeof(path), repodir, "description");
		if (!(fp = fopen(path, "r"))) {
			joinpath(path, sizeof(path), repodir, ".git/description");
			fp = fopen(path, "r");
		}
		description[0] = '\0';
		if (fp) {
			if (!fgets(description, sizeof(description), fp))
				description[0] = '\0';
			fclose(fp);
		}

		writelog(stdout);
	}

	/* cleanup */
	git_repository_free(repo);
	git_libgit2_shutdown();

	return ret;
}
