;;; let's hide the loading
(let ((*standard-output* (make-broadcast-stream)))
    (require 'asdf))

(defparameter *tries* 3)
(defparameter *alerts* '())
(defparameter *states-dir* "~/.reed-alert/states/")
(ensure-directories-exist *states-dir*)

(defun color(num1 num2)
  (format nil "~a[~a;~am" #\Escape num1 num2))

(defparameter *red*    (color 1 31))
(defparameter *white*  (color 0 70))
(defparameter *green*  (color 1 32))
(defparameter *yellow* (color 0 33))

;; simple hash function (Fowler Noll Vo)
;; https://en.wikipedia.org/wiki/Fowler%E2%80%93Noll%E2%80%93Vo_hash_function
(defun fnv-hash(string)
  "return a hash from a string"
  (let ((FNV_prime 2)
        (hash 26123230013))
    (loop for octet-of-data across string
       do
         (setf hash (* FNV_prime
                       (logxor hash (char-code octet-of-data)))))
    hash))

;; common-lisp don't have a split string function natively
(defun replace-all (string part replacement &key (test #'char=))
  (with-output-to-string (out)
			 (loop with part-length = (length part)
			       for old-pos = 0 then (+ pos part-length)
			       for pos = (search part string
						 :start2 old-pos
						 :test test)
			       do (write-string string out
						:start old-pos
						:end (or pos (length string)))
			       when pos do (write-string replacement out)
			       while pos)))

(defmacro create-probe(name &body code)
  `(progn
     (defparameter ,name ',name)
     (defun ,name(params) ,@code)))

(defun get-file-size(path)
  (with-open-file (stream path)
    (and stream (file-length path))))

(defun command-return-code(command)
  (let ((code (nth-value 2 (uiop:run-program command :ignore-error-status t))))
    (if (= 0 code)
	t
	(list nil (format nil "return code = ~a" code)))))

(defmacro alert(name string)
  `(progn
     (defparameter ,name ',name)
     (push (list ',name ,string)
           *alerts*)))

(defmacro strcat(&body body)
  `(progn
     (concatenate 'string ,@body)))

(defun trigger-alert(level function params result state)
  (let* ((notifier-command (assoc level *alerts*))
         (command-string (cadr notifier-command)))
    (setf command-string (replace-all command-string "%state%"    (if (eql 'error state) "Start" "End")))
    (setf command-string (replace-all command-string "%result%"   (format nil "~a" result)))
    (setf command-string (replace-all command-string "%hostname%" (machine-instance)))
    (setf command-string (replace-all command-string "%os%"       (software-type)))
    (setf command-string (replace-all command-string "%function%" (format nil "~a" function)))
    (setf command-string (replace-all command-string "%params%"   (format nil "~a" params)))
    (setf command-string (replace-all command-string "%desc%"     (getf params :desc "")))
    (setf command-string (replace-all command-string "%newline%"  (string #\Newline)))
    (setf command-string (replace-all command-string "%level%"    level))
    (setf command-string (replace-all command-string "%date%"
                                      (multiple-value-bind
                                            (second minute hour day month year)
                                          (get-decoded-time)
                                        (format nil "~a/~a/~a ~a:~a:~a" year month day hour minute second))))
    command-string))
    
(defmacro stop-if-error(&body body)
  `(progn
     (and ,@body)))

(defmacro escalation(&body body)
  `(progn
     (or ,@body)))

(defun =>(level fonction &rest params)
  (format t "[~a~10a ~20A~a] ~45A" *yellow* level fonction *white* (getf params :desc params))
  (let* ((hash (fnv-hash (format nil "~{~a~}" (remove-if #'symbolp params))))
         (result (funcall fonction params))
         (filename (format nil "~a-~a-~a" level fonction hash))
         (filepath (format nil "~a/~a" *states-dir* filename)))

    ;; we open the file to read the number of tries
    ;; if no fail then we have 0 try
    (let* ((tries (if (not (probe-file filepath))
                      0
                      (with-open-file (stream filepath :direction :input)
                        (parse-integer (read-line stream 0 nil)))))
           (triggered-before? (>= tries (getf params :try *tries*))))

      ;; if result is a list then the check had fail a return both nil and the error value
      ;; if result is not a list, then it was successful
      (if (not (listp result))
          (progn

            ;; we delete the file with previous states
            (when (probe-file filepath)
              (delete-file filepath))

            ;; it was a failure and then it's back to normal state
            (if triggered-before?
                (progn
                  (uiop:run-program (trigger-alert level fonction params t 'success) :output t)
                  (format t " => ~afailure => success~a~%" *green* *white*))
                (progn
                  ;; last time was a success
                  (format t " => ~asuccess~a~%" *green* *white*)))
            ;; in any case we return t because it's ok
            t)

          ;; failure handling
          (let ((trigger-now? (= (+ 1 tries) (getf params :try *tries*)))) ; we add +1 because it's failing right now
            (format t " => ~aerror (~a failure(s) before)~a~a~%" *red* tries *white* (if trigger-now? " NOTIFIED" ""))

            ;; more error than limit, send alert once
            (when trigger-now?
              (uiop:run-program (trigger-alert level fonction params (cadr result) 'error) :output t))
            ;; increment the number of tries by 1
            (with-open-file (stream-out filepath :direction :output
                                        :if-exists :supersede)
              (format stream-out "~a~%~a~%" (+ 1 tries) params))
            nil)))))

;; abort when using ctrl+c instead of dropping to debugger
#+ecl
(ext:set-signal-handler ext:+sigint+ #'quit)
