(require 'asdf)

(defparameter *alerts* '())

(defun color(num1 num2)
  (format nil "~a[~a;~am" #\Escape num1 num2))

(defparameter *red*    (color 1 31))
(defparameter *white*  (color 0 70))
(defparameter *green*  (color 1 32))
(defparameter *yellow* (color 0 33))

;; common-lisp don't have a split string function natively
(defun replace-all (string part replacement &key (test #'char=))
  (with-output-to-string (out)
			 (loop with part-length = (length part)
			       for old-pos = 0 then (+ pos part-length)
			       for pos = (search part string
						 :start2 old-pos
						 :test test)
			       do (write-string string out
						:start old-pos
						:end (or pos (length string)))
			       when pos do (write-string replacement out)
			       while pos)))

(defmacro create-probe(name &body code)
  `(progn
     (defparameter ,name ',name)
     (defun ,name(params) ,@code)))

(defun get-file-size(path)
  (with-open-file (stream path)
    (and stream (file-length path))))

(defun command-return-code(command)
  (let ((code (nth-value 2 (uiop:run-program command :ignore-error-status t))))
    (if (= 0 code)
	t
	(list nil (format nil "return code = ~a" code)))))

(defmacro alert(name string)
  `(progn
     (defparameter ,name ',name)
     (push (list ',name ,string)
                *alerts*)))

(defun trigger-alert(level function params result)
  (let* ((notifier-command (assoc level *alerts*))
         (command-string (cadr notifier-command)))
    (setf command-string (replace-all command-string "%result%"   (format nil "~a" result)))
    (setf command-string (replace-all command-string "%hostname%" (machine-instance)))
    (setf command-string (replace-all command-string "%os%"       (software-type)))
    (setf command-string (replace-all command-string "%function%" (format nil "~a" function)))
    (setf command-string (replace-all command-string "%params%"   (format nil "~a" params)))
    (setf command-string (replace-all command-string "%desc%"     (getf params :desc "")))
    (setf command-string (replace-all command-string "%newline%"  (string #\Newline)))
    (setf command-string (replace-all command-string "%level%"    level))
    (setf command-string (replace-all command-string "%date%"
                                      (multiple-value-bind
                                            (second minute hour day month year)
                                          (get-decoded-time)
                                        (format nil "~a/~a/~a ~a:~a:~a" year month day hour minute second))))
    command-string))
    
(defmacro stop-if-error(&body body)
  `(progn
     (and ,@body)))

(defmacro escalation(&body body)
  `(progn
     (or ,@body)))

(defun =>(level fonction &rest params)
  (format t "[~a~a ~20A~a] ~35A" *yellow* level fonction *white* (getf params :desc params))
  (let ((result (funcall fonction params)))
    (if (not (listp result))
        (progn
          (format t " => ~asuccess~a~%" *green* *white*)
          t)
        (progn
          (format t " => ~aerror~a~%" *red* *white*)
          (uiop:run-program (trigger-alert level fonction params (cadr result)) :output t)
          nil))))

(load "probes.lisp")
