/* Routines to read and write repository parameters */
#include <assert.h>
#include <errno.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>

#include <sodium.h>

#include "config.h"
#include "misc.h"
#include "state.h"

#define VMIN            0
#define VMAJ            1
#define VMINMASK        0xff
#define VMAJSHIFT       8
#define VMAJMASK        0xff

#define CALGOSHIFT      16
#define CALGOMASK       0x7
#define CNONETYPE       0
#define CSNAPPYTYPE     1
#define CLZ4TYPE        2

#define EALGOSHIFT      19
#define EALGOMASK       0x7
#define ENONETYPE       0
#define ECHACHATYPE     1

#define NONCESIZE	crypto_aead_xchacha20poly1305_ietf_NPUBBYTES
#define MSEEDSIZE	4
#define CSEEDSIZE	(MSEEDSIZE + crypto_aead_xchacha20poly1305_ietf_ABYTES)
#define SHDRSIZE	(8 + NONCESIZE + CSEEDSIZE)

/* misc helpers */
extern int pack(unsigned char *, char *, ...);
extern int unpack(unsigned char *, char *, ...);

struct shdr {
	uint64_t flags;
	unsigned char nonce[NONCESIZE];
	unsigned char seed[CSEEDSIZE];
};

/* Unpack state header */
static int
unpackshdr(unsigned char *buf, struct shdr *shdr)
{
	char fmt[BUFSIZ];
	int n;

	snprintf(fmt, sizeof(fmt), "q'%d'%d", NONCESIZE, CSEEDSIZE);
	n = unpack(buf, fmt,
	           &shdr->flags,
	           shdr->nonce,
	           shdr->seed);
	assert(n == SHDRSIZE);
	return n;
}

/* Pack state header */
static int
packshdr(unsigned char *buf, struct shdr *shdr)
{
	char fmt[BUFSIZ];
	int n;

	snprintf(fmt, sizeof(fmt), "q'%d'%d", NONCESIZE, CSEEDSIZE);
	n = pack(buf, fmt,
	         shdr->flags,
	         shdr->nonce,
	         shdr->seed);
	assert(n == SHDRSIZE);
	return n;
}

int
writestate(int fd, struct param *par)
{
	unsigned char buf[SHDRSIZE];
	struct shdr shdr;

	if (sodium_init() < 0) {
		seterr("sodium_init: failed");
		return -1;
	}

	/* Set version */
	shdr.flags = (VMAJ << VMAJSHIFT) | VMIN;

	/* Set compression type */
	if (strcasecmp(par->calgo, "none") == 0) {
		shdr.flags |= CNONETYPE << CALGOSHIFT;
	} else if (strcasecmp(par->calgo, "snappy") == 0) {
		shdr.flags |= CSNAPPYTYPE << CALGOSHIFT;
	} else if (strcasecmp(par->calgo, "lz4") == 0) {
		shdr.flags |= CLZ4TYPE << CALGOSHIFT;
	} else {
		seterr("invalid compression type: %s", par->calgo);
		return -1;
	}

	/* Clear seed + authentication tag */
	memset(shdr.seed, 0, sizeof(shdr.seed));

	/* Pack seed */
	shdr.seed[0] = par->seed;
	shdr.seed[1] = par->seed >> 8;
	shdr.seed[2] = par->seed >> 16;
	shdr.seed[3] = par->seed >> 24;

	/* Set encryption type */
	if (strcasecmp(par->ealgo, "none") == 0) {
		shdr.flags |= ENONETYPE << EALGOSHIFT;
		memset(shdr.nonce, 0, sizeof(shdr.nonce));
	} else if (strcasecmp(par->ealgo, "XChaCha20-Poly1305") == 0) {
		unsigned long long elen;

		shdr.flags |= ECHACHATYPE << EALGOSHIFT;
		randombytes_buf(shdr.nonce, sizeof(shdr.nonce));
		crypto_aead_xchacha20poly1305_ietf_encrypt(shdr.seed, &elen,
		                                           shdr.seed, MSEEDSIZE,
		                                           NULL, 0, NULL,
		                                           shdr.nonce, par->key);
		assert(elen == CSEEDSIZE);
	} else {
		seterr("invalid encryption type: %s", par->ealgo);
		return -1;
	}

	packshdr(buf, &shdr);
	if (xwrite(fd, buf, SHDRSIZE) != SHDRSIZE) {
		seterr("failed to write state header: %s", strerror(errno));
		return -1;
	}
	return 0;
}

int
readstate(int fd, struct param *par)
{
	unsigned char buf[SHDRSIZE];
	struct shdr shdr;
	unsigned long long dlen;
	int algo;

	if (sodium_init() < 0) {
		seterr("sodium_init: failed");
		return -1;
	}

	if (xread(fd, buf, SHDRSIZE) != SHDRSIZE) {
		seterr("failed to read state header: %s", strerror(errno));
		return -1;
	}
	unpackshdr(buf, &shdr);

	/* If the major version is different, the format is incompatible */
	if (((shdr.flags >> VMAJSHIFT) & VMAJMASK) != VMAJ) {
		seterr("state header version mismatch");
		return -1;
	}

	/* Populate param compression algo */
	algo = (shdr.flags >> CALGOSHIFT) & CALGOMASK;
	switch (algo) {
	case CNONETYPE:
		par->calgo = "none";
		break;
	case CSNAPPYTYPE:
		par->calgo = "snappy";
		break;
	case CLZ4TYPE:
		par->calgo = "lz4";
		break;
	default:
		seterr("invalid compression type: %d", algo);
		return -1;
	}

	/* Populate param encryption algo */
	algo = (shdr.flags >> EALGOSHIFT) & EALGOMASK;
	switch (algo) {
	case ENONETYPE:
		par->ealgo = "none";
		break;
	case ECHACHATYPE:
		par->ealgo = "XChaCha20-Poly1305";
		if (crypto_aead_xchacha20poly1305_ietf_decrypt(shdr.seed, &dlen,
		                                               NULL,
		                                               shdr.seed, CSEEDSIZE,
		                                               NULL, 0,
		                                               shdr.nonce, par->key) < 0) {
			seterr("authentication failed");
			return -1;
		}
		assert(dlen == MSEEDSIZE);
		break;
	default:
		seterr("invalid encryption type: %d", algo);
		return -1;
	}

	/* Unpack seed */
	par->seed = (uint32_t)shdr.seed[0];
	par->seed |= (uint32_t)shdr.seed[1] << 8;
	par->seed |= (uint32_t)shdr.seed[2] << 16;
	par->seed |= (uint32_t)shdr.seed[3] << 24;

	return 0;
}
