#include <sys/types.h>
#include <sys/stat.h>

#include <err.h>
#include <fcntl.h>
#include <limits.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "arg.h"
#include "block.h"
#include "chunker.h"
#include "config.h"
#include "key.h"
#include "lock.h"
#include "misc.h"
#include "snap.h"
#include "state.h"

struct param param;
int verbose;
char *argv0;

static void
loadstate(char *repo)
{
	char path[PATH_MAX];
	int fd;

	if (snprintf(path, sizeof(path), "%s/state", repo) >= sizeof(path))
		errx(1, "snprintf: %s: path too long", path);
	fd = open(path, O_RDONLY);
	if (fd < 0)
		err(1, "open: %s", path);
	if (readstate(fd, &param) < 0)
		printerr("readstate: %s", path);
	if (close(fd) < 0)
		err(1, "close: %s", path);
}

static void
loadkey(char *keyfile)
{
	int fd;

	if (keyfile == NULL)
		return;

	fd = open(keyfile, O_RDONLY);
	if (fd < 0)
		err(1, "open: %s", keyfile);
	if (readkey(fd, param.key, sizeof(param.key)) < 0)
		printerr("readkey: %s", keyfile);
	param.keyloaded = 1;
	if (close(fd) < 0)
		err(1, "close: %s", keyfile);
}

static void
pack(struct sctx *sctx, struct bctx *bctx)
{
	struct chunker *c;

	if ((c = copen(0, BSIZEMIN, BSIZEMAX, HMASKBITS, WINSIZE,
	               param.seed)) == NULL)
		printerr("copen");

	while (cfill(c) > 0) {
		unsigned char md[MDSIZE];
		void *buf;
		size_t n;

		buf = cget(c, &n);
		if (buf == NULL)
			printerr("cget");
		if (bput(bctx, buf, n, md) < 0)
			printerr("bput");
		if (sput(sctx, md) < 0)
			printerr("sput");
		cdrain(c);
	}
	cclose(c);
}

static void
usage(void)
{
	fprintf(stderr, "usage: %s [-v] [-k keyfile] [-r repo] name\n", argv0);
	exit(1);
}

int
main(int argc, char *argv[])
{
	char spath[PATH_MAX];
	char bpath[PATH_MAX];
	struct sctx *sctx;
	struct bctx *bctx;
	char *keyfile = NULL;
	char *repo = ".";
	int lfd;

	ARGBEGIN {
	case 'k':
		keyfile = EARGF(usage());
		break;
	case 'r':
		repo = EARGF(usage());
		break;		
	case 'v':
		verbose++;
		break;
	default:
		usage();
	} ARGEND

	if (argc != 1)
		usage();

	if (snprintf(spath, sizeof(spath), "%s/archive/%s",
	             repo, argv[0]) >= sizeof(spath))
		errx(1, "snprintf: %s: path too long", spath);
	if (snprintf(bpath, sizeof(bpath), "%s/storage",
	             repo) >= sizeof(bpath))
		errx(1, "snprintf: %s: path too long", bpath);

	if ((lfd = lockrepo(repo)) < 0)
		errx(1, "failed to lock repository");

	loadkey(keyfile);
	loadstate(repo);

	if (screat(spath, 0600, &sctx) < 0)
		printerr("screat: %s", spath);
	if (bopen(bpath, B_RDWR, 0600, &bctx) < 0)
		printerr("bopen: %s", bpath);

	pack(sctx, bctx);

	if (bclose(bctx) < 0)
		printerr("bclose: %s", bpath);
	if (sclose(sctx) < 0)
		printerr("sclose: %s", spath);

	if (unlockrepo(lfd) < 0)
		errx(1, "failed to unlock repository");
	return 0;
}
