/* Encryption layer implementation */
#include <sys/types.h>
#include <sys/stat.h>

#include <assert.h>
#include <fcntl.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>

#include <sodium.h>

#include "block.h"
#include "config.h"
#include "misc.h"
#include "state.h"

extern struct param param;

#define EDNONETYPE	0x300
#define EDCHACHATYPE	0x301
#define NONCESIZE	crypto_aead_xchacha20poly1305_ietf_NPUBBYTES
#define EDSIZE		(8 + 8 + NONCESIZE)

extern int pack(unsigned char *, char *, ...);
extern int unpack(unsigned char *, char *, ...);

static int becreat(struct bctx *bctx, char *path, int mode);
static int beopen(struct bctx *bctx, char *path, int flags, int mode);
static int beput(struct bctx *bctx, void *buf, size_t n, unsigned char *md);
static int beget(struct bctx *bctx, unsigned char *md, void *buf, size_t *n);
static int berm(struct bctx *bctx, unsigned char *md);
static int begc(struct bctx *bctx);
static int besync(struct bctx *bctx);
static int beclose(struct bctx *bctx);

static struct bops bops = {
	.creat = becreat,
	.open = beopen,
	.put = beput,
	.get = beget,
	.rm = berm,
	.gc = begc,
	.sync = besync,
	.close = beclose,
};

/* Encryption layer context */
struct ectx {
	int type;		/* encryption algorithm type for new blocks */
};

/* Encryption descriptor */
struct ed {
	uint16_t type;			/* encryption algorithm type */
	unsigned char reserved[6];	/* should be set to 0 when writing */
	uint64_t size;			/* size of encrypted block */
	unsigned char nonce[NONCESIZE];	/* unpredictable nonce used when encrypting */
};

/* Unpack encryption descriptor */
static int
unpacked(unsigned char *buf, struct ed *ed)
{
	char fmt[BUFSIZ];
	int n;

	snprintf(fmt, sizeof(fmt), "s'6q'%d", NONCESIZE);
	n = unpack(buf, fmt,
	           &ed->type,
	           ed->reserved,
	           &ed->size,
	           ed->nonce);

	assert(n == EDSIZE);
	return n;
}

/* Pack encryption descriptor */
static int
packed(unsigned char *buf, struct ed *ed)
{
	char fmt[BUFSIZ];
	int n;

	snprintf(fmt, sizeof(fmt), "s'6q'%d", NONCESIZE);
	n = pack(buf, fmt,
	         ed->type,
	         ed->reserved,
	         ed->size,
	         ed->nonce);

	assert(n == EDSIZE);
	return n;
}

static int
becreat(struct bctx *bctx, char *path, int mode)
{
	struct ectx *ectx;
	int type;

	/* Determine algorithm type */
	if (strcasecmp(param.ealgo, "none") == 0) {
		type = EDNONETYPE;
	} else if (strcasecmp(param.ealgo, "XChaCha20-Poly1305") == 0) {
		type = EDCHACHATYPE;
	} else {
		seterr("invalid encryption type: %s", param.ealgo);
		return -1;
	}

	/* Ensure a key has been provided if caller requested encryption */
	if (type != EDNONETYPE && !param.keyloaded) {
		seterr("expected encryption key");
		return -1;
	}

	if (sodium_init() < 0) {
		seterr("sodium_init: failed");
		return -1;
	}

	bctx->ectx = calloc(1, sizeof(struct ectx));
	if (bctx->ectx == NULL) {
		seterr("calloc: out of memory");
		return -1;
	}
	ectx = bctx->ectx;
	ectx->type = type;

	if (bstorageops()->creat(bctx, path, mode) < 0) {
		free(ectx);
		return -1;
	}
	return 0;
}

static int
beopen(struct bctx *bctx, char *path, int flags, int mode)
{
	struct ectx *ectx;
	int type;

	/* Determine algorithm type */
	if (strcasecmp(param.ealgo, "none") == 0) {
		type = EDNONETYPE;
	} else if (strcasecmp(param.ealgo, "XChaCha20-Poly1305") == 0) {
		type = EDCHACHATYPE;
	} else {
		seterr("invalid encryption type: %s", param.ealgo);
		return -1;
	}

	/* Ensure a key has been provided if caller requested encryption */
	if (type != EDNONETYPE && !param.keyloaded) {
		seterr("expected encryption key");
		return -1;
	}

	if (sodium_init() < 0) {
		seterr("sodium_init: failed");
		return -1;
	}

	bctx->ectx = calloc(1, sizeof(struct ectx));
	if (bctx->ectx == NULL) {
		seterr("calloc: out of memory");
		return -1;
	}
	ectx = bctx->ectx;
	ectx->type = type;

	if (bstorageops()->open(bctx, path, flags, mode) < 0) {
		free(ectx);
		return -1;
	}
	return 0;
}

static int
beput(struct bctx *bctx, void *buf, size_t n, unsigned char *md)
{
	struct ectx *ectx;
	struct ed ed;
	unsigned char *ebuf;
	unsigned long long elen;
	size_t en;

	/* Calculate size of encrypted block */
	ectx = bctx->ectx;
	switch (ectx->type) {
	case EDNONETYPE:
		en = n;
		break;
	case EDCHACHATYPE:
		en = n + crypto_aead_xchacha20poly1305_ietf_ABYTES;
		break;
	}

	ebuf = malloc(EDSIZE + en);
	if (ebuf == NULL) {
		seterr("malloc: out of memory");
		return -1;
	}

	/* Prepare encryption descriptor */
	ed.type = ectx->type;
	memset(ed.reserved, 0, sizeof(ed.reserved));
	ed.size = en;

	/* Fill nonce buffer */
	switch (ectx->type) {
	case EDNONETYPE:
		memset(ed.nonce, 0, sizeof(ed.nonce));
		break;
	case EDCHACHATYPE:
		randombytes_buf(ed.nonce, sizeof(ed.nonce));
		break;
	}

	/* Prepend encryption descriptor */
	packed(ebuf, &ed);

	/* Encrypt block */
	switch (ectx->type) {
	case EDNONETYPE:
		memcpy(&ebuf[EDSIZE], buf, en);
		break;
	case EDCHACHATYPE:
		crypto_aead_xchacha20poly1305_ietf_encrypt(&ebuf[EDSIZE], &elen,
		                                           buf, n, ebuf, EDSIZE, NULL,
		                                           ed.nonce, param.key);
		assert(elen == en);
		break;
	}

	if (bstorageops()->put(bctx, ebuf, EDSIZE + en, md) < 0) {
		free(ebuf);
		return -1;
	}

	free(ebuf);
	return ed.size;
}

static int
beget(struct bctx *bctx, unsigned char *md, void *buf, size_t *n)
{
	struct ed ed;
	struct ectx *ectx;
	unsigned char *ebuf;
	unsigned long long dlen;
	size_t dn, size;

	/* Calculate maximum size of encrypted block */
	size = EDSIZE + *n + crypto_aead_xchacha20poly1305_ietf_ABYTES;

	ebuf = malloc(size);
	if (ebuf == NULL) {
		seterr("malloc: out of memory");
		return -1;
	}

	if (bstorageops()->get(bctx, md, ebuf, &size) < 0) {
		free(ebuf);
		return -1;
	}

	unpacked(ebuf, &ed);

	/* Decrypt block */
	ectx = bctx->ectx;
	switch (ed.type) {
	case EDNONETYPE:
		dn = ed.size;
		if (*n < dn) {
			free(ebuf);
			seterr("buffer too small");
			return -1;
		}
		memcpy(buf, &ebuf[EDSIZE], dn);
		break;
	case EDCHACHATYPE:
		dn = ed.size - crypto_aead_xchacha20poly1305_ietf_ABYTES;
		if (*n < dn) {
			free(ebuf);
			seterr("buffer too small");
			return -1;
		}

		if (crypto_aead_xchacha20poly1305_ietf_decrypt(buf, &dlen,
		                                               NULL,
		                                               &ebuf[EDSIZE], ed.size,
		                                               ebuf, EDSIZE,
		                                               ed.nonce, param.key) < 0) {
			free(ebuf);
			seterr("authentication failed");
			return -1;
		}

		assert(dn == dlen);
		break;
	}

	free(ebuf);
	*n = dn;
	return 0;
}

static int
berm(struct bctx *bctx, unsigned char *md)
{
	return bstorageops()->rm(bctx, md);
}

static int
begc(struct bctx *bctx)
{
	return bstorageops()->gc(bctx);
}

static int
besync(struct bctx *bctx)
{
	return bstorageops()->sync(bctx);
}

static int
beclose(struct bctx *bctx)
{
	struct ectx *ectx = bctx->ectx;

	free(ectx);
	return bstorageops()->close(bctx);
}

struct bops *
bencryptops(void)
{
	return &bops;
}
