#include <sys/types.h>
#include <sys/stat.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>

#include <sodium.h>

#include "arg.h"
#include "block.h"
#include "config.h"
#include "key.h"
#include "lock.h"
#include "misc.h"
#include "snap.h"
#include "state.h"

struct param param;
int verbose;
char *argv0;

static void
savestate(char *repo)
{
	char path[PATH_MAX];
	int fd;

	if (snprintf(path, sizeof(path), "%s/state", repo) >= sizeof(path))
		errx(1, "snprintf: %s: path too long", path);
	fd = open(path, O_RDWR | O_CREAT | O_EXCL, 0600);
	if (fd < 0)
		err(1, "open: %s", path);
	if (writestate(fd, &param) < 0)
		printerr("writestate: %s", path);
	if (close(fd) < 0)
		err(1, "close: %s", path);
}

static void
loadkey(char *keyf)
{
	int fd;

	if (keyf == NULL)
		return;

	fd = open(keyf, O_RDONLY);
	if (fd < 0)
		err(1, "open: %s", keyf);
	if (readkey(fd, param.key, sizeof(param.key)) < 0)
		printerr("readkey: %s", keyf);
	param.keyloaded = 1;
	if (close(fd) < 0)
		err(1, "close: %s", keyf);
}

static void
usage(void)
{
	fprintf(stderr, "usage: %s [-v] [-E algo] [-Z algo] [-k keyfile] [repo]\n", argv0);
	exit(1);
}

int
main(int argc, char *argv[])
{
	char spath[PATH_MAX];
	char bpath[PATH_MAX];
	struct bctx *bctx;
	char *keyf = NULL;
	char *repo;
	int lfd;

	param.calgo = "snappy";
	param.ealgo = "none";

	ARGBEGIN {
	case 'k':
		keyf = EARGF(usage());
		break;
	case 'E':
		param.ealgo = EARGF(usage());
		break;
	case 'Z':
		param.calgo = EARGF(usage());
		break;
	case 'v':
		verbose++;
		break;
	default:
		usage();
	} ARGEND

	switch (argc) {
	case 0:
		repo = ".";
		break;
	case 1:
		repo = argv[0];
		break;
	default:
		usage();
	};

	if (sodium_init() < 0)
		errx(1, "sodium_init: failed");

	if (strcasecmp(param.ealgo, "none") == 0) {
		param.seed = 0;
	} else if (strcasecmp(param.ealgo, "XChaCha20-Poly1305") == 0) {
		if (keyf == NULL)
			errx(1, "expected encryption key");
		param.seed = randombytes_uniform(0xffffffff);
	}

	if (snprintf(spath, sizeof(spath), "%s/%s",
	             repo, ARCHIVEPATH) >= sizeof(spath))
		errx(1, "snprintf: %s: path too long", spath);
	if (snprintf(bpath, sizeof(bpath), "%s/%s",
	             repo, STORAGEPATH) >= sizeof(bpath))
		errx(1, "snprintf: %s: path too long", bpath);

	if (mkdir(repo, 0700) < 0 && errno != EEXIST)
		err(1, "mkdir: %s", repo);

	if ((lfd = lockrepo(repo)) < 0)
		errx(1, "failed to lock repository");

	if (mkdir(spath, 0700) < 0)
		err(1, "mkdir: %s", spath);

	loadkey(keyf);
	savestate(repo);

	if (bcreat(bpath, 0600, &bctx) < 0)
		printerr("bcreat: %s", bpath);
	if (bclose(bctx) < 0)
		printerr("bclose: %s", bpath);

	if (unlockrepo(lfd) < 0)
		errx(1, "failed to unlock repository");
	return 0;
}
