#include <sys/types.h>
#include <sys/stat.h>

#include <err.h>
#include <fcntl.h>
#include <limits.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "arg.h"
#include "block.h"
#include "config.h"
#include "key.h"
#include "lock.h"
#include "misc.h"
#include "snap.h"
#include "state.h"

struct param param;
int verbose;
char *argv0;

static void
loadstate(char *repo)
{
	char path[PATH_MAX];
	int fd;

	if (snprintf(path, sizeof(path), "%s/state", repo) >= sizeof(path))
		errx(1, "snprintf: %s: path too long", path);
	fd = open(path, O_RDONLY);
	if (fd < 0)
		err(1, "open: %s", path);
	if (readstate(fd, &param) < 0)
		printerr("readstate: %s", path);
	if (close(fd) < 0)
		err(1, "close: %s", path);
}

static void
loadkey(char *keyfile)
{
	int fd;

	if (keyfile == NULL)
		return;

	fd = open(keyfile, O_RDONLY);
	if (fd < 0)
		err(1, "open: %s", keyfile);
	if (readkey(fd, param.key, sizeof(param.key)) < 0)
		printerr("readkey: %s", keyfile);
	param.keyloaded = 1;
	if (close(fd) < 0)
		err(1, "close: %s", keyfile);
}

static void
usage(void)
{
	fprintf(stderr, "usage: %s [-v] [-k keyfile] [repo]\n", argv0);
	exit(1);
}

int
main(int argc, char *argv[])
{
	char path[PATH_MAX];
	struct bctx *bctx;
	char *keyfile = NULL;
	char *repo;
	int lfd;

	ARGBEGIN {
	case 'k':
		keyfile = EARGF(usage());
		break;
	case 'v':
		verbose++;
		break;
	default:
		usage();
	} ARGEND

	switch (argc) {
	case 0:
		repo = ".";
		break;
	case 1:
		repo = argv[0];
		break;
	default:
		usage();
	};

	if (snprintf(path, sizeof(path), "%s/%s",
	             repo, STORAGEPATH) >= sizeof(path))
		errx(1, "snprintf: %s: path too long", path);

	if ((lfd = lockrepo(repo)) < 0)
		errx(1, "failed to lock repository");

	loadkey(keyfile);
	loadstate(repo);

	if (bopen(path, B_RDWR, 0600, &bctx) < 0)
		printerr("bopen: %s", path);
	if (bgc(bctx) < 0)
		printerr("bgc: %s", path);
	if (bclose(bctx) < 0)
		printerr("bclose: %s", path);

	if (unlockrepo(lfd) < 0)
		errx(1, "failed to unlock repository");
	return 0;
}
