/* Compression layer implementation */
#include <sys/types.h>
#include <sys/stat.h>

#include <assert.h>
#include <fcntl.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>

#include <lz4.h>
#include <snappy-c.h>

#include "block.h"
#include "config.h"
#include "misc.h"
#include "state.h"

#define CDNONETYPE	0x200
#define CDSNAPPYTYPE	0x201
#define CDLZ4TYPE	0x202
#define CDSIZE		(8 + 8)

extern struct param param;

extern int pack(unsigned char *, char *, ...);
extern int unpack(unsigned char *, char *, ...);

static int bccreat(struct bctx *bctx, char *path, int mode);
static int bcopen(struct bctx *bctx, char *path, int flags, int mode);
static int bcput(struct bctx *bctx, void *buf, size_t n, unsigned char *md);
static int bcget(struct bctx *bctx, unsigned char *md, void *buf, size_t *n);
static int bcrm(struct bctx *bctx, unsigned char *md);
static int bcgc(struct bctx *bctx);
static int bcsync(struct bctx *bctx);
static int bcclose(struct bctx *bctx);

static struct bops bops = {
	.creat = bccreat,
	.open = bcopen,
	.put = bcput,
	.get = bcget,
	.rm = bcrm,
	.gc = bcgc,
	.sync = bcsync,
	.close = bcclose,
};

/* Compression layer context */
struct cctx {
	int type;	/* compression algorithm type for new blocks */
};

/* Compression descriptor */
struct cd {
	uint16_t type;			/* compression algorithm type */
	unsigned char reserved[6];	/* should be set to 0 when writing */
	uint64_t size;			/* size of compressed block */
};

/* Unpack compression descriptor */
static int
unpackcd(void *buf, struct cd *cd)
{
	int n;

	n = unpack(buf, "s'6q",
	           &cd->type,
	           cd->reserved,
	           &cd->size);

	assert(n == CDSIZE);
	return n;
}

/* Pack compression descriptor */
static int
packcd(void *buf, struct cd *cd)
{
	int n;

	n = pack(buf, "s'6q",
	         cd->type,
	         cd->reserved,
	         cd->size);

	assert(n == CDSIZE);
	return n;
}

static int
bccreat(struct bctx *bctx, char *path, int mode)
{
	struct cctx *cctx;
	int type;

	if (strcasecmp(param.calgo, "none") == 0) {
		type = CDNONETYPE;
	} else if (strcasecmp(param.calgo, "snappy") == 0) {
		type = CDSNAPPYTYPE;
	} else if (strcasecmp(param.calgo, "lz4") == 0) {
		type = CDLZ4TYPE;
	} else {
		seterr("invalid compression type: %s", param.calgo);
		return -1;
	}

	bctx->cctx = calloc(1, sizeof(struct cctx));
	if (bctx->cctx == NULL) {
		seterr("calloc: out of memory");
		return -1;
	}
	cctx = bctx->cctx;
	cctx->type = type;

	if (bencryptops()->creat(bctx, path, mode) < 0) {
		free(cctx);
		return -1;
	}
	return 0;
}

static int
bcopen(struct bctx *bctx, char *path, int flags, int mode)
{
	struct cctx *cctx;
	int type;

	if (strcasecmp(param.calgo, "none") == 0) {
		type = CDNONETYPE;
	} else if (strcasecmp(param.calgo, "snappy") == 0) {
		type = CDSNAPPYTYPE;
	} else if (strcasecmp(param.calgo, "lz4") == 0) {
		type = CDLZ4TYPE;
	} else {
		seterr("invalid compression type: %s", param.calgo);
		return -1;
	}

	bctx->cctx = calloc(1, sizeof(struct cctx));
	if (bctx->cctx == NULL) {
		seterr("calloc: out of memory");
		return -1;
	}
	cctx = bctx->cctx;
	cctx->type = type;

	if (bencryptops()->open(bctx, path, flags, mode) < 0) {
		free(cctx);
		return -1;
	}
	return 0;
}

static int
bcput(struct bctx *bctx, void *buf, size_t n, unsigned char *md)
{
	struct cctx *cctx;
	struct cd cd;
	char *cbuf;
	size_t cn;
	int r;

	/* Calculate compressed block size */
	cctx = bctx->cctx;
	switch (cctx->type) {
	case CDNONETYPE:
		cn = n;
		break;
	case CDSNAPPYTYPE:
		cn = snappy_max_compressed_length(n);
		break;
	case CDLZ4TYPE:
		cn = LZ4_compressBound(n);
		break;
	}

	cbuf = malloc(CDSIZE + cn);
	if (cbuf == NULL) {
		seterr("malloc: out of memory");
		return -1;
	}

	/* Compress block */
	switch (cctx->type) {
	case CDNONETYPE:
		memcpy(&cbuf[CDSIZE], buf, cn);
		break;
	case CDSNAPPYTYPE:
		if (snappy_compress(buf, n, &cbuf[CDSIZE], &cn) != SNAPPY_OK) {
			free(cbuf);
			seterr("snappy_compress: failed");
			return -1;
		}
		break;
	case CDLZ4TYPE:
		r = LZ4_compress_default(buf, &cbuf[CDSIZE], n, cn);
		if (r < 0) {
			free(cbuf);
			seterr("LZ4_compress_default: failed");
			return -1;
		}
		cn = r;
		break;
	}

	/* Prepare compression descriptor */
	cd.type = cctx->type;
	memset(cd.reserved, 0, sizeof(cd.reserved));
	cd.size = cn;
	/* Prepend compression descriptor */
	packcd(cbuf, &cd);

	if (bencryptops()->put(bctx, cbuf, CDSIZE + cn, md) < 0) {
		free(cbuf);
		return -1;
	}

	free(cbuf);
	return cd.size;
}

static int
bcget(struct bctx *bctx, unsigned char *md, void *buf, size_t *n)
{
	struct cd cd;
	char *cbuf;
	size_t cn, un, size;
	int r;

	/* Calculate maximum compressed block size */
	size = *n;
	cn = snappy_max_compressed_length(*n);
	if (cn > size)
		size = cn;
	cn = LZ4_compressBound(*n);
	if (cn > size)
		size = cn;
	size += CDSIZE;

	cbuf = malloc(size);
	if (cbuf == NULL) {
		seterr("malloc: out of memory");
		return -1;
	}

	if (bencryptops()->get(bctx, md, cbuf, &size) < 0) {
		free(cbuf);
		return -1;
	}

	unpackcd(cbuf, &cd);

	/* Decompress block */
	switch (cd.type) {
	case CDNONETYPE:
		un = cd.size;
		if (*n < un) {
			free(cbuf);
			seterr("buffer too small");
			return -1;
		}
		memcpy(buf, &cbuf[CDSIZE], un);
		break;
	case CDSNAPPYTYPE:
		if (snappy_uncompressed_length(&cbuf[CDSIZE], cd.size,
		                               &un) != SNAPPY_OK) {
			free(cbuf);
			seterr("snappy_uncompressed_length: failed");
			return -1;
		}

		if (*n < un) {
			free(cbuf);
			seterr("buffer too small");
			return -1;
		}

		if (snappy_uncompress(&cbuf[CDSIZE], cd.size, buf,
		                      &un) != SNAPPY_OK) {
			free(cbuf);
			seterr("snappy_uncompress: failed");
			return -1;
		}
		break;
	case CDLZ4TYPE:
		r = LZ4_decompress_safe(&cbuf[CDSIZE], buf, cd.size, *n);
		if (r < 0) {
			free(cbuf);
			seterr("LZ4_decompress_safe: failed");
			return -1;
		}
		un = r;
		break;
	}

	free(cbuf);
	*n = un;
	return 0;
}

static int
bcrm(struct bctx *bctx, unsigned char *md)
{
	return bencryptops()->rm(bctx, md);
}

static int
bcgc(struct bctx *bctx)
{
	return bencryptops()->gc(bctx);
}

static int
bcsync(struct bctx *bctx)
{
	return bencryptops()->sync(bctx);
}

static int
bcclose(struct bctx *bctx)
{
	struct cctx *cctx = bctx->cctx;

	free(cctx);
	return bencryptops()->close(bctx);
}

struct bops *
bcompressops(void)
{
	return &bops;
}
