#include "config.h"

/*
 * These are the actual sizes of the structs in the
 * file format itself.  The types are serialized/deserialized
 * using the helpers from types.c.  Any modification made to
 * the structs below will need to be reflected here and in types.c.
 */
#define MSG_SIZE	256
#define MD_SIZE		32

#define SNAP_HDR_SIZE	104
#define BLK_HDR_SIZE	16
#define BLK_DESC_SIZE	(MD_SIZE + 16)
#define SNAPSHOT_SIZE	(8 + MSG_SIZE + MD_SIZE + 8)

/* file format version */
#define VER_MIN	2
#define VER_MAJ	0

/* snapshot header and block header flags */
#define VER_MIN_MASK	0xff
#define VER_MAJ_SHIFT	8
#define VER_MAJ_MASK	0xff

/* block header flags */
#define HASH_ALGO_SHIFT		19
#define HASH_ALGO_MASK		0x7	/* max 8 hash algos */
#define COMPR_ALGO_SHIFT	16
#define COMPR_ALGO_MASK		0x7	/* max 8 compression algos */

enum compr_algo {
	COMPR_NONE,
	COMPR_LZ4,
	COMPR_SNAPPY,
	NR_COMPRS,
};

enum hash_algo {
	HASH_BLAKE2B,
	HASH_BLAKE2BP,
	HASH_BLAKE2S,
	HASH_BLAKE2SP,
	NR_HASHES,
};

struct chunker;
struct icache;

struct stats {
	uint64_t orig_size;	/* original store size */
	uint64_t compr_size;	/* compressed store size */
	uint64_t dedup_size;	/* deduplicated store size */
	uint64_t min_blk_size;
	uint64_t max_blk_size;
	uint64_t nr_blks;	/* number of unique blocks */
	uint64_t reserved[4];
};

struct snap_hdr {
	uint64_t flags;
	uint64_t size;		/* size of snapshots file */
	uint64_t nr_snaps;
	struct stats st;
};

struct blk_hdr {
	uint64_t flags;
	uint64_t size;		/* size of store file */
};

struct blk_desc {
	uint8_t md[MD_SIZE];	/* hash of block */
	uint64_t offset;	/* offset into store file */
	uint64_t size;		/* size of block */
};

struct snap {
	uint64_t size;		/* size of snapshot (including block descriptors) */
	uint8_t msg[MSG_SIZE];	/* arbitrary message attached to snapshot */
	uint8_t md[MD_SIZE];	/* hash of snapshot (hash of all block descriptor hashes) */
	uint64_t nr_blk_descs;
	struct blk_desc blk_desc[];
};

struct compr_ctx {
	struct compr_ops *ops;
};

struct hash_ctx {
	union {
		blake2b_state blake2b_ctx;
		blake2bp_state blake2bp_ctx;
		blake2s_state blake2s_ctx;
		blake2sp_state blake2sp_ctx;
	} u;
	struct hash_ops *ops;
};

/* dedup.c */
extern int verbose;

/* chunker.c */
struct chunker *alloc_chunker(int fd, size_t min_size, size_t max_size,
                              size_t mask, size_t win_size);
void free_chunker(struct chunker *chunker);
ssize_t fill_chunker(struct chunker *chunker);
uint8_t *get_chunk(struct chunker *chunker, size_t *chunk_size);
void drain_chunker(struct chunker *chunker);

/* compress-none.c */
int none_init(struct compr_ctx *ctx);
size_t none_size(struct compr_ctx *ctx, size_t n);
size_t none_compr(struct compr_ctx *ctx, const void *in, void *out,
                  size_t insize, size_t outsize);
size_t none_decompr(struct compr_ctx *ctx, const void *in, void *out,
                    size_t insize, size_t outsize);
int none_final(struct compr_ctx *ctx);

/* compress-lz4.c */
int lz4_init(struct compr_ctx *ctx);
size_t lz4_size(struct compr_ctx *ctx, size_t n);
size_t lz4_compr(struct compr_ctx *ctx, const void *in, void *out,
                 size_t insize, size_t outsize);
size_t lz4_decompr(struct compr_ctx *ctx, const void *in, void *out,
                   size_t insize, size_t outsize);
int lz4_final(struct compr_ctx *ctx);

/* compress-snappy.c */
int snappy_init(struct compr_ctx *ctx);
size_t snappy_size(struct compr_ctx *ctx, size_t n);
size_t snappy_compr(struct compr_ctx *ctx, const void *in, void *out,
                    size_t insize, size_t outsize);
size_t snappy_decompr(struct compr_ctx *ctx, const void *in, void *out,
                      size_t insize, size_t outsize);
int snappy_final(struct compr_ctx *ctx);

/* compress.c */
int compr_init(struct compr_ctx *ctx, int type);
int compr_size(struct compr_ctx *ctx, size_t n);
size_t compr(struct compr_ctx *ctx, const void *in, void *out,
             size_t insize, size_t outsize);
size_t decompr(struct compr_ctx *ctx, const void *in, void *out,
               size_t insize, size_t outsize);
int compr_final(struct compr_ctx *ctx);
int compr_name2type(char *name);
char *compr_type2name(int type);

/* hash-blake2b.c */
int blake2bi(struct hash_ctx *ctx, size_t n);
int blake2bu(struct hash_ctx *ctx, const void *buf, size_t n);
int blake2bf(struct hash_ctx *ctx, void *buf, size_t n);

/* hash-blake2bp.c */
int blake2bpi(struct hash_ctx *ctx, size_t n);
int blake2bpu(struct hash_ctx *ctx, const void *buf, size_t n);
int blake2bpf(struct hash_ctx *ctx, void *buf, size_t n);

/* hash-blake2s.c */
int blake2si(struct hash_ctx *ctx, size_t n);
int blake2su(struct hash_ctx *ctx, const void *buf, size_t n);
int blake2sf(struct hash_ctx *ctx, void *buf, size_t n);

/* hash-blake2sp.c */
int blake2spi(struct hash_ctx *ctx, size_t n);
int blake2spu(struct hash_ctx *ctx, const void *buf, size_t n);
int blake2spf(struct hash_ctx *ctx, void *buf, size_t n);

/* hash.c */
int hash_init(struct hash_ctx *ctx, int type, size_t n);
int hash_update(struct hash_ctx *ctx, const void *buf, size_t n);
int hash_final(struct hash_ctx *ctx, void *buf, size_t n);
int hash_name2type(char *name);
char *hash_type2name(int type);

/* icache.c */
struct icache *alloc_icache(void);
void free_icache(struct icache *icache);
void insert_icache(struct icache *icache, struct blk_desc *desc);
int lookup_icache(struct icache *icache, struct blk_desc *desc);
void icache_stats(struct icache *icache, unsigned long long *hits,
                  unsigned long long *misses);

/* pack.c */
int pack(unsigned char *dst, char *fmt, ...);

/* unpack.c */
int unpack(unsigned char *src, char *fmt, ...);

/* types.c */
void read_snap_hdr(int fd, struct snap_hdr *hdr);
void write_snap_hdr(int fd, struct snap_hdr *hdr);
void read_blk_hdr(int fd, struct blk_hdr *hdr);
void write_blk_hdr(int fd, struct blk_hdr *hdr);
void read_blk_desc(int fd, struct blk_desc *desc);
void write_blk_desc(int fd, struct blk_desc *desc);
void read_snap(int fd, struct snap *snap);
void read_snap_descs(int fd, struct snap *snap);
void write_snap(int fd, struct snap *snap);
void write_snap_blk_descs(int fd, struct snap *snap);

/* utils.c */
void str2bin(char *s, uint8_t *d);
off_t xlseek(int fd, off_t offset, int whence);
ssize_t xread(int fd, void *buf, size_t nbytes);
ssize_t xwrite(int fd, const void *buf, size_t nbytes);
