#include <sys/types.h>

#include <err.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

#include "blake2.h"
#include "dedup.h"
#include "tree.h"

struct node {
	struct blk_desc desc;
	RB_ENTRY(node) e;
};
RB_HEAD(icache_head, node);

struct icache {
	struct icache_head nodes;
	unsigned long long hits;
	unsigned long long misses;
};

static int
node_cmp(struct node *e1, struct node *e2)
{
	int r;

	r = memcmp(e1->desc.md, e2->desc.md, sizeof(e1->desc.md));
	if (r > 0)
		return 1;
	else if (r < 0)
		return -1;
	return 0;
}
static RB_PROTOTYPE(icache_head, node, e, node_cmp);
static RB_GENERATE(icache_head, node, e, node_cmp);

static struct node *
alloc_node(struct blk_desc *desc)
{
	struct node *node;

	node = calloc(1, sizeof(*node));
	if (node == NULL)
		err(1, "calloc");
	node->desc = *desc;
	return node;
}

static void
free_node(struct node *node)
{
	free(node);
}

struct icache *
alloc_icache(void)
{
	struct icache *icache;

	icache = calloc(1, sizeof(*icache));
	if (icache == NULL)
		err(1, "calloc");
	RB_INIT(&icache->nodes);
	return icache;
}

void
free_icache(struct icache *icache)
{
	struct node *node, *tmp;

	RB_FOREACH_SAFE(node, icache_head, &icache->nodes, tmp) {
		RB_REMOVE(icache_head, &icache->nodes, node);
		free_node(node);
	}
	free(icache);
}

void
insert_icache(struct icache *icache, struct blk_desc *desc)
{
	struct node *node;

	node = alloc_node(desc);
	if (RB_INSERT(icache_head, &icache->nodes, node) != NULL)
		free_node(node);
}

int
lookup_icache(struct icache *icache, struct blk_desc *desc)
{
	struct node *node, key;

	key.desc = *desc;
	node = RB_FIND(icache_head, &icache->nodes, &key);
	if (node != NULL) {
		icache->hits++;
		*desc = node->desc;
		return 0;
	}
	icache->misses++;
	return -1;
}

void
icache_stats(struct icache *icache, unsigned long long *hits,
             unsigned long long *misses)
{
	*hits = icache->hits;
	*misses = icache->misses;
}

