#include <sys/types.h>

#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>

#include "blake2.h"
#include "dedup.h"

static struct hash_ops {
	int (*init)(struct hash_ctx *ctx, size_t n);
	int (*update)(struct hash_ctx *ctx, const void *buf, size_t n);
	int (*final)(struct hash_ctx *ctx, void *buf, size_t n);
} hashes[NR_HASHES] = {
	{
		.init = blake2bi,
		.update = blake2bu,
		.final = blake2bf,
	},
	{
		.init = blake2bpi,
		.update = blake2bpu,
		.final = blake2bpf,
	},
	{
		.init = blake2si,
		.update = blake2su,
		.final = blake2sf,
	},
	{
		.init = blake2spi,
		.update = blake2spu,
		.final = blake2spf,
	},
};

static struct algomap {
	char *name;
	int type;
} algomap[] = {
	{ .name = "blake2b", .type = HASH_BLAKE2B },
	{ .name = "blake2bp", .type = HASH_BLAKE2BP },
	{ .name = "blake2s", .type = HASH_BLAKE2S },
	{ .name = "blake2sp", .type = HASH_BLAKE2SP },
	{ .name = NULL, .type = -1 },
};

int
hash_init(struct hash_ctx *ctx, int type, size_t n)
{
	if (type < 0 || type >= NR_HASHES)
		return -1;

	ctx->ops = &hashes[type];
	return (*ctx->ops->init)(ctx, n);
}

int
hash_update(struct hash_ctx *ctx, const void *buf, size_t n)
{
	return (*ctx->ops->update)(ctx, buf, n);
}

int
hash_final(struct hash_ctx *ctx, void *buf, size_t n)
{
	return (*ctx->ops->final)(ctx, buf, n);
}

int
hash_name2type(char *name)
{
	struct algomap *algo;

	for (algo = &algomap[0]; algo->name != NULL; algo++)
		if (strcasecmp(algo->name, name) == 0)
			break;
	if (algo->name == NULL)
		return -1;
	return algo->type;
}

char *
hash_type2name(int type)
{
	struct algomap *algo;

	for (algo = &algomap[0]; algo->name != NULL; algo++)
		if (algo->type == type)
			break;
	if (algo->name == NULL)
		return NULL;
	return algo->name;
}
