#include <sys/types.h>

#include <err.h>
#include <stdint.h>
#include <string.h>
#include <strings.h>

#include "blake2.h"
#include "dedup.h"

static struct compr_ops {
	int (*init)(struct compr_ctx *ctx);
	size_t (*size)(struct compr_ctx *ctx, size_t n);
	size_t (*compr)(struct compr_ctx *ctx, const void *in, void *out,
	                size_t insize, size_t outsize);
	size_t (*decompr)(struct compr_ctx *ctx, const void *in, void *out,
	                  size_t insize, size_t outsize);
	int (*final)(struct compr_ctx *ctx);
} comprs[NR_COMPRS] = {
	{
		.init = none_init,
		.size = none_size,
		.compr = none_compr,
		.decompr = none_decompr,
		.final = none_final,
	},
	{
		.init = lz4_init,
		.size = lz4_size,
		.compr = lz4_compr,
		.decompr = lz4_decompr,
		.final = lz4_final,
	},
	{
		.init = snappy_init,
		.size = snappy_size,
		.compr = snappy_compr,
		.decompr = snappy_decompr,
		.final = snappy_final,
	},
};

static struct algomap {
	char *name;
	int type;
} algomap[] = {
	{ .name = "none", .type = COMPR_NONE },
	{ .name = "lz4", .type = COMPR_LZ4 },
	{ .name = "snappy", .type = COMPR_SNAPPY },
	{ .name = NULL, .type = -1 },
};

int
compr_init(struct compr_ctx *ctx, int type)
{
	if (type < 0 || type >= NR_COMPRS)
		return -1;

	ctx->ops = &comprs[type];
	return (*ctx->ops->init)(ctx);
}

int
compr_size(struct compr_ctx *ctx, size_t n)
{
	return (*ctx->ops->size)(ctx, n);
}

size_t
compr(struct compr_ctx *ctx, const void *in, void *out,
      size_t insize, size_t outsize)
{
	return (*ctx->ops->compr)(ctx, in, out, insize, outsize);
}

size_t
decompr(struct compr_ctx *ctx, const void *in, void *out,
        size_t insize, size_t outsize)
{
	return (*ctx->ops->decompr)(ctx, in, out, insize, outsize);
}

int
compr_final(struct compr_ctx *ctx)
{
	return (*ctx->ops->final)(ctx);
}

int
compr_name2type(char *name)
{
	struct algomap *algo;

	for (algo = &algomap[0]; algo->name != NULL; algo++)
		if (strcasecmp(algo->name, name) == 0)
			break;
	if (algo->name == NULL)
		return -1;
	return algo->type;
}

char *
compr_type2name(int type)
{
	struct algomap *algo;

	for (algo = &algomap[0]; algo->name != NULL; algo++)
		if (algo->type == type)
			break;
	if (algo->name == NULL)
		return NULL;
	return algo->name;
}
