#include "config.h"

/*
 * These are the actual sizes of the structs in the
 * file format itself.  The types are serialized/deserialized
 * using the helpers from types.c.  Any modification made to
 * the structs below will need to be reflected here and in types.c.
 */
#define MSGSIZE	256
#define MDSIZE	32

#define SNAP_HDR_SIZE	104
#define BLK_HDR_SIZE	16
#define BLK_DESC_SIZE	(MDSIZE + 16)
#define SNAPSHOT_SIZE	(8 + MSGSIZE + MDSIZE + 8)

/* file format version */
#define VER_MIN	2
#define VER_MAJ	0

#define VER_MIN_MASK	0xff
#define VER_MAJ_SHIFT	8
#define VER_MAJ_MASK	0xff

#define COMPR_ENABLED_SHIFT	16
#define COMPR_ENABLED_MASK	0x1

struct cache;
struct chunker;

struct stats {
	uint64_t orig_size;	/* original store size */
	uint64_t compr_size;	/* compressed store size */
	uint64_t dedup_size;	/* deduplicated store size */
	uint64_t min_blk_size;
	uint64_t max_blk_size;
	uint64_t nr_blks;	/* number of unique blocks */
	uint64_t reserved[4];
};

struct snap_hdr {
	uint64_t flags;		/* bottom 16 bits are maj/min version */
	uint64_t size;		/* size of snapshots file */
	uint64_t nr_snaps;
	struct stats st;
};

struct blk_hdr {
	uint64_t flags;		/* bottom 16 bits are maj/min version */
	uint64_t size;		/* size of store file */
};

struct blk_desc {
	uint8_t md[MDSIZE];	/* hash of block */
	uint64_t offset;	/* offset into store file */
	uint64_t size;		/* size of block */
};

struct snap {
	uint64_t size;		/* size of snapshot (including block descriptors) */
	uint8_t msg[MSGSIZE];	/* arbitrary message attached to snapshot */
	uint8_t md[MDSIZE];	/* hash of snapshot (hash of all block descriptor hashes) */
	uint64_t nr_blk_descs;
	struct blk_desc blk_desc[];
};

/* compress.c */
extern int compr_enabled;

/* dedup.c */
extern int verbose;

/* cache.c */
struct cache *alloc_cache(void);
void free_cache(struct cache *cache);
void add_cache_entry(struct cache *cache, struct blk_desc *desc);
int lookup_cache_entry(struct cache *cache, struct blk_desc *desc);

/* chunker.c */
struct chunker *alloc_chunker(int fd, size_t min_size, size_t max_size,
                              size_t mask, size_t win_size);
void free_chunker(struct chunker *chunker);
ssize_t fill_chunker(struct chunker *chunker);
uint8_t *get_chunk(struct chunker *chunker, size_t *chunk_size);
void drain_chunker(struct chunker *chunker);

/* compress.c */
size_t compr_size(size_t size);
size_t compr(uint8_t *in, uint8_t *out, size_t insize, size_t outsize);
size_t decompr(uint8_t *in, uint8_t *out, size_t insize, size_t outsize);

/* pack.c */
int pack(unsigned char *dst, char *fmt, ...);

/* unpack.c */
int unpack(unsigned char *src, char *fmt, ...);

/* types.c */
void read_snap_hdr(int fd, struct snap_hdr *hdr);
void write_snap_hdr(int fd, struct snap_hdr *hdr);
void read_blk_hdr(int fd, struct blk_hdr *hdr);
void write_blk_hdr(int fd, struct blk_hdr *hdr);
void read_blk_desc(int fd, struct blk_desc *desc);
void write_blk_desc(int fd, struct blk_desc *desc);
void read_snap(int fd, struct snap *snap);
void read_snap_descs(int fd, struct snap *snap);
void write_snap(int fd, struct snap *snap);
void write_snap_blk_descs(int fd, struct snap *snap);

/* utils.c */
void str2bin(char *s, uint8_t *d);
off_t xlseek(int fd, off_t offset, int whence);
ssize_t xread(int fd, void *buf, size_t nbytes);
ssize_t xwrite(int fd, const void *buf, size_t nbytes);
