#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>

#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "arg.h"
#include "blake2.h"
#include "dedup.h"

#define SNAPSF ".snapshots"
#define STOREF ".store"

enum {
	WALK_CONTINUE,
	WALK_STOP
};

struct extract_args {
	uint8_t *md;
	int fd;
	int ret;
};

static struct snapshot_hdr snap_hdr;
static struct blk_hdr blk_hdr;
static struct cache *cache;
static int ifd;
static int sfd;
static unsigned long long cache_hits;
static unsigned long long cache_misses;

int verbose;
char *argv0;

static void
print_md(FILE *fp, uint8_t *md, size_t size)
{
	size_t i;

	for (i = 0; i < size; i++)
		fprintf(fp, "%02x", md[i]);
}

static void
print_stats(struct stats *st)
{
	if (st->nr_blks == 0)
		return;

	fprintf(stderr, "Original size: %llu bytes\n",
	        (unsigned long long)st->orig_size);
	fprintf(stderr, "Compressed size: %llu bytes\n",
	        (unsigned long long)st->compr_size);
	fprintf(stderr, "Deduplicated size: %llu bytes\n",
	        (unsigned long long)st->dedup_size);
	fprintf(stderr, "Min/avg/max block size: %llu/%llu/%llu bytes\n",
	        (unsigned long long)st->min_blk_size,
	        (unsigned long long)st->dedup_size / st->nr_blks,
	        (unsigned long long)st->max_blk_size);
	fprintf(stderr, "Number of unique blocks: %llu\n",
	        (unsigned long long)st->nr_blks);
	fprintf(stderr, "Cache hits: %llu\n", cache_hits);
	fprintf(stderr, "Cache misses: %llu\n", cache_misses);
}

static struct snapshot *
alloc_snap(void)
{
	struct snapshot *snap;

	snap = calloc(1, sizeof(*snap));
	if (snap == NULL)
		err(1, "%s", __func__);
	return snap;
}

static void
free_snap(struct snapshot *snap)
{
	free(snap);
}

/*
 * The snapshot hash is calculated over the
 * hash of its block descriptors.
 */
static void
hash_snap(struct snapshot *snap, uint8_t *md)
{
	blake2b_state ctx;
	uint64_t i;

	blake2b_init(&ctx, MDSIZE);
	for (i = 0; i < snap->nr_blk_descs; i++) {
		struct blk_desc *blk_desc;

		blk_desc = &snap->blk_desc[i];
		blake2b_update(&ctx, blk_desc->md, sizeof(blk_desc->md));
	}
	blake2b_final(&ctx, md, MDSIZE);
}

static struct snapshot *
grow_snap(struct snapshot *snap, uint64_t nr_blk_descs)
{
	size_t size;

	if (nr_blk_descs > SIZE_MAX / sizeof(snap->blk_desc[0]))
		errx(1, "%s: overflow", __func__);
	size = nr_blk_descs * sizeof(snap->blk_desc[0]);

	if (size > SIZE_MAX - sizeof(*snap))
		errx(1, "%s: overflow", __func__);
	size += sizeof(*snap);

	snap = realloc(snap, size);
	if (snap == NULL)
		err(1, "%s", __func__);
	return snap;
}

static void
append_snap(struct snapshot *snap)
{
	if (snap->nr_blk_descs > UINT64_MAX / BLK_DESC_SIZE)
		errx(1, "%s: overflow", __func__);
	snap->size = snap->nr_blk_descs * BLK_DESC_SIZE;

	if (snap->size > UINT64_MAX - SNAPSHOT_SIZE)
		errx(1, "%s: overflow", __func__);
	snap->size += SNAPSHOT_SIZE;

	xlseek(ifd, snap_hdr.size, SEEK_SET);
	write_snapshot(ifd, snap);
	write_snapshot_blk_descs(ifd, snap);

	if (snap_hdr.size > UINT64_MAX - snap->size)
		errx(1, "%s: overflow", __func__);
	snap_hdr.size += snap->size;

	if (snap_hdr.nr_snapshots > UINT64_MAX - 1)
		errx(1, "%s: overflow", __func__);
	snap_hdr.nr_snapshots++;
}

static uint8_t *
alloc_buf(size_t size)
{
	void *p;

	p = calloc(1, size);
	if (p == NULL)
		err(1, "%s", __func__);
	return p;
}

static void
free_buf(uint8_t *buf)
{
	free(buf);
}

static void
hash_blk(uint8_t *buf, size_t size, uint8_t *md)
{
	blake2b_state ctx;

	blake2b_init(&ctx, MDSIZE);
	blake2b_update(&ctx, buf, size);
	blake2b_final(&ctx, md, MDSIZE);
}

static void
read_blk(uint8_t *buf, struct blk_desc *blk_desc)
{
	ssize_t n;

	xlseek(sfd, blk_desc->offset, SEEK_SET);
	n = xread(sfd, buf, blk_desc->size);
	if (n == 0)
		errx(1, "%s: unexpected EOF", __func__);
	if (n != blk_desc->size)
		errx(1, "%s: short read", __func__);
}

static void
append_blk(uint8_t *buf, struct blk_desc *blk_desc)
{
	xlseek(sfd, blk_hdr.size, SEEK_SET);
	xwrite(sfd, buf, blk_desc->size);

	if (blk_hdr.size > UINT64_MAX - blk_desc->size)
		errx(1, "%s: overflow", __func__);
	blk_hdr.size += blk_desc->size;
}

static void
dedup_chunk(struct snapshot *snap, uint8_t *chunkp, size_t chunk_size)
{
	uint8_t md[MDSIZE];
	struct blk_desc blk_desc;
	uint8_t *compr_buf;
	size_t n;

	compr_buf = alloc_buf(compr_size(BLKSIZE_MAX));

	n = compr(chunkp, compr_buf, chunk_size, compr_size(BLKSIZE_MAX));
	hash_blk(compr_buf, n, md);

	snap_hdr.st.orig_size += chunk_size;
	snap_hdr.st.compr_size += n;

	memcpy(blk_desc.md, md, sizeof(blk_desc.md));
	if (lookup_cache_entry(cache, &blk_desc) < 0) {
		blk_desc.offset = blk_hdr.size;
		blk_desc.size = n;

		snap->blk_desc[snap->nr_blk_descs++] = blk_desc;
		append_blk(compr_buf, &blk_desc);

		add_cache_entry(cache, &blk_desc);
		cache_misses++;

		snap_hdr.st.dedup_size += blk_desc.size;
		snap_hdr.st.nr_blks++;

		if (blk_desc.size > snap_hdr.st.max_blk_size)
			snap_hdr.st.max_blk_size = blk_desc.size;
		if (blk_desc.size < snap_hdr.st.min_blk_size)
			snap_hdr.st.min_blk_size = blk_desc.size;
	} else {
		snap->blk_desc[snap->nr_blk_descs++] = blk_desc;
		cache_hits++;
	}

	free(compr_buf);
}

static void
dedup(int fd, char *msg)
{
	struct snapshot *snap;
	struct chunker *chunker;

	snap = alloc_snap();
	chunker = alloc_chunker(fd, BLKSIZE_MIN, BLKSIZE_MAX,
	                        HASHMASK_BITS, WINSIZE);

	while (fill_chunker(chunker) > 0) {
		uint8_t *chunkp;
		size_t chunk_size;

		chunkp = get_chunk(chunker, &chunk_size);
		snap = grow_snap(snap, snap->nr_blk_descs + 1);
		dedup_chunk(snap, chunkp, chunk_size);
		drain_chunker(chunker);
	}

	if (snap->nr_blk_descs > 0) {
		if (msg != NULL) {
			size_t size;

			size = strlen(msg) + 1;
			if (size > sizeof(snap->msg))
				size = sizeof(snap->msg);
			memcpy(snap->msg, msg, size);
			snap->msg[size - 1] = '\0';
		}
		hash_snap(snap, snap->md);
		append_snap(snap);
	}

	free_chunker(chunker);
	free_snap(snap);
}

static int
extract(struct snapshot *snap, void *arg)
{
	uint8_t *buf[2];
	struct extract_args *args = arg;
	uint64_t i;

	if (memcmp(snap->md, args->md, sizeof(snap->md)) != 0)
		return WALK_CONTINUE;

	buf[0] = alloc_buf(BLKSIZE_MAX);
	buf[1] = alloc_buf(compr_size(BLKSIZE_MAX));
	for (i = 0; i < snap->nr_blk_descs; i++) {
		struct blk_desc *blk_desc;
		size_t blksize;

		blk_desc = &snap->blk_desc[i];
		read_blk(buf[1], blk_desc);
		blksize = decompr(buf[1], buf[0], blk_desc->size, BLKSIZE_MAX);
		xwrite(args->fd, buf[0], blksize);
	}
	free_buf(buf[1]);
	free_buf(buf[0]);
	args->ret = 0;
	return WALK_STOP;
}

/*
 * For each snapshot, hash every block and check if the hash
 * matches the one in the corresponding block descriptor.
 */
static int
check_snap(struct snapshot *snap, void *arg)
{
	uint8_t *buf;
	int *ret = arg;
	uint64_t i;

	if (verbose > 0) {
		fprintf(stderr, "Checking snapshot: ");
		print_md(stderr, snap->md, sizeof(snap->md));
		fputc('\n', stderr);
	}

	buf = alloc_buf(compr_size(BLKSIZE_MAX));
	for (i = 0; i < snap->nr_blk_descs; i++) {
		uint8_t md[MDSIZE];
		blake2b_state ctx;
		struct blk_desc *blk_desc;

		blk_desc = &snap->blk_desc[i];
		read_blk(buf, blk_desc);

		blake2b_init(&ctx, MDSIZE);
		blake2b_update(&ctx, buf, blk_desc->size);
		blake2b_final(&ctx, md, MDSIZE);

		if (memcmp(blk_desc->md, md, sizeof(blk_desc->md)) == 0)
			continue;

		fprintf(stderr, "Block hash mismatch\n");
		fprintf(stderr, "  Expected hash: ");
		print_md(stderr, blk_desc->md, sizeof(blk_desc->md));
		fputc('\n', stderr);
		fprintf(stderr, "  Actual hash: ");
		print_md(stderr, md, sizeof(md));
		fputc('\n', stderr);
		fprintf(stderr, "  Offset: %llu\n",
		        (unsigned long long)blk_desc->offset);
		fprintf(stderr, "  Size: %llu\n",
		        (unsigned long long)blk_desc->size);
		*ret = -1;
	}
	free_buf(buf);
	return WALK_CONTINUE;
}

static int
build_cache(struct snapshot *snap, void *arg)
{
	uint8_t *buf;
	uint64_t i;

	buf = alloc_buf(compr_size(BLKSIZE_MAX));
	for (i = 0; i < snap->nr_blk_descs; i++) {
		struct blk_desc *blk_desc;

		blk_desc = &snap->blk_desc[i];
		add_cache_entry(cache, blk_desc);
	}
	free(buf);
	return WALK_CONTINUE;
}

static int
list(struct snapshot *snap, void *arg)
{
	print_md(stdout, snap->md, sizeof(snap->md));
	if (snap->msg[0] != '\0')
		printf("\t%s\n", snap->msg);
	else
		putchar('\n');
	return WALK_CONTINUE;
}

/* Walk through all snapshots and call fn() on each one */
static void
walk_snap(int (*fn)(struct snapshot *, void *), void *arg)
{
	uint64_t i;

	xlseek(ifd, SNAP_HDR_SIZE, SEEK_SET);
	for (i = 0; i < snap_hdr.nr_snapshots; i++) {
		struct snapshot *snap;
		int ret;

		snap = alloc_snap();
		read_snapshot(ifd, snap);
		snap = grow_snap(snap, snap->nr_blk_descs);
		read_snapshot_descs(ifd, snap);

		ret = (*fn)(snap, arg);
		free(snap);
		if (ret == WALK_STOP)
			break;
	}
}

static void
match_ver(uint64_t v)
{
	uint8_t maj, min;

	min = v & VER_MIN_MASK;
	maj = (v >> VER_MAJ_SHIFT) & VER_MAJ_MASK;
	if (maj == VER_MAJ && min == VER_MIN)
		return;
	errx(1, "format version mismatch: expected %u.%u but got %u.%u",
	     VER_MAJ, VER_MIN, maj, min);
}

static void
init_blk_hdr(void)
{
	blk_hdr.flags = (VER_MAJ << VER_MAJ_SHIFT) | VER_MIN;
	blk_hdr.flags |= compr_enabled << COMPR_ENABLED_SHIFT;
	blk_hdr.size = BLK_HDR_SIZE;
}

static void
load_blk_hdr(void)
{
	uint64_t v;

	xlseek(sfd, 0, SEEK_SET);
	read_blk_hdr(sfd, &blk_hdr);
	match_ver(blk_hdr.flags);

	v = blk_hdr.flags >> COMPR_ENABLED_SHIFT;
	v &= COMPR_ENABLED_MASK;
	compr_enabled = v;
}

static void
save_blk_hdr(void)
{
	xlseek(sfd, 0, SEEK_SET);
	write_blk_hdr(sfd, &blk_hdr);
}

static void
init_snap_hdr(void)
{
	snap_hdr.flags = (VER_MAJ << VER_MAJ_SHIFT) | VER_MIN;
	snap_hdr.size = SNAP_HDR_SIZE;
	snap_hdr.st.min_blk_size = compr_size(BLKSIZE_MAX);
}

static void
load_snap_hdr(void)
{
	xlseek(ifd, 0, SEEK_SET);
	read_snap_hdr(ifd, &snap_hdr);
	match_ver(snap_hdr.flags);
}

static void
save_snap_hdr(void)
{
	xlseek(ifd, 0, SEEK_SET);
	write_snap_hdr(ifd, &snap_hdr);
}

static void
init(int iflag)
{
	int flags;

	flags = O_RDWR;
	if (iflag)
		flags |= O_CREAT | O_EXCL;

	ifd = open(SNAPSF, flags, 0600);
	if (ifd < 0)
		err(1, "open %s", SNAPSF);

	sfd = open(STOREF, flags, 0600);
	if (sfd < 0)
		err(1, "open %s", STOREF);

	if (flock(ifd, LOCK_NB | LOCK_EX) < 0 ||
	    flock(sfd, LOCK_NB | LOCK_EX) < 0)
		err(1, "flock");

	if (iflag) {
		init_snap_hdr();
		init_blk_hdr();
	} else {
		load_snap_hdr();
		load_blk_hdr();
	}

	cache = alloc_cache();
	walk_snap(build_cache, NULL);
}

static void
term(void)
{
	if (verbose > 0)
		print_stats(&snap_hdr.st);

	free_cache(cache);

	save_blk_hdr();
	save_snap_hdr();

	fsync(sfd);
	fsync(ifd);

	close(sfd);
	close(ifd);
}

static void
usage(void)
{
	fprintf(stderr, "usage: %s [-Zcilv] [-e id] [-r root] [-m message] [file]\n", argv0);
	exit(1);
}

int
main(int argc, char *argv[])
{
	uint8_t md[MDSIZE];
	char *id = NULL, *root = NULL, *msg = NULL;
	int iflag = 0, lflag = 0, cflag = 0;
	int fd = -1;

	ARGBEGIN {
	case 'Z':
		compr_enabled = 0;
		break;
	case 'c':
		cflag = 1;
		break;
	case 'e':
		id = EARGF(usage());
		break;
	case 'i':
		iflag = 1;
		break;
	case 'l':
		lflag = 1;
		break;
	case 'r':
		root = EARGF(usage());
		break;
	case 'm':
		msg = EARGF(usage());
		break;
	case 'v':
		verbose++;
		break;
	default:
		usage();
	} ARGEND

	if (argc > 1) {
		usage();
	} else if (argc == 1) {
		if (id) {
			fd = open(argv[0], O_RDWR | O_CREAT, 0600);
			if (fd < 0)
				err(1, "open %s", argv[0]);
		} else {
			fd = open(argv[0], O_RDONLY);
			if (fd < 0)
				err(1, "open %s", argv[0]);
		}
	} else {
		if (id)
			fd = STDOUT_FILENO;
		else
			fd = STDIN_FILENO;
	}

	if (root != NULL) {
		mkdir(root, 0700);
		if (chdir(root) < 0)
			err(1, "chdir: %s", root);
	}

	init(iflag);

	if (iflag) {
		term();
		return 0;
	}

	if (cflag) {
		int ret;

		ret = 0;
		walk_snap(check_snap, &ret);
		if (ret != 0)
			errx(1, "%s or %s is corrupted", SNAPSF, STOREF);

		term();
		return 0;
	}

	if (lflag) {
		walk_snap(list, NULL);
		term();
		return 0;
	}

	if (id) {
		struct extract_args args;

		str2bin(id, md);
		args.md = md;
		args.fd = fd;
		args.ret = -1;
		walk_snap(extract, &args);
		if (args.ret != 0)
			errx(1, "unknown snapshot: %s", id);
	} else {
		dedup(fd, msg);
	}

	term();
	return 0;
}
