#include <assert.h>
#include <err.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>

#include "dedup.h"

void
read_snap_hdr(int fd, struct snapshot_hdr *hdr)
{
	uint8_t buf[SNAP_HDR_SIZE];
	int n;

	if (xread(fd, buf, sizeof(buf)) == 0)
		errx(1, "read_snap_hdr: unexpected EOF");

	n = unpack(buf, "qqq",
	           &hdr->flags,
	           &hdr->size,
	           &hdr->nr_snapshots);

	n += unpack(&buf[n], "qqqqqq",
	            &hdr->st.orig_size,
	            &hdr->st.compr_size,
	            &hdr->st.dedup_size,
	            &hdr->st.min_blk_size,
	            &hdr->st.max_blk_size,
	            &hdr->st.nr_blks);

	n += unpack(&buf[n], "qqqq",
	            &hdr->st.reserved[0],
	            &hdr->st.reserved[1],
	            &hdr->st.reserved[2],
	            &hdr->st.reserved[3]);

	assert(n == SNAP_HDR_SIZE);
}

void
write_snap_hdr(int fd, struct snapshot_hdr *hdr)
{
	uint8_t buf[SNAP_HDR_SIZE];
	int n;

	n = pack(buf, "qqq",
	         hdr->flags,
	         hdr->size,
	         hdr->nr_snapshots);

	n += pack(&buf[n], "qqqqqq",
	          hdr->st.orig_size,
	          hdr->st.compr_size,
	          hdr->st.dedup_size,
	          hdr->st.min_blk_size,
	          hdr->st.max_blk_size,
	          hdr->st.nr_blks);

	n += pack(&buf[n], "qqqq",
	          hdr->st.reserved[0],
	          hdr->st.reserved[1],
	          hdr->st.reserved[2],
	          hdr->st.reserved[3]);

	assert(n == SNAP_HDR_SIZE);
	xwrite(fd, buf, n);
}

void
read_blk_hdr(int fd, struct blk_hdr *hdr)
{
	uint8_t buf[BLK_HDR_SIZE];
	int n;

	if (xread(fd, buf, sizeof(buf)) == 0)
		errx(1, "read_blk_desc: unexpected EOF");

	n = unpack(buf, "qq",
	           &hdr->flags,
	           &hdr->size);

	assert(n == BLK_HDR_SIZE);
}

void
write_blk_hdr(int fd, struct blk_hdr *hdr)
{
	uint8_t buf[BLK_HDR_SIZE];
	int n;

	n = pack(buf, "qq",
	         hdr->flags,
	         hdr->size);

	assert(n == BLK_HDR_SIZE);
	xwrite(fd, buf, n);
}

void
read_blk_desc(int fd, struct blk_desc *desc)
{
	uint8_t buf[BLK_DESC_SIZE];
	char fmt[BUFSIZ];
	int n;

	if (xread(fd, buf, sizeof(buf)) == 0)
		errx(1, "read_blk_desc: unexpected EOF");

	snprintf(fmt, sizeof(fmt), "'%dqq", MDSIZE);
	n = unpack(buf, fmt,
	           desc->md,
	           &desc->offset,
	           &desc->size);

	assert(n == BLK_DESC_SIZE);
}

void
write_blk_desc(int fd, struct blk_desc *desc)
{
	uint8_t buf[BLK_DESC_SIZE];
	char fmt[BUFSIZ];
	int n;

	snprintf(fmt, sizeof(fmt), "'%dqq", MDSIZE);
	n = pack(buf, fmt,
	         desc->md,
	         desc->offset,
	         desc->size);

	assert(n == BLK_DESC_SIZE);
	xwrite(fd, buf, n);
}

void
read_snapshot(int fd, struct snapshot *snap)
{
	uint8_t buf[SNAPSHOT_SIZE];
	char fmt[BUFSIZ];
	int n;

	if (xread(fd, buf, sizeof(buf)) == 0)
		errx(1, "read_blk_desc: unexpected EOF");

	snprintf(fmt, sizeof(fmt), "q'%d'%dq", MSGSIZE, MDSIZE);
	n = unpack(buf, fmt,
	           &snap->size,
	           snap->msg,
	           snap->md,
	           &snap->nr_blk_descs);

	assert(n == SNAPSHOT_SIZE);
};

void
read_snapshot_descs(int fd, struct snapshot *snap)
{
	uint64_t i;

	for (i = 0; i < snap->nr_blk_descs; i++)
		read_blk_desc(fd, &snap->blk_desc[i]);
}

void
write_snapshot(int fd, struct snapshot *snap)
{
	uint8_t buf[SNAPSHOT_SIZE];
	char fmt[BUFSIZ];
	int n;

	snprintf(fmt, sizeof(fmt), "q'%d'%dq", MSGSIZE, MDSIZE);
	n = pack(buf, fmt,
	         snap->size,
	         snap->msg,
	         snap->md,
	         snap->nr_blk_descs);

	assert(n == SNAPSHOT_SIZE);
	xwrite(fd, buf, n);
}

void
write_snapshot_blk_descs(int fd, struct snapshot *snap)
{
	uint64_t i;

	for (i = 0; i < snap->nr_blk_descs; i++)
		write_blk_desc(fd, &snap->blk_desc[i]);
}
