#include <sys/stat.h>
#include <sys/file.h>
#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <lz4.h>
#include <openssl/sha.h>

#include "arg.h"
#include "tree.h"

#define SNAPSF ".snapshots"
#define STOREF ".store"
#define CACHEF ".cache"

#define BLKSIZE 4096
#define WINSIZE 511
#define HASHMSK ((1ul << 10) - 1)
#define MSGSIZE 256
#define MDSIZE SHA256_DIGEST_LENGTH

/* file format version */
#define VER_MIN 1
#define VER_MAJ 0

#define ROTL(x, y) (((x) << (y)) | ((x) >> (32 - (y))))

enum {
	WALK_CONTINUE,
	WALK_STOP
};

struct stats {
	uint64_t orig_size;
	uint64_t comp_size;
	uint64_t dedup_size;
	uint64_t min_blk_size;
	uint64_t max_blk_size;
	uint64_t nr_blks;
	uint64_t reserved[6];
};

struct snapshot_hdr {
	uint64_t flags;
	uint64_t nr_snapshots;
	uint64_t store_size;
	uint64_t reserved[4];
	struct stats st;
};

struct blk_desc {
	uint8_t md[MDSIZE];
	uint64_t offset;
	uint64_t size;
};

struct snapshot {
	uint64_t size;
	uint8_t msg[MSGSIZE];
	uint8_t md[MDSIZE];	/* hash of file */
	uint64_t nr_blk_descs;
	struct blk_desc blk_desc[];
};

struct cache_entry {
	struct blk_desc blk_desc;
	RB_ENTRY(cache_entry) e;
};

struct extract_args {
	uint8_t *md;
	int fd;
};

RB_HEAD(cache, cache_entry) cache_head;
struct snapshot_hdr snaphdr;
int ifd;
int sfd;
int cfd;
int verbose;
int cache_dirty;
unsigned long long cache_hits;
unsigned long long cache_misses;
char *argv0;

/*
 * Static table for use in buzhash algorithm.
 * 256 * 32 bits randomly generated unique integers
 *
 * To get better pseudo-random results, there is exactly the same number
 * of 0 and 1 spread amongst these integers. It means that there is
 * exactly 50% chance that a XOR operation would flip all the bits in
 * the hash.
 */
uint32_t buz[] = {
	0xbc9fa594,0x30a8f827,0xced627a7,0xdb46a745,0xcfa4a9e8,0x77cccb59,0xddb66276,0x3adc532f,
	0xfe8b67d3,0x8155b59e,0x0c893666,0x1d757009,0x17394ee4,0x85d94c07,0xcacd52da,0x076c6f79,
	0xead0a798,0x6c7ccb4a,0x2639a1b8,0x3aa5ae32,0x3e6218d2,0xb290d980,0xa5149521,0x4b426119,
	0xd3230fc7,0x677c1cc4,0x2b64603c,0x01fe92a8,0xbe358296,0xa7e7fac7,0xf509bf41,0x04b017ad,
	0xf900344c,0x8e14e202,0xb2a6e9b4,0x3db3c311,0x960286a8,0xf6bf0468,0xed54ec94,0xf358070c,
	0x6a4795dd,0x3f7b925c,0x5e13a060,0xfaecbafe,0x03c8bb55,0x8a56ba88,0x633e3b49,0xe036bbbe,
	0x1ed3dbb5,0x76e8ad74,0x79d346ab,0x44b4ccc4,0x71eb22d3,0xa1aa3f24,0x50e05b81,0xa3b450d3,
	0x7f5caffb,0xa1990650,0x54c44800,0xda134b65,0x72362eea,0xbd12b8e6,0xf7c99fdc,0x020d48c7,
	0x9d9c3d46,0x32b75615,0xe61923cf,0xadc09d8f,0xab11376b,0xd66fe4cd,0xb3b086b6,0xb8345b9f,
	0x59029667,0xae0e937c,0xcbd4d4ba,0x720bb3fb,0x5f7d2ca3,0xec24ba15,0x6b40109b,0xf0a54587,
	0x3acf9420,0x466e981d,0xc66dc124,0x150ef7b4,0xc3ce718e,0x136774f5,0x46684ab4,0xb4b490f0,
	0x26508a8b,0xf12febc8,0x4b99171b,0xfc373c84,0x339b5677,0x41703ff3,0x7cadbbd7,0x15ea24e2,
	0x7a2f9783,0xed6a383a,0x649eb072,0x79970941,0x2abd28ad,0x4375e00c,0x9df084f7,0x6fdeec6c,
	0x6619ac6d,0x7d256f4d,0x9b8e658a,0x3d7627e9,0xd5a98d45,0x15f84223,0x9b6acef5,0xf876be67,
	0xe3ae7089,0x84e2b64a,0x6818a969,0x86e9ba4e,0xa24a5b57,0x61570cf1,0xa5f8fc91,0x879d8383,
	0x91b13866,0x75e87961,0x16db8138,0x5a2ff6b8,0x8f664e9b,0x894e1496,0x88235c5b,0xcdb3b580,
	0xa2e80109,0xb0f88a82,0xd12cd340,0x93fbc37d,0xf4d1eb82,0xce42f309,0x16ffd2c2,0xb4dfef2b,
	0xb8b1a33e,0x4708a5e6,0xba66dd88,0xa9ec0da6,0x6f8ee2c9,0xad8b9993,0x1d6a25a8,0x1f3d08ce,
	0x149c04e7,0x5cd1fa51,0xb84c89c7,0xeced6f8c,0xe328b30f,0x084fa836,0x6d1bb1b7,0x94c78ea5,
	0x14973034,0xf1a1bcef,0x48b798d2,0xded9ca9e,0x5fd965d0,0x92544eb1,0x5e80f189,0xcbbf5e15,
	0x4d8121f0,0x5dd3b92f,0xd9ea98fb,0x2dbf5644,0x0fbcb9b7,0x20a1db53,0x7c3fcc98,0x36744fbd,
	0xced08954,0x8e7c5efe,0x3c5f6733,0x657477be,0x3630a02d,0x38bcbda0,0xb7702575,0x4a7f4bce,
	0x0e7660fe,0x4dcb91b5,0x4fd7ffd3,0x041821c1,0xa846a181,0xc8048e9e,0xd4b05072,0x986e0509,
	0xa00aaeeb,0x02e3526a,0x2fac4843,0xfa98e805,0x923ecd8d,0x395d9546,0x8674c3cd,0xae5a8a71,
	0x966dfe45,0x5c9ceba5,0x0830a1cf,0xa1750981,0x8f604480,0x28ea0c9a,0x0da12413,0x98b0b3c5,
	0xa21d473a,0x96ce4308,0xe9a1001b,0x8bbacb44,0x18bad3f4,0xe3121acb,0x46a9b45f,0x92cd9704,
	0xc1a7c619,0x3281e361,0x462e8c79,0x9e572f93,0x7239e5f0,0x67d8e6ba,0x13747ce3,0xf01ee64a,
	0xe7d0ae12,0xeea04088,0xe5b36767,0x17558eae,0x678ffbe6,0xe0bbc866,0x0c24adec,0xa9cbb869,
	0x3fd44ee1,0x9ca4ca06,0x04c0ef00,0x04589a21,0x9cf9c819,0x976f6ca1,0x8a30e66a,0x004d6f7e,
	0x384c8851,0x5bc97eb8,0xc6c49339,0x5aa386c7,0x74bdf8af,0x9b713750,0x4112f8c2,0x2895dae1,
	0xf576d905,0x9de98bce,0xb2b26bcd,0xd46707a0,0x147fbb46,0xa52c6e50,0xe43128fc,0x374ad964,
	0x8dfd4d53,0xc4d0c087,0x31dfb5ca,0xa44589b5,0x6b637e2e,0x663f6b45,0xd2d8baa0,0x1dac7e4c
};

/* Buzhash: https://en.wikipedia.org/wiki/Rolling_hash#Cyclic_polynomial */
uint32_t
buzh_init(uint8_t *buf, size_t size)
{
	size_t i;
	uint32_t fp;

	for (i = size - 1, fp = 0; i > 0; i--, buf++)
		fp ^= ROTL(buz[*buf], i % 32);

	return fp ^ buz[*buf];
}

uint32_t
buzh_update(uint32_t fp, uint8_t in, uint8_t out, size_t size)
{
	return ROTL(fp, 1) ^ ROTL(buz[out], size % 32) ^ buz[in];
}

uint64_t
chunk_blk(uint8_t *buf, size_t size)
{
	size_t i;
	uint32_t fp;

	/* buzhash should be at least WINSIZE */
	if (size < WINSIZE)
		return size;

	/*
	 * To achieve better deduplication, we chunk blocks based on a
	 * recurring pattern occuring on the data stream. A fixed window
	 * of WINSIZE bytes is slid over the data, and a rolling hash is
	 * computed for this window.
	 * When the rolling hash matches a given pattern (see HASHMSK),
	 * the block is chunked at the end of that window, thus making
	 * WINSIZE the smallest possible block size.
	 */
	fp = buzh_init(buf, WINSIZE);
	for (i = 0; i < size - WINSIZE; i++) {
		if (i > 0)
			fp = buzh_update(fp, buf[i - 1], buf[WINSIZE + i - 1],
			                 WINSIZE);
		if ((fp & HASHMSK) == 0)
			return i + WINSIZE;
	}
	return size;
}

size_t
comp_size(size_t size)
{
	return LZ4_compressBound(size);
}

size_t
comp(uint8_t *in, uint8_t *out, size_t insize, size_t outsize)
{
	int ret;

	ret = LZ4_compress_default((char *)in, (char *)out, insize, outsize);
	if (ret < 0)
		errx(1, "LZ4_compress_default failed");
	return ret;
}

size_t
decomp(uint8_t *in, uint8_t *out, size_t insize, size_t outsize)
{
	int ret;

	ret = LZ4_decompress_safe((char *)in, (char *)out, insize, outsize);
	if (ret < 0)
		errx(1, "LZ4_decompress_safe failed");
	return ret;
}

void
print_md(FILE *fp, uint8_t *md, size_t size)
{
	size_t i;

	for (i = 0; i < size; i++)
		fprintf(fp, "%02x", md[i]);
}

void
print_stats(struct stats *st)
{
	if (st->nr_blks == 0)
		return;

	fprintf(stderr, "original size: %llu bytes\n",
	        (unsigned long long)st->orig_size);
	fprintf(stderr, "compressed size: %llu bytes\n",
	        (unsigned long long)st->comp_size);
	fprintf(stderr, "deduplicated size: %llu bytes\n",
	        (unsigned long long)st->dedup_size);
	fprintf(stderr, "min/avg/max block size: %llu/%llu/%llu\n",
	        (unsigned long long)st->min_blk_size,
	        (unsigned long long)st->dedup_size / st->nr_blks,
	        (unsigned long long)st->max_blk_size);
	fprintf(stderr, "number of blocks: %llu\n",
	        (unsigned long long)st->nr_blks);
	fprintf(stderr, "cache hits: %llu\n", cache_hits);
	fprintf(stderr, "cache misses: %llu\n", cache_misses);
}

void
str2bin(char *s, uint8_t *d)
{
	size_t i, size = strlen(s) / 2;

	for (i = 0; i < size; i++, s += 2)
		sscanf(s, "%2hhx", &d[i]);
}

off_t
xlseek(int fd, off_t offset, int whence)
{
	off_t ret;

	ret = lseek(fd, offset, whence);
	if (ret < 0)
		err(1, "lseek");
	return ret;
}

ssize_t
xread(int fd, void *buf, size_t nbytes)
{
	uint8_t *bp = buf;
	ssize_t total = 0;

	while (nbytes > 0) {
		ssize_t n;

		n = read(fd, &bp[total], nbytes);
		if (n < 0)
			err(1, "read");
		else if (n == 0)
			return total;
		total += n;
		nbytes -= n;
	}
	return total;
}

ssize_t
xwrite(int fd, const void *buf, size_t nbytes)
{
	const uint8_t *bp = buf;
	ssize_t total = 0;

	while (nbytes > 0) {
		ssize_t n;

		n = write(fd, &bp[total], nbytes);
		if (n < 0)
			err(1, "write");
		else if (n == 0)
			return total;
		total += n;
		nbytes -= n;
	}
	return total;
}

int
cache_entry_cmp(struct cache_entry *e1, struct cache_entry *e2)
{
	int r;

	r = memcmp(e1->blk_desc.md, e2->blk_desc.md, sizeof(e1->blk_desc.md));
	if (r > 0)
		return 1;
	else if (r < 0)
		return -1;
	return 0;
}
RB_PROTOTYPE(cache, cache_entry, e, cache_entry_cmp);
RB_GENERATE(cache, cache_entry, e, cache_entry_cmp);

struct cache_entry *
alloc_cache_entry(void)
{
	struct cache_entry *ent;

	ent = calloc(1, sizeof(*ent));
	if (ent == NULL)
		err(1, "calloc");
	return ent;
}

void
add_cache_entry(struct cache_entry *ent)
{
	RB_INSERT(cache, &cache_head, ent);
}

void
flush_cache(void)
{
	struct cache_entry *ent;

	if (!cache_dirty)
		return;

	xlseek(cfd, 0, SEEK_SET);
	RB_FOREACH(ent, cache, &cache_head)
		xwrite(cfd, &ent->blk_desc, sizeof(ent->blk_desc));
}

void
free_cache(void)
{
	struct cache_entry *ent, *tmp;

	RB_FOREACH_SAFE(ent, cache, &cache_head, tmp) {
		RB_REMOVE(cache, &cache_head, ent);
		free(ent);
	}
}

uint64_t
cache_nr_entries(void)
{
	struct stat sb;

	if (fstat(cfd, &sb) < 0)
		err(1, "fstat");
	return sb.st_size / sizeof(struct blk_desc);
}

void
append_snap(struct snapshot *snap)
{
	/* Update snapshot header */
	snaphdr.nr_snapshots++;
	xlseek(ifd, 0, SEEK_SET);
	xwrite(ifd, &snaphdr, sizeof(snaphdr));

	/* Append snapshot */
	xlseek(ifd, 0, SEEK_END);
	snap->size = sizeof(*snap);
	snap->size += snap->nr_blk_descs * sizeof(snap->blk_desc[0]);
	xwrite(ifd, snap, snap->size);
}

struct snapshot *
alloc_snap(void)
{
	struct snapshot *snap;

	snap = calloc(1, sizeof(*snap));
	if (snap == NULL)
		err(1, "calloc");
	return snap;
}

struct snapshot *
grow_snap(struct snapshot *snap, uint64_t nr_blk_descs)
{
	size_t size;

	size = sizeof(*snap);
	size += nr_blk_descs * sizeof(snap->blk_desc[0]);
	snap = realloc(snap, size);
	if (snap == NULL)
		err(1, "realloc");
	return snap;
}

uint8_t *
alloc_buf(size_t size)
{
	void *p;

	p = calloc(1, size);
	if (p == NULL)
		err(1, "calloc");
	return p;
}

void
hash_blk(uint8_t *buf, size_t size, uint8_t *md)
{
	SHA256_CTX ctx;

	SHA256_Init(&ctx);
	SHA256_Update(&ctx, buf, size);
	SHA256_Final(md, &ctx);
}

void
read_blk(uint8_t *buf, struct blk_desc *blk_desc)
{
	xlseek(sfd, blk_desc->offset, SEEK_SET);
	if (xread(sfd, buf, blk_desc->size) == 0)
		errx(1, "read: unexpected EOF");
}

void
append_blk(uint8_t *buf, struct blk_desc *blk_desc)
{
	xlseek(sfd, snaphdr.store_size, SEEK_SET);
	xwrite(sfd, buf, blk_desc->size);
	snaphdr.store_size += blk_desc->size;
}

int
lookup_blk_desc(uint8_t *md, struct blk_desc *blk_desc)
{
	struct cache_entry *ent, key;

	memcpy(key.blk_desc.md, md, sizeof(key.blk_desc.md));
	ent = RB_FIND(cache, &cache_head, &key);
	if (ent != NULL) {
		*blk_desc = ent->blk_desc;
		return 0;
	}
	return -1;
}

void
dedup(int fd, char *msg)
{
	uint8_t *buf[2];
	struct snapshot *snap;
	SHA256_CTX ctx;
	ssize_t n, bufsize;

	buf[0] = alloc_buf(BLKSIZE);
	buf[1] = alloc_buf(comp_size(BLKSIZE));
	snap = alloc_snap();

	bufsize = 0;
	SHA256_Init(&ctx);
	while ((n = xread(fd, buf[0] + bufsize, BLKSIZE - bufsize)) > 0 ||
	        bufsize > 0) {

		uint8_t md[MDSIZE];
		struct blk_desc blk_desc;
		size_t blksize, csize;
		uint8_t *inp = buf[0]; /* input buf */
		uint8_t *outp = buf[1]; /* compressed buf */

		if (n > 0) {
			bufsize += n;
			snaphdr.st.orig_size += n;
		}

		blksize = chunk_blk(inp, bufsize);
		csize = comp(inp, outp, blksize, comp_size(BLKSIZE));

		snaphdr.st.comp_size += csize;

		hash_blk(outp, csize, md);

		/* Calculate file hash one block at a time */
		SHA256_Update(&ctx, inp, blksize);

		snap = grow_snap(snap, snap->nr_blk_descs + 1);

		if (lookup_blk_desc(md, &blk_desc) < 0) {
			struct cache_entry *ent;

			memcpy(blk_desc.md, md, sizeof(blk_desc.md));
			blk_desc.offset = snaphdr.store_size;
			blk_desc.size = csize;

			snap->blk_desc[snap->nr_blk_descs++] = blk_desc;

			append_blk(outp, &blk_desc);

			ent = alloc_cache_entry();
			ent->blk_desc = blk_desc;
			add_cache_entry(ent);
			cache_dirty = 1;
			cache_misses++;

			snaphdr.st.dedup_size += blk_desc.size;
			snaphdr.st.nr_blks++;

			if (blk_desc.size > snaphdr.st.max_blk_size)
				snaphdr.st.max_blk_size = blk_desc.size;
			if (blk_desc.size < snaphdr.st.min_blk_size)
				snaphdr.st.min_blk_size = blk_desc.size;
		} else {
			snap->blk_desc[snap->nr_blk_descs++] = blk_desc;
			cache_hits++;
		}

		memmove(inp, inp + blksize, bufsize - blksize);
		bufsize -= blksize;
	}

	if (snap->nr_blk_descs > 0) {
		SHA256_Final(snap->md, &ctx);

		if (msg != NULL) {
			size_t size;

			size = strlen(msg) + 1;
			if (size > sizeof(snap->msg))
				size = sizeof(snap->msg);
			memcpy(snap->msg, msg, size);
			snap->msg[size - 1] = '\0';
		}

		append_snap(snap);
	}

	free(snap);
	free(buf[1]);
	free(buf[0]);
}

int
extract(struct snapshot *snap, void *arg)
{
	uint8_t *buf[2];
	struct extract_args *args = arg;
	uint64_t i;

	if (memcmp(snap->md, args->md, sizeof(snap->md)) != 0)
		return WALK_CONTINUE;

	buf[0] = alloc_buf(BLKSIZE);
	buf[1] = alloc_buf(comp_size(BLKSIZE));
	for (i = 0; i < snap->nr_blk_descs; i++) {
		size_t blksize;

		read_blk(buf[1], &snap->blk_desc[i]);
		blksize = decomp(buf[1], buf[0], snap->blk_desc[i].size, BLKSIZE);
		xwrite(args->fd, buf[0], blksize);
	}
	free(buf[1]);
	free(buf[0]);
	return WALK_STOP;
}

int
check(struct snapshot *snap, void *arg)
{
	uint8_t md[MDSIZE];
	uint8_t *buf;
	SHA256_CTX ctx;
	uint64_t i;

	buf = alloc_buf(comp_size(BLKSIZE));
	/*
	 * Calculate hash for each block and compare
	 * against snapshot entry block descriptor
	 */
	for (i = 0; i < snap->nr_blk_descs; i++) {
		read_blk(buf, &snap->blk_desc[i]);

		SHA256_Init(&ctx);
		SHA256_Update(&ctx, buf, snap->blk_desc[i].size);
		SHA256_Final(md, &ctx);

		if (memcmp(snap->blk_desc[i].md, md,
		           sizeof(snap->blk_desc[i]).md) == 0)
			continue;

		fprintf(stderr, "Block hash mismatch\n");
		fprintf(stderr, "  Expected hash: ");
		print_md(stderr, snap->md, sizeof(snap->md));
		fputc('\n', stderr);
		fprintf(stderr, "  Actual hash: ");
		print_md(stderr, md, sizeof(md));
		fputc('\n', stderr);
		fprintf(stderr, "  Offset: %llu\n",
		        (unsigned long long)snap->blk_desc[i].offset);
		fprintf(stderr, "  Size: %llu\n",
		        (unsigned long long)snap->blk_desc[i].size);
	}
	free(buf);
	return WALK_CONTINUE;
}

int
list(struct snapshot *snap, void *arg)
{
	print_md(stdout, snap->md, sizeof(snap->md));
	if (snap->msg[0] != '\0')
		printf("\t%s\n", snap->msg);
	else
		putchar('\n');
	return WALK_CONTINUE;
}

int
rebuild_cache(struct snapshot *snap, void *arg)
{
	uint8_t md[MDSIZE];
	uint8_t *buf;
	SHA256_CTX ctx;
	uint64_t i;

	buf = alloc_buf(comp_size(BLKSIZE));
	for (i = 0; i < snap->nr_blk_descs; i++) {
		struct cache_entry *ent;

		read_blk(buf, &snap->blk_desc[i]);

		SHA256_Init(&ctx);
		SHA256_Update(&ctx, buf, snap->blk_desc[i].size);
		SHA256_Final(md, &ctx);

		ent = alloc_cache_entry();
		memcpy(ent->blk_desc.md, md, sizeof(ent->blk_desc.md));
		ent->blk_desc = snap->blk_desc[i];
		add_cache_entry(ent);
		cache_dirty = 1;
	}
	free(buf);
	return WALK_CONTINUE;
}

/* Walk through all snapshots and call fn() on each one */
void
walk(int (*fn)(struct snapshot *, void *), void *arg)
{
	struct snapshot *snap;
	uint64_t i;

	snap = alloc_snap();
	xlseek(ifd, sizeof(snaphdr), SEEK_SET);
	for (i = 0; i < snaphdr.nr_snapshots; i++) {
		if (xread(ifd, snap, sizeof(*snap)) == 0)
			errx(1, "read: unexpected EOF");

		snap = grow_snap(snap, snap->nr_blk_descs);
		if (xread(ifd, snap->blk_desc,
		          snap->nr_blk_descs * sizeof(snap->blk_desc[0])) == 0)
			errx(1, "read: unexpected EOF");

		if ((*fn)(snap, arg) == WALK_STOP)
			break;
	}
	free(snap);
}

void
init_cache(void)
{
	uint64_t nents, i;

	nents = cache_nr_entries();
	xlseek(cfd, 0, SEEK_SET);
	for (i = 0; i < nents; i++) {
		struct cache_entry *ent;

		ent = alloc_cache_entry();
		if (xread(cfd, &ent->blk_desc, sizeof(ent->blk_desc)) == 0)
			errx(1, "read: unexpected EOF");
		add_cache_entry(ent);
	}
}

void
init(void)
{
	struct stat sb;

	ifd = open(SNAPSF, O_RDWR | O_CREAT, 0600);
	if (ifd < 0)
		err(1, "open %s", SNAPSF);

	sfd = open(STOREF, O_RDWR | O_CREAT, 0600);
	if (sfd < 0)
		err(1, "open %s", STOREF);

	cfd = open(CACHEF, O_RDWR | O_CREAT, 0600);
	if (cfd < 0)
		err(1, "open %s", CACHEF);

	if (flock(ifd, LOCK_NB | LOCK_EX) < 0 ||
	    flock(sfd, LOCK_NB | LOCK_EX) < 0 ||
	    flock(cfd, LOCK_NB | LOCK_EX) < 0)
		errx(1, "busy lock");

	if (fstat(ifd, &sb) < 0)
		err(1, "fstat %s", SNAPSF);
	if (sb.st_size != 0) {
		uint8_t maj, min;

		xread(ifd, &snaphdr, sizeof(snaphdr));
		min = snaphdr.flags & 0xff;
		maj = (snaphdr.flags >> 8) & 0xff;

		if (maj != VER_MAJ || min != VER_MIN)
			errx(1, "expected snapshot format version %u.%u but got %u.%u",
			     VER_MAJ, VER_MIN, maj, min);
	} else {
		snaphdr.flags = (VER_MAJ << 8) | VER_MIN;
		xwrite(ifd, &snaphdr, sizeof(snaphdr));
		snaphdr.st.min_blk_size = comp_size(BLKSIZE);
	}

	if (cache_nr_entries() != 0)
		init_cache();
	else
		walk(rebuild_cache, NULL);
}

void
term(void)
{
	if (verbose)
		print_stats(&snaphdr.st);
	flush_cache();
	free_cache();

	fsync(ifd);
	fsync(sfd);
	fsync(cfd);

	close(ifd);
	close(sfd);
	close(cfd);
}

void
usage(void)
{
	fprintf(stderr, "usage: %s [-clv] [-e id] [-r root] [-m message] [file]\n", argv0);
	exit(1);
}

int
main(int argc, char *argv[])
{
	uint8_t md[MDSIZE];
	char *id = NULL, *root = NULL, *msg = NULL;
	int fd = -1, lflag = 0, cflag = 0;

	ARGBEGIN {
	case 'c':
		cflag = 1;
		break;
	case 'e':
		id = EARGF(usage());
		break;
	case 'l':
		lflag = 1;
		break;
	case 'r':
		root = EARGF(usage());
		break;
	case 'm':
		msg = EARGF(usage());
		break;
	case 'v':
		verbose = 1;
		break;
	default:
		usage();
	} ARGEND

	if (argc > 1) {
		usage();
	} else if (argc == 1) {
		if (id) {
			fd = open(argv[0], O_RDWR | O_CREAT, 0600);
			if (fd < 0)
				err(1, "open %s", argv[0]);
		} else {
			fd = open(argv[0], O_RDONLY);
			if (fd < 0)
				err(1, "open %s", argv[0]);
		}
	} else {
		if (id)
			fd = STDOUT_FILENO;
		else
			fd = STDIN_FILENO;
	}

	if (root != NULL) {
		mkdir(root, 0700);
		if (chdir(root) < 0)
			err(1, "chdir: %s", root);
	}

	init();

	if (cflag) {
		walk(check, NULL);
		term();
		return 0;
	}

	if (lflag) {
		walk(list, NULL);
		term();
		return 0;
	}

	if (id) {
		str2bin(id, md);
		walk(extract, &(struct extract_args){ .md = md, .fd = fd });
	} else {
		dedup(fd, msg);
	}

	term();
	return 0;
}
