
;FUNC:    $MUL_QD
;
;DESC:    Multiplies an unsigned quadword by an unsigned dword, returning an
;    unsigned quadword result.
;
;IN: DS:DI          address of unsigned quadword ("qword")
;    CX;BX          unsigned dword multiplier ("dword")
;
;OUT:     "qword"        unsigned quadword result ("qword"*"dword")
;    AX        destroyed
;    BX        destroyed
;    CX        destroyed
;    DX        destroyed
;    SI        destroyed
;    (DS:)DI        address of last word of result (in "qword")
;
;NOTES:   The unsigned quadword value in "qword" is multiplied by the unsigned
;    dword value in CX;BX.  The unsigned quadword result is returned in
;    "qword".
;
;    The result is undefined if arithmetic overflow occurs.
;
;ALG:     The algorithm used by this function is an extension of the
;    algorithm used for MUL_QW.  In this function, two quadword*word
;    multiplications are performed in tandem as each word of the quadword
;    is processed--one for the low word of the multiplier (mul1) and one
;    for the high word (mul2).  As each mul1 multiplication is performed,
;    the high word of the previous mul1 is added into the low word of the
;    current mul1.  The same is true for each mul2.  Then this additive
;    mul1 result is added into the similar result for the *prior* mul2.
;    This is essentially just long multiplication performed in columns
;    rather than rows.
;
;    This algorithm is complicated mostly because three additions must
;    be performed to generate each word of the result, so three carrys
;    must be maintained simultaneously.  These are held in CH:
;    bit 7 = CF from adding two mul1 words together
;    bit 1 = CF from adding two mul2 words together
;    bit 0 = CF from adding the above two words together
;
;    The positions of the instructions in this function have been
;    optimized to improve throughput since fast, 2-byte instructions
;    (ROL, SHR, ADC, etc.) become slow on 8088's if they are adjacent to
;    each other.  Where possible, slower instructions have been
;    interposed to minimize this effect.
;
;    MUL_QQ depends on the OUT conditions listed above.

mul_qd    proc    near

_temp          equ  <word ptr [bp-6]>
_cx       equ  <word ptr [bp-4]>
_bx       equ  <word ptr [bp-2]>
_bp       equ  <word ptr [bp]>

          push ax bx cx dx si di
          push bp
          mov  bp,sp          ;set up stack frame
          push bx        ;_BX = original BX
          push cx        ;_CX = original CX
          xor  bx,bx          ;BX = result of mul2 for this word (0)
          push bx        ;_TEMP = high word from prior mul2 (0)
          xor  si,si          ;SI = high word from prior mul1 (0)
          mov  cx,4      ;CH = prev CF (0), CL = loop count (4)
$mul_qd_010:   mov  ax,[di]        ;get next word from quadword
          mul  _bx       ;mul next word by mul1 word
          shl  ch,1      ;get CF from last ADD (bit 7)
          adc  ax,si          ;add in high word from prior mul1
          rcr  ch,1      ;save CF from this ADD (bit 7)
          mov  si,dx          ;SI saves high word from this mul1
          shr  ch,1      ;get CF from last ADD (bit 0)
          adc  ax,bx          ;add prior comb word into result w/CF
          xchg [di],ax        ;store result low word, AX = qwrd word
          rcl  ch,1      ;save CF from this ADD (bit 0)
          dec  cl        ;all 4 words of result generated?
           jz  $mul_qd_090    ;  y: done (skip last MUL/ADC)
          mov  bx,_temp  ;move prior high mul2 into comb word
          mul  _cx       ;mul this word by mul2 word
          ror  ch,1      ;prepare to access bit 1
          mov  _temp,dx  ;save new high word of mul2
          shr  ch,1      ;get CF from last ADD (bit 1)
          adc  bx,ax          ;combine last high mul2 & this lo mul2
          inc  di
          rcl  ch,1      ;save CF from this ADD (bit 1)
          inc  di        ;DI points to next higher word
          rol  ch,1      ;restore bits 7,1,0 to their positions
          jmp  $mul_qd_010
$mul_qd_090:   mov  sp,bp          ;deallocate _BX,_CX,_TEMP (restore SP)
          pop  bp
          pop  di si dx cx bx ax
          ret
mul_qd    endp



;FUNC:    DIV_QD
;
;DESC:    Divides an unsigned quadword by an unsigned dword, returning an
;    unsigned quadword result.
;
;IN: DS:DI          address of unsigned quadword dividend ("qword")
;    CX;BX          unsigned dword divisor
;
;OUT:     "qword"        unsigned quadword quotient
;    DX;AX          unsigned dword remainder
;

 div_qd   proc near

 push     bx cx si di bp
;-----------------------------------------------------------------------------
; Set up registers and memory locations for the divide.
; Most commonly-accessed words are in registers wherever possible:
;
; DX;AX = Remainder (remainder of prior divide + high bit from dividend)
; BP;BX = Divisor
; [DI+4];[DI+2];[DI];SI = Dividend, also accumulates quotient
; CX = Looping count (number of bits still to be processed)
;
; Initially, SI and AX are switched so leading zeros can be skipped 8 bits at
; a time (SI can't be used for this since it is strictly a 16 bit register).
;-----------------------------------------------------------------------------

          mov  bp,cx          ;BP;BX holds divisor
          xor  dx,dx          ;DX;AX holds remainder of prior divide
          xor  ax,ax          ;(init remainder to zero)
          mov  si,64          ;CX = bits remaining (use SI for now)
          mov  cx,[di]        ;[DI+4];[DI+2];[DI];SI (CX for now)
          inc  di        ;  hold dividend (in high bits) and
          inc  di        ;  quotient (in low bits)

;-----------------------------------------------------------------------------
; Speed it up by skipping 8 bits at a time until bitwise division must be done
;-----------------------------------------------------------------------------

div_qd_010:    test dh,dh          ;is there room to shift left 8 bits?
           jnz div_qd_030     ;  n: divide the rest by bits
          mov  dh,dl          ;try shifting 8 bits at once,
          mov  dl,ah          ;  instead of one at a time
          mov  ah,al          ;(this can be done until the partial
          mov  al,[di+5] ;  remainder > divisor)
          cmp  dx,bp          ;would this shift go too far?
           jb  div_qd_020     ;  n: complete the shift
           ja  div_qd_025     ;  y: don't shift these 8 bits
          cmp  ax,bx          ;(maybe)
           jae div_qd_025     ;  y: don't shift these 8 bits
div_qd_020:    xchg ch,cl          ;can shift 8 bits at once
          xchg [di],cl        ;shift rest of remainder;dividend
          xchg [di+1],cl ;  to the left 8 bits
          xchg [di+2],cl
          xchg [di+3],cl
          xchg [di+4],cl
          mov  [di+5],cl
          xor  cl,cl
          sub  si,8      ;now there are 8 fewer bits to divide
           jnz div_qd_010     ;  more remain, try to skip 8 more
          jmp  short div_qd_090;  done, return quot=0, rem=0
div_qd_025:    mov  al,ah
          mov  ah,dl          ;can't skip the last 8 bits
          mov  dl,dh          ;undo the exchange done at div_qd_010
          xor  dh,dh          ;  when seeing if its could be skipped
div_qd_030:    xchg cx,si          ;fix CX=loop cnt, SI=low dividend word

;-----------------------------------------------------------------------------
; Do long division.  Dividend is shifted left, one bit at a time, into
; the remainder registers (DX;AX).  Quotient accumulates into the low bits
; of the dividend locations.  If divisor can divide into accumulated
; remainder then subtract divisor from it once and place a 1 in the lowest
; bit of the quotient, else put a zero there.  (Only 1's and 0's are allowed
; in quotient since this is *binary* long division.)
;-----------------------------------------------------------------------------

div_qd_040:    shl  si,1      ;shift dividend left one bit
          rcl  word ptr [di],1     ;  (makes room for new quotient bit
          rcl  word ptr [di+2],1; at bottom, shifts high bit out
          rcl  word ptr [di+4],1; the top for appending to remainder)
          rcl  ax,1      ;append high dividend bit onto
          rcl  dx,1      ;  low end of remainder
           jc  div_qd_050     ;(if CF set, rem is > divisor)
          cmp  dx,bp          ;is remainder bigger than divisor?
           jb  div_qd_060     ;  n: rem smaller, shift & try again
           ja  div_qd_050     ;  y: sub divisor once & set quotient
          cmp  ax,bx          ;high words same, is rem same/bigger?
           jb  div_qd_060     ;  n: rem smaller, shift & try again
div_qd_050:    sub  ax,bx          ;rem >= divisor, can divide 1x,
          sbb  dx,bp          ;  subtract divisor from remainder
          inc  si        ;set bit in quotient indicating divide
div_qd_060:     loop     div_qd_040     ;continue until all bits are done

;-----------------------------------------------------------------------------
; Done, set up return values and exit
;-----------------------------------------------------------------------------

div_qd_090:    mov  [di-2],si ;return low word of quotient
          pop  bp di si cx bx ;return remainder in DX;AX
          ret
div_qd  endp


;NAME:    SPLIT_PATH
;
;DESC:    Returns a path string from a pathname.
;
;IN: DS:SI          address of an ASCIIZ pathname ("pathname")
;    ES:DI          address of the destination buffer ("buffer")
;
;OUT:     "buffer"  contains an ASCIIZ path string
;
 split_path     proc near
		push	si
          call     str_path  ;get path string pointer and length
          rep  movsb
          mov  [di],cl     ;NULL terminate the resulting string
		pop	si

          ret
 split_path endp


;NAME:    FIX_PATH
;
;DESC:    Converts a path string from internal format to DOS format.
;
;IN: DS:SI          address of an ASCIIZ path string ("path")
;
;OUT:     JE if a change was made to "path".
;      "path"  copy of "path" without an extraneous terminal
;              backslash character (`\`)
;    JNE if "path" was already in DOS format.
;
 fix_path  proc near
          push     si
          call     str_len        ;get length of path string
          cmp  cx,3      ;is it less than 3 chars long?
           jb  fix_path_040   ;  y - JNE return with no change
          add  si,cx
          dec  si        ;  n - point to last string char
          cmp  byte ptr [si],'\' ;terminal backslash?
           jne fix_path_040   ;      n - JNE return with no change
          cmp  byte ptr [si-1],':' ;end pair ":\"?
           je  fix_path_060   ;      y - JNE return with no change
          mov  byte ptr [si],0     ;      n - remove terminal backslash
          cmp  cx,cx          ;force JE return, removed backslash
fix_path_040:  pop si
          ret
fix_path_060:  test sp,sp          ;force JNE return
          jmp  fix_path_040
 fix_path	endp


;FUNC:    STR_PATH
;
;DESC:    Locates a path substring in a string
;
;IN: DS:SI          address of the string ("string")
;
;OUT:     (DS:)SI        address of the path substring in "string"
;    CX        length of the path substring
;
;ALG:     Sets a pointer past a ':' if one exists.  This is the begininng of
;    a path substring.  Scans for the last '\'.   This is the end of a
;    path substring.  The length is the difference between the two.

 str_path  proc near
          push ax
          mov  al,':'
          call     str_chr        ;is there a ':' character?
           jc  str_ext_020    ;  n: leave SI at start of "string"
          inc  si        ;  y: SI points past ':'
str_ext_020:   mov  cx,si          ;set CX to start of the path
          mov  al,'\'         ;set search char
          call     str_rchr  ;reverse search for a '\'
           jc  str_path_100   ;no path element if not found
          xchg cx,si          ;restore start of path pointer
          sub  cx,si          ;calc substring length and
          inc  cx        ;  adjust length to include '\'
str_path_080:  pop  ax
          ret

str_path_100:  xor  cx,cx          ;set CX = 0
          jmp  str_path_080   ;exit, no path found

 str_path endp

;FUNC:    STR_CHR
;
;DESC:    Searches a string for the first occurrence of a character
;
;IN: DS:SI          address of the string to search ("string")
;    AL        the character to search for ("char")
;
;OUT:     JNC if "char" is in "string"
;      (DS:)SI address of the first occurrence of "char" in "string"
;    JC if "char" is not in "string"
;

 str_chr proc near
          push bx
          mov  bx,si          ;save begining of string ptr
          dec  si        ;setup to start with an INC
str_chr_010:   inc  si        ;point to the next char
          cmp  [si],al        ;is char equal to user char?
           je  str_chr_080    ;  y: match, JNC return
          cmp  byte ptr [si],1     ;found terminal NULL?
           jae str_chr_010    ;  n: keep looking for match
          mov  si,bx          ;restore begining of string ptr
str_chr_080:   pop  bx
          ret
 str_chr endp


;FUNC:    STR_LEN
;
;DESC:    Returns the length of a string
;
;IN: DS:SI          address of the string ("string")
;
;OUT:     CX        length of "string"
;
 str_len  proc near
		push	di
		push	si
		push	ax
          mov  di,si          ;setup DI for scas
          mov  cx,-1          ;init counter to 65535
          xor  al,al          ;setup to scan for a NULL
          repne     scasb          ;scan for it
          not  cx        ;fix the counter
          dec  cx        ;don't count the NULL
		pop	ax
		pop	si
		pop	di
          ret
 str_len endp

;FUNC:    STR_RCHR
;
;DESC:    Searches a string for the last occurrence of a character
;
;IN: DS:SI          address of the string to search ("string")
;    AL        the character to search for ("char")
;
;OUT:     JNC if "char" is in "string"
;      (DS:)SI address of the last occurrence of "char" in "string"
;    JC if "char" is not in "string"
;


 str_rchr  proc near
          push bx
          mov  bx,si          ;SI = BX->beginning of "string"
          call     str_end        ;SI->NULL of "string"
          xchg bx,si          ;BX->NULL, SI->beginning
str_rchr_010:  cmp  bx,1      ;first byte in segment?
           jc  str_rchr_080   ;  y: "char" not in "string", JC return
          dec  bx        ;BX->previous char
          cmp  bx,si          ;BX past beginning of "string"?
           jc  str_rchr_080   ;  y: JC return
          cmp  al,[bx]        ;does this char match "char"?
           jne str_rchr_010   ;  n: go check next one
          mov  si,bx          ;  y: SI->matching char
str_rchr_080:  pop  bx
          ret
 str_rchr endp

;FUNC:    STR_END
;
;DESC:    Returns the address of the end of a string.
;
;IN: DS:SI          address of the string ("string")
;
;OUT:     (DS:)SI        address of the terminating NULL in "string"
;
 str_end        proc near
          push     ax
		push	    cx
          xchg di,si          ;set DI as string offset
          mov  cx,-1          ;search up to 65535 chars
          xor  al,al          ;look for the NULL
          repne     scasb          ;DI points past the NULL
          dec  di        ;fix the overrun
          xchg di,si          ;restore DI
          pop cx
		pop	ax
          ret
 str_end   endp

;----------------------------------------------------------------------------
; Parses a parameter for a file filter (e.g. *.asm) and stores it in the
; file_mask location if it is found.
;   On entry: SI points to start of path/filename
;   Returns:  Nothing
;   Modifies: file_mask, DI, AX
;----------------------------------------------------------------------------

 PARSE_FILE_FILTER PROC NEAR

	  push	si
	  push	di
       MOV     BX,SI                   ;save start of source
	  push	cx                      ;this section here is for the purpose of
	  push	bx                      ; evaluating and storing a file filter if
							    ; one was input
	  mov	di,offset tmp_buff	    ;copy parameter to buffer
;the following two lines added DF4 pass 5.  (and fixed in pass 8).
;Keeps things frpm getting messed up when a .. is the command line parameter.  It's a bit of a kludge and this
;whole routine could also be better.  Something like c:\*.asm parses as c:*.asm
;for example.  I think I'm just going to leave it for now.
          cmp   [si],02E2Eh		; 2E2E = ..
          je            _PFF_8
 _PFF_05:
	  push	di
	  call	str_len                 ;get its length
	  pop	di
	  push	cx              	    ;save length
	  rep	movsb
	  mov	byte ptr [di],0
	  pop	cx
	  mov	bx,di
	  sub 	bx,cx			   ;move bx to beginning of string
	  dec	bx				   ;lines added DF4 pass 8 -- think we went to
							   ;backwards search w/o changing bx location
 _PFF_1:
	  dec	di
	  cmp	di,bx			    ;are we at the beginning of the string?
	  je		_PFF_8                  ;yes? there's no filename
	  cmp	byte ptr [di],'\'	    ;is parameter terminated by a path delimiter?
							    ;or was no wildcard found before one?
	  je		_PFF_8			    ;yes? There's no filename
	  cmp	byte ptr [di],':'	    ;ditto for drive specifier
	  je		_PFF_8
	  cmp	byte ptr [di],'?'
	  je		_PFF_2			    ;we have a filespec -- process
	  cmp	byte ptr [di],'*'
	  je		_PFF_2			    ;here too
	  jmp	_PFF_1			    ;nothing yet, loop
 _PFF_2:						    ;there appears to be a filespec, process
	  dec	di				    ;backtrack to the position before
	  cmp	di,offset tmp_buff      ;the filename begins (\ or beginning
							    ;of path
	  jl		_PFF_3
	  cmp	byte ptr [di],'\'
	  je		_PFF_3
	  cmp	byte ptr [di],':'	    ;or a drive designator for that matter
	  je	     _PFF_3
	  jmp	_PFF_2
 _PFF_3:
	  mov	si,di
	  mov	di,offset file_mask	;PREPARE TO COPY TO FILE MASK
	  push	di
	  mov	cx,13				;we clean out the file_mask area
	  mov	al,0
	  rep	stosb
	  pop	di                       ;then restore the original pointer
	  inc	si
	  xor	cx,cx			     ;keep track of length for use
								;in next atep
 _PFF_4:
	  inc 	cx
	  mov	al,byte ptr [si]
	  cmp	al,0			          ;copy until we hit null at end of
	  je		_PFF_5                   ;path/filename
	  call	to_upper				;capitalize (for later display)
	  mov	byte ptr [si],al
	  movsb
	  jmp	_PFF_4
 _PFF_5:
	  pop	bx					;get beginning of parameter in command
	  push	bx
	  mov	di,bx				;line back
 _PFF_6:
	  cmp	byte ptr [di],0
	  je		_PFF_7
	  inc	di
	  jmp	_PFF_6
 _PFF_7:
;	  dec	cx
	  sub	di,cx
	  cmp	byte ptr [di],'\'		;if there's a trailing '\' after the
								;parsing off of the widcard filter,
	  je		_PFF_75                  ;we need to strip it off since DOS
	  inc	di                       ;doesn't like it
	  dec	cx

 _PFF_75:
	  mov	al,0
	  rep	stosb

 _PFF_8:
	  pop	bx
	  pop	cx
	  pop	di
	  pop	si
	  ret

 PARSE_FILE_FILTER ENDP

;FUNC:    TO_UPPER
;
;DESC:    Converts a lowercase ASCII letter (a-z) to uppercase (A-Z).
;
;IN: AL        character to convert ("char")
;
;OUT:     AL        character, converted to uppercase only if it was
;              originally lowercase
;

 to_upper  proc near
          cmp  al,"a"         ;below lowercase letters?
           jb  to_upper_080   ;  y: return w/no conversion
          cmp  al,"z"         ;above lowercase letters?
           ja  to_upper_080   ;  y: return w/no conversion
          and  al,0DFh        ;  n: convert to uppercase
to_upper_080:  ret

 to_upper endp

;-----------------------------------------------------------------------------
; On entry: al contains unpacked BCD number
; On exit:  al contains ASCII number
; Modifies: al only.  ah preserved.
;-----------------------------------------------------------------------------

 unbcd    proc near
     push      bx dx cx
     xor       bx,bx
     xor       dx,dx
     mov       ch,ah
     mov       bl,al
     and       bl,0Fh
     mov       cl,4
     shr       al,cl
     mov       dl,10
     mul       bl
     add       al,bl
     mov       ah,ch
     pop       cx dx bx
     ret
 unbcd    endp

;FUNC:    WORD_TO_ASCE
;
;DESC:    Converts an unsigned dword to an ASCII string (given a radix)
;
;IN: AX        unsigned word ("word")
;    BL        radix of resulting ASCII string, 2-36 ("radix")
;    DS:SI          address to store resulting string ("outstr")
;
;OUT:     "outstr"  ASCII digits generated from "word"
;    (DS:)SI        points past last digit written out
;

 word_to_asce       proc near

          push bx ax cx        ;setup for Euclid's algorithm
          xor  cx,cx          ;set count to zero
          cmp  bl,2      ;is radix less than 2?
           jb  w_to_a_800     ;  y: exit, n: cont. with routine
w_to_a_020:    call     div_wb         ;div unsigned word by radix
          push bx        ;keep remainder
          inc  cx        ;count number of remainders on stack
          cmp  ax,0h          ;is quotient zero?
           jne w_to_a_020     ;  n: loop again
w_to_a_030:    pop  bx        ;get next remainder
          mov  al,bh          ;put remainder in AL
          add  al,"0"         ;convert to decimal
          cmp  al,"9"         ;convert AL for A to Z?
           jna w_to_a_050     ;  n: skip alpha conversion
          add  al,7h          ;  y: convert to alpha
w_to_a_050:    mov  [si],al        ;write digit to end of string
          inc  si
           loop     w_to_a_030     ;if count=0, stop; else loop again
w_to_a_800:    pop cx ax bx
          ret
word_to_asce  endp

;FUNC:    DIV_WB
;
;DESC:    Divides an unsigned word by an unsigned byte, returning an
;    unsigned word result.
;
;IN: AX        unsigned word dividend
;    BL        unsigned byte divisor
;
;OUT:     AX        unsigned word quotient
;    BH        unsigned byte remainder
;

div_wb         proc near
          or   bl,bl          ;trying to divide by zero?
          jz  div_wb_090     ;  y: can't do it, quit
          push dx
          xor  dx,dx          ;perform word divide
          xor  bh,bh          ;  to avoid divide overflow
          div  bx        ;return quotient in AX
          mov  bh,dl          ;return remainder in BH
          pop  dx
div_wb_090:    ret
div_wb    endp


;FUNC:    DIV_DW
;
;DESC:    Divides an unsigned dword by an unsigned word, returning an
;    unsigned dword result.
;
;IN: DX;AX          unsigned dword dividend
;    BX        unsigned word divisor
;
;OUT:     DX;AX          unsigned dword quotient
;    CX        unsigned word remainder
;
div_dw         proc near
          or   bx,bx          ;trying to divide by zero?
           jz  div_dw_090     ;  y: can't do it, quit
          xor  cx,cx
          cmp  dx,bx          ;is a divide overflow possible?
           jb  div_dw_070     ;  n: only need to divide once
          xchg ax,cx          ;DX;AX = 0;high word
          xchg ax,dx          ;CX = low word
          div  bx        ;divide 0;high word by divisor
          xchg ax,cx          ;DX;AX = rem of high divide;low word
div_dw_070:    div  bx        ;CX = high word of quotient
          xchg dx,cx          ;return DX;AX = quotient, CX = rem
div_dw_090:    ret
div_dw    endp


;Following functions added to DF 5.00 alpha 1 sources
;added to 461A for new sysinfo routines
;==============================================================================
;FUNC:	STR_REV
;
;DESC:	Reverses the order of the characters in a string
;
;IN:	DS:SI		address of the string ("string")
;
;OUT:	"string"	characters are reversed
;
;==============================================================================

str_rev  proc     near
          push     ax di si es
		push	ds
		pop	es		;setup ES to string's segment
		mov	di,si		;setup DI to point to string
          call str_enddi ;point DI to string's NULL
		dec	di		;make DI point to the last char
		cmp	si,di		;if a NULL or single char string
		 jnb	str_rev_080	;  nothing to reverse, so exit
		xchg	si,di		;SI -> last char, DI -> first char
str_rev_020:	mov	al,[di]		;get front char
		xchg	al,[si]		;put it in back spot, back char in AL
		stosb			;put back char in front spot, INC DI
		dec	si		;move SI back one char
		cmp	si,di		;compare the front and back pointers
		 ja	str_rev_020	;if not same or crossed yet, do next
str_rev_080:   pop es si di ax
		ret
str_rev endp

;==============================================================================
;FUNC:	STR_ENDDI
;
;DESC:	Returns the address of the end of a destination string.
;
;IN:	ES:DI		address of the destination string ("string")
;
;OUT:	(ES:)DI		address of the terminating NULL in "string"
;
;==============================================================================

str_enddi proc near
          push     ax cx
		mov	cx,-1		;search up to 65535 chars
		xor	al,al		;look for the NULL
		repne	scasb		;DI points past the NULL
		dec	di		;fix the overrun
          pop cx ax
		ret
str_enddi endp

;==============================================================================
;FUNC:	STR_CHRN
;
;DESC:	Searches a string for the first occurrence of a character which
;	is NOT the given character.
;
;IN:	DS:SI		address of the string to search ("string")
;	AL		the character to compare against ("char")
;
;OUT:	JNC if a character different than "char" was found in "string"
;	  (DS:)SI	address of the first character in "string" that
;			is not the same as "char".
;	JC if all characters in "string" match "char"
;
;==============================================================================

str_chrn  proc near
		push	bx
		mov	bx,si		;save begining of string ptr
		dec	si		;setup to start with an INC
str_chrn_010:	inc	si		;point to the next char
		cmp	byte ptr [si],1	;at end of string?
		 jb	str_chrn_090	;  y: JC return
		cmp	[si],al		;else compare char to user char
		 je	str_chrn_010	;if match, check next char
		clc			;else JNC return
str_chrn_080:	pop	bx
		ret
str_chrn_090:	mov	si,bx		;restore beg of string ptr
		jmp	str_chrn_080	;(carry flag is still set)
str_chrn endp
