// process.c -- Finger daemon processors for special cases
// Copyright 1995 Dean Troyer (troyer@indirect.com)
// All rights reserved.
//
// See fingerd.txt for license information
//
// Here's where the 'action' of the finger server is implemented
//
// This module is compiled with Unicode turned on to support
// the Win32 APIs that require Unicode
//
// 0.4  02Feb95     dlt     Initial release
// 0.5  05Mar95     dlt     Add 'proc' handler; cleanup
//      27Jul95     Anthon V. Lobastoff (tony@xeopsa.nstu.nsk.su)
//                    Modified functions: SendFingerAll
//                    Output for all active sessions added

#define UNICODE
#define _UNICODE
#include <windows.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <time.h>
#include <winperf.h>
#include "fingerd.h"
#include "util.h"
#include "perf.h"

#include <lm.h>

#define BUFSIZE 512

// Error reporting messages
typedef struct {                    // associates an error code with text
   UINT err;
   char *sztext;
} ERRENTRY;

ERRENTRY WSErrors[] =               // error text for windows sockets errors
{
   WSAVERNOTSUPPORTED,  "This version of Windows Sockets is not supported",
   WSASYSNOTREADY,      "Windows Sockets is not present or is not responding",
};


// Global Data
char LMDomain[DNLEN+2];
extern DWORD UserInfo;                  // Enable for user information
extern DWORD ProcInfo;                  // Enable for process information

extern DWORD dwDebug;

// Misc macros
#define dim(x) (sizeof(x) / sizeof(x[0]))


// SendID - Sends the service ID information
//
// Parameters:
//   skOut  - an open socket
//
// Return value:
//   none

void SendID(SOCKET skOut)
{
//    TCHAR Buf[BUFSIZE];

//    gethostname(Buf, 99);
//    send(*skOutput, Buf, strlen(Buf), 0);
    
    SendLine(skOut, "%s %s - %s\n%s\n", INTERNAL_NAME, VERSION, DISPLAY_NAME,
             COPYRIGHT);
    SendLine(skOut, "%s values:\n", Finger_Reg_Key);
    SendLine(skOut, "%s: %s\n", Reg_LMDomain, LMDomain);
    SendLine(skOut, "%s: 0x%x\n", Reg_UserInfo, UserInfo);
    SendLine(skOut, "%s: 0x%x\n", Reg_ProcInfo, ProcInfo);
}  // SendID


// SendFinger - Sends the information about the requested userid
//
// Parameters:
//   skOut  - an open socket
//   UserID - the userid to query & respond
//
// Return value:
//   none

void SendFinger(SOCKET skOut, LPSTR UserID)
{
    NET_API_STATUS dwRetCode;
    LPUSER_INFO_3 lpInfo;
    WCHAR User[UNLEN+1];            // User name
    WCHAR Dom[DNLEN+1];             // Domain name
    LPWSTR lpDom;
    struct tm *t;
    LPSTR p;
    char Buf[BUFSIZE];
        
    if (p = strchr(UserID, '\\')) {
        *p = '\0';                  // Split domain & username
        p++;
        ANSIToUnicode(UserID, Dom, sizeof(Dom));
        /* Try to find a DC for the domain */
        if ((dwRetCode = NetGetDCName(NULL, (LPWSTR)&Dom, (LPBYTE*)&lpDom)) 
             != NERR_Success) {
            /* Can't find a domain controller, fail */
            SendLineW(skOut, 
                TEXT("Error: can't find domain controller for %s\n"), 
                lpDom);
            return;
        }
        ANSIToUnicode(p, (LPWSTR)User, sizeof(User));
    } else {
        // Determine domain/machine name & convert to Unicode
        if (strlen(LMDomain)) {
            ANSIToUnicode(LMDomain, (LPWSTR)Dom, sizeof(Dom));
            if (LMDomain[0] == '\\' && LMDomain[1] == '\\') { 
                /* Assume it's a machine name */
                lpDom = (LPWSTR)&Dom;
            } else {
                /* Try to find a DC for the domain */
                if ((dwRetCode = NetGetDCName(NULL, (LPWSTR)&Dom, (LPBYTE*)&lpDom)) 
                     != NERR_Success) {
                    /* Can't find a domain controller, fail */
                    SendLineW(skOut, 
                        TEXT("Error: can't find domain controller for %s\n"), 
                        lpDom);
                    return;
                }
            }
        } else {
            char Buf[32];

            // Default to the local machine
            gethostname(Buf, 32);
            ANSIToUnicode(Buf, (LPTSTR)Dom, sizeof(Dom));
            /* No domain or machine specified */
            lpDom = NULL;
        }
        ANSIToUnicode(UserID, (LPTSTR)User, sizeof(User));
    }
    
    if ((dwRetCode = NetUserGetInfo(lpDom,      /* Domain */
            User,                               /* Username */
            3,                                  // Level
            (LPBYTE *) &lpInfo))                // Return buffer
             == NERR_Success) {
        SendLineW(skOut, 
            TEXT("Login name: %ws\\%ws\t\t\tIn real life: %ws\nDirectory: %ws\n"), 
            Dom,
            lpInfo->usri3_name,
            lpInfo->usri3_full_name,
            lpInfo->usri3_home_dir);
        t = localtime(&lpInfo->usri3_last_logon);
        SendLine(skOut, "Last login %s\n", asctime(localtime(&lpInfo->usri3_last_logon)));
        UnicodeToANSI(lpInfo->usri3_home_dir, Buf, sizeof(Buf));
        SendPF(skOut, Buf, "Project");
        UnicodeToANSI(lpInfo->usri3_home_dir, Buf, sizeof(Buf));
        SendPF(skOut, Buf, "Plan");
        NetApiBufferFree(lpInfo);
    } else {  // if
        switch (dwRetCode) {
            case NERR_UserNotFound:
                SendLine(skOut, "unknown userid.\n");
                break;
            case ERROR_INVALID_LEVEL:
                SendLine(skOut, "%s does not have accounts.\n", lpDom);
                break;
            default:
                SendLine(skOut, "error: %ld\n", dwRetCode);
        }  // switch
    }  // if !
}  // SendFinger


// SendFingerAll - Sends the list of those logged on to skOut
//
// Parameters:
//   skOut  - an open socket
//
// Return value:
//   none
//
// Comments:
//   Under NT, how does one define 'logged on' in the absence
//   of a local logon context?  Let's list those with connections...

void SendFingerAll(SOCKET skOut)
{
    NET_API_STATUS dwRetCode;
    LPWKSTA_USER_INFO_1 lpInfo = NULL;
    LPSESSION_INFO_10   SIBuf  = NULL;
    LPUSER_INFO_3       UIBuf  = NULL;
    DWORD dwEntries, dwTotal;
    DWORD hResume, iTime;
    LPWSTR FullName;
    WCHAR LogonTime[30];
    TCHAR IdleTime[10];
    int hr, mn, sc;
    unsigned int i;
    char *lpxx;

    static char Format[] = "%-10.10s  %-20.20s %-8.8s %-20.20s %-18.18s";
    static TCHAR wFormat[] = TEXT("%-10.10ws %-20.20ws %-8.8ws %-20.20ws %-18.18ws");
  
    hResume = 0;
    // NetWkstaUserEnum to get all users
    if ((dwRetCode = NetWkstaUserEnum(NULL,     // Do the local machine
            1,                                  // Level 1?
            (LPBYTE *) &lpInfo,                 // Return buffer
            65536,                              // Max return data len
            &dwEntries,                         // Number of entries returned
            &dwTotal,                           // Number of entries possible from here
            &hResume)) == NERR_Success) {       // Resume handle
        SendLine(skOut, Format, "Login", "Name", "Idle", "When", "Where");
        if (dwEntries != 0) {
            for (i=0; i<dwEntries; ++i) {
                WCHAR Dom[DNLEN+1];             // Domain name
                LPWSTR lpDom = (LPWSTR)&Dom;

                wcscpy(Dom,TEXT("\\\\"));
                wcscat(Dom,lpInfo[i].wkui1_logon_server);
                if (NetUserGetInfo(lpDom,
                                   lpInfo[i].wkui1_username,
                                   3L,
                                   (LPBYTE*)&UIBuf) == NERR_Success) {
                    FullName  = UIBuf->usri3_full_name;
                    lpxx = asctime(localtime(&(UIBuf->usri3_last_logon)));
                    ANSIToUnicode(lpxx, LogonTime, sizeof(LogonTime));
                } else {
                    FullName = TEXT("");
                    wcscpy(LogonTime, TEXT(""));
                }
                SendLineW(skOut,
                          wFormat,
                          lpInfo[i].wkui1_username,
                          FullName,
                          TEXT(""),
                          LogonTime,
                          lpInfo[i].wkui1_logon_server);
            }  // for
        }  // if
    }  // if
    if (lpInfo) 
        NetApiBufferFree(lpInfo);
    if (UIBuf)
        NetApiBufferFree(UIBuf);
    UIBuf = NULL;

    hResume=0;
    do {
        // NetSessionEnum to get all sessions
        dwRetCode = NetSessionEnum(NULL,            // Server name - LOCAL
                                   NULL,            // Client name - ALL
                                   NULL,            // User name - ALL
                                   10,              // Level
                                   (LPBYTE *)&SIBuf,// Buffer
                                   1024,            // Preferred Buffer len
                                   &dwEntries,      // Entries read
                                   &dwTotal,        // Total entries
                                   &hResume);       // Resume handle
        if (((dwRetCode==NERR_Success) || (dwRetCode==ERROR_MORE_DATA))
           && (SIBuf != NULL)) {
            for(i=0; i<dwEntries; i++) {
                if (SIBuf[i].sesi10_username[0]!='\0') {
                    if (NetUserGetInfo(NULL,
                                       SIBuf[i].sesi10_username,
                                       3L,
                                       (LPBYTE*)&UIBuf) == NERR_Success) {
                        FullName  = UIBuf->usri3_full_name;
                        lpxx = asctime(localtime(&(UIBuf->usri3_last_logon)));
                        ANSIToUnicode(lpxx, LogonTime, sizeof(LogonTime));
                    } else {
                        FullName  = TEXT("");
                        wcscpy(LogonTime,TEXT(""));
                    }
                    iTime = SIBuf[i].sesi10_idle_time;
                    hr    = iTime/3600;
                    mn    = iTime/60-hr*60;
                    sc    = iTime%60;
                    wsprintfW(IdleTime, TEXT("%d:%02d:%02d"), hr, mn, sc);
                    SendLineW(skOut,
                              wFormat,
                              SIBuf[i].sesi10_username,
                              FullName,
                              IdleTime,
                              LogonTime,
                              SIBuf[i].sesi10_cname);
                }  // if
            }  // for
        }  // if
   } while ((dwEntries!=0) && (dwTotal!=0) && (dwRetCode==ERROR_MORE_DATA));
   if (SIBuf)
       NetApiBufferFree(SIBuf);
   if (UIBuf)
       NetApiBufferFree(UIBuf);
}  // SendFingerAll


// SendPF - Sends the user's plan or project file
//
// Parameters:
//   skOut   - an open socket
//   HomeDir - users home directory
//
// Return value:
//   TRUE if successful
//
// Comments:
//   Assumes that HomeDir is big enough to accept .plan - should fix this

BOOL SendPF(SOCKET skOut, LPSTR HomeDir, LPSTR PP)
{
    FILE *infile;
    BOOL Result = TRUE;
    
    SendLine(skOut, "%s:\n", PP);
    strcat(HomeDir, "\\.");
    strcat(HomeDir, PP);
    if (infile = fopen(HomeDir, "rb")) {       // Open for read
        SendFilePrim(infile, skOut);
        fclose(infile);
    } else {
        SendLine(skOut, "no %s.\n", PP);
        Result = FALSE;
    }
    return(Result);
}  // SendPF


// SendFile - Sends the named file to the client
//
// Parameters:
//   skOut    - an open socket
//   FileName - the file to send
//
// Return value:
//   TRUE if successful

BOOL SendFile(SOCKET *skOut, LPSTR FileName)
{
    FILE *infile;
    BOOL Result = TRUE;
    
//    Blocking(*skOut);                        // going synchronous now
    if (infile = fopen(FileName, "rb")) {       // Open for read
        SendFilePrim(infile, *skOut);
        fclose(infile);
    } else {
        SendError(*skOut, _strerror(NULL));  // file access error text
        Result = FALSE;
        LogMessage(TEXT("SendFile: file not found"));
    }
    return(Result);
}  // SendFile


// SendFilePrim - Sends the open infile down the skOutput stream
//
// Parameters:
//   infile - an open file descriptor
//   skOut  - an open socket
//
// Return value:
//   none

void SendFilePrim(FILE *infile, SOCKET skOut)
{
    char Buf[XFERBUFLEN];                       // Send buffer
    int nchars;

    do {
        nchars = fread(Buf, 1, sizeof(Buf), infile);
        if (ferror(infile)) {
            SendError(skOut, _strerror(NULL));
            return;
        }

        if ((nchars > 0) && (send(skOut, Buf, nchars, 0) == SOCKET_ERROR)) {
            LogWSError();
            return;
        }
    } while (!feof(infile));
}  // SendFilePrim


// SendLine - Send a text buffer on a socket
//
// Parameters:
//   skOut  - an open socket
//   fmtstr - format string
//   <args> - arguments for the format string
//
// Return value:
//   Returns 0 on success, -1 on SOCKET_ERROR
 
int SendLine(SOCKET skOut, LPSTR fmtstr, ...)
{
    va_list args;
    char Buf[XFERBUFLEN];

    va_start(args, fmtstr);
    wvsprintfA(Buf, fmtstr, args);
//    vsprintf(Buf, fmtstr, args);
    va_end(args);

    if (send(skOut, Buf, strlen(Buf), 0) == SOCKET_ERROR) {
        LogWSError();
        return(-1);
    }
    return(0);
}  // SendLine


// SendLineW - Send a text buffer on a socket
//
// Parameters:
//   skOut  - an open socket
//   fmtstr - format string
//   <args> - arguments for the format string
//
// Return value:
//   Returns 0 on success, -1 on SOCKET_ERROR
//
// Comments:
//   SendLineW sends a Unicode buffer

int SendLineW(SOCKET skOut, LPTSTR fmtstr, ...)
{
    va_list args;
    CHAR  BufA[XFERBUFLEN];
    TCHAR BufW[XFERBUFLEN];

    va_start(args, fmtstr);
    wvsprintf(BufW, fmtstr, args);
    va_end(args);

    UnicodeToANSI(BufW, BufA, BUFSIZE);
    if (send(skOut, BufA, strlen(BufA), 0) == SOCKET_ERROR) {
        LogWSError();
        return(-1);
    }
    return(0);
}  // SendLineW


// SendError - Sends an error string to client.
//
// Parameters:
//   skOut  - an open socket
//   errstr - the error text
//
// Return value:
//   none

void SendError(SOCKET skOut, char *errstr)
{
   static char *prefix = "\r\nserver error: ";

   SendLine(skOut, prefix);
   SendLine(skOut, errstr);
}  // SendError

// LogWSError - Logs the winsock error in the Application Event Log
void LogWSError()
{
    UINT err;
    
    err = WSAGetLastError();
    LogEvent(ec_WINSOCK, err, WSErrorString(err));
}  // LogWSError

// WSErrorString - Translates winsock error to appropriate string.
char *WSErrorString(UINT err)
{
   int i;
   static char szerr[80];

   for (i = 0; i < dim(WSErrors); i++)
      if (err == WSErrors[i].err)
         return(WSErrors[i].sztext);

   wsprintfA(szerr, "Windows Sockets reports error %04x (%d)", err, err);
   return(szerr);
}  // WSErrorString


// SendProcess - Send local process information

void SendProcess(SOCKET skOut)
{
    DWORD j;
    LPVOID pMem, pData;
    LONG lj;      // lj used for instances loop. NumInstances is LONG
                        // and not DWORD as stated in the documentation.

    PPERF_DATA_BLOCK          pPDB;
    PPERF_OBJECT_TYPE         pPOT;
    PPERF_COUNTER_DEFINITION  pPCD;
    PPERF_INSTANCE_DEFINITION pPID;
    PPERF_COUNTER_BLOCK       pPCB;

    PPERF_COUNTER_DEFINITION  pPCD_ID;
    PPERF_COUNTER_DEFINITION  pPCD_Thread;

    if (j = PerformanceKeyRetrieve( TEXT("230"), &pMem)) {
        SendLine(skOut, "Error %d getting performance information\n", j);
        RegCloseKey(HKEY_PERFORMANCE_DATA);
        return;
    }

    // Type the data on the screen ...
    pPDB = (PPERF_DATA_BLOCK)pMem;
    pPOT = FirstObject(pPDB);

    if (dwDebug)
        wprintf(TEXT("Num object types: %d\n"), pPDB->NumObjectTypes);

    if (dwDebug)
        wprintf(TEXT("Object: %d\n"), pPOT->ObjectNameTitleIndex);
    while ((pPOT->ObjectNameTitleIndex != 230) && (pPOT->ObjectNameTitleIndex != 0)) {
        pPOT = NextObject(pPOT);
        if (dwDebug)
            wprintf(TEXT("Object: %d\n"), pPOT->ObjectNameTitleIndex);
    }

    //pPOT->NumInstances
    pPCD = FirstCounter(pPOT);

    __try
    {
        // Locate the Process ID counter definition ...
        pPCD_ID = pPCD_Thread = NULL;
        for (j=0; j<pPOT->NumCounters; j++) {
            if (pPCD->CounterNameTitleIndex == 784) // ID
                pPCD_ID = pPCD;
            if (pPCD->CounterNameTitleIndex == 680) // Thread count
                pPCD_Thread = pPCD;
                
            pPCD = NextCounter(pPCD);
        }

        SendLineW(skOut, TEXT("  PROCESS NAME       PID    Threads\n"));
        SendLineW(skOut, TEXT("-----------------  -------  -------\n"));

        pPID = FirstInstance(pPOT);
        for (lj=0; lj<pPOT->NumInstances; lj++) {
            // Locate the data block for the process ID for this instance ...
            pPCB = (PPERF_COUNTER_BLOCK)((LPBYTE)pPID + pPID->ByteLength);
            pData = (LPBYTE)pPCB + pPCD_ID->CounterOffset;

            if (pPID->NameLength) {
                SendLineW(skOut, TEXT("%17s  "), (LPCTSTR)((LPBYTE)pPID + pPID->NameOffset));
                wprintf(TEXT("%17s  "), (LPCTSTR)((LPBYTE)pPID + pPID->NameOffset));
            } else {
                SendLineW(skOut, TEXT("UNNAMED(%d)  "), lj);
                if (dwDebug)
                    wprintf(TEXT("UNNAMED(%d)  "), lj);
            }         
            SendLineW(skOut, TEXT("%7u  "), *((DWORD *)pData));
            if (dwDebug)
                wprintf(TEXT("%7u  "), *((DWORD *)pData));

            pData = (LPBYTE)pPCB + pPCD_Thread->CounterOffset;
            SendLineW(skOut, TEXT("%7u\n"), *((DWORD *)pData));
            if (dwDebug)
                wprintf(TEXT("%7u\n"), *((DWORD *)pData));

            pPID = NextInstance(pPID);
        }
    }
    __except(GetExceptionCode() == EXCEPTION_ACCESS_VIOLATION ? EXCEPTION_EXECUTE_HANDLER : EXCEPTION_CONTINUE_SEARCH)
    {
        // Just handle the exception ...
        SendLineW(skOut, TEXT("\n\nACCESS VIOLATION\n\n"));
    }

    RegCloseKey(HKEY_PERFORMANCE_DATA);
    HeapFree(GetProcessHeap(), 0, pMem);
    
    return;
}  // SendProcess

