/* savedir.c -- save the list of files in a directory in a string
   Copyright (C) 1990, 1997, 1998 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* Written by David MacKenzie <djm@gnu.ai.mit.edu>. */

#if HAVE_CONFIG_H
# include <config.h>
#endif

#include <sys/types.h>

#if HAVE_UNISTD_H
# include <unistd.h>
#endif

#if HAVE_DIRENT_H
# include <dirent.h>
# define NAMLEN(dirent) strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif

#ifdef CLOSEDIR_VOID
/* Fake a return value. */
# define CLOSEDIR(d) (closedir (d), 0)
#else
# define CLOSEDIR(d) closedir (d)
#endif

#ifdef STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#else
char *malloc ();
char *realloc ();
#endif
#ifndef NULL
# define NULL 0
#endif
#ifndef MAX
# define MAX(a,b) (((a)>(b))?(a):(b))
#endif

#ifndef stpcpy
char *stpcpy ();
#endif

#include "savedir.h"

/* Return a freshly allocated string containing the filenames
   in directory DIR, separated by '\0' characters;
   the end is marked by two '\0' characters in a row.
   NAME_SIZE is the number of bytes to initially allocate
   for the string; it will be enlarged as needed.
   Return NULL if DIR cannot be opened or if out of memory.

   Note regarding st_size for directories  This value is used as
   a hint for the initial buffer size required to hold all the
   names of entries in a directory.  It is dynamically realloced
   if necessary, but *only* if the directory is found to have
   names that need to be stored in NAME_SPACE.  However NAME_SPACE
   is unconditionally null terminated, regardless of whether
   anything gets stored there or not.  Consider the special case 
   of an empty directory on a filesystem where st_size for
   directories is always zero (such as '/' on BeOS).  In this case
   NAME_SPACE will be allocated with zero length and it is wrong
   to write anything to it, including two terminating nulls.  Thus
   always allocate NAME_SPACE with at least two bytes.
   
   Also consider what happens if no entries at all are found, such
   as an empty directory on a NFS mounted partition, which as the 
   code comment indicates, may not have the usual '.' and '..' dirs.
   Since the first terminating null comes from the stpcpy function
   this null will never be written and the buffer is only guaranteed
   to be have one null byte rather than the double null bytes that
   the function API specifies.  So preterminate the buffer to guard
   against this case. */

char *
savedir (dir, name_size)
     const char *dir;
     unsigned int name_size;
{
  DIR *dirp;
  struct dirent *dp;
  char *name_space;
  char *namep;

  dirp = opendir (dir);
  if (dirp == NULL)
    return NULL;

  name_size = MAX (name_size, 2);
  name_space = (char *) malloc (name_size);
  if (name_space == NULL)
    {
      closedir (dirp);
      return NULL;
    }
  namep = name_space;
  *namep = '\0';
  *(namep + 1) = '\0';

  while ((dp = readdir (dirp)) != NULL)
    {
      /* Skip "." and ".." (some NFS filesystems' directories lack them). */
      if (dp->d_name[0] != '.'
	  || (dp->d_name[1] != '\0'
	      && (dp->d_name[1] != '.' || dp->d_name[2] != '\0')))
	{
	  unsigned size_needed = (namep - name_space) + NAMLEN (dp) + 2;

	  if (size_needed > name_size)
	    {
	      char *new_name_space;

	      while (size_needed > name_size)
		name_size += 1024;

	      new_name_space = realloc (name_space, name_size);
	      if (new_name_space == NULL)
		{
		  closedir (dirp);
		  return NULL;
		}
	      namep += new_name_space - name_space;
	      name_space = new_name_space;
	    }
	  namep = stpcpy (namep, dp->d_name) + 1;
	}
    }
  *namep = '\0';
  if (CLOSEDIR (dirp))
    {
      free (name_space);
      return NULL;
    }
  return name_space;
}
