;***********************************************************************
;*                                                                     *
;* PORTS.ASP (C) 1990 DATASTORM TECHNOLOGIES, INC.                     *
;*                                                                     *
;* An ASPECT script for checking the status of a communications port   *
;* and for demonstrating some of the "low-level" control possible in   *
;* ASPECT.                                                             *
;***********************************************************************

integer IER, IIR, LCR, MCR, LSR, MSR, base_address, comport, irq
integer ier_val, iir_val, lcr_val, mcr_val, lsr_val, msr_val
integer anykey, statsave, vidseg, rowbytes
string result

;* ******************************************************************* *

proc main
   vidsave 0                           ; save the current screen
   clear                               ; wipe off screen
   set keys on                         ; script file processes all keystokes
   set rxdata on                       ; don't let incoming data mess up the screen
   fetch statline statsave             ; save current status line condition
   set statline off                    ; turn off status line
   if mono                             ; set up video segment value
      vidseg = 0xB000                  ; mono base address
   else
      vidseg = 0xB800                  ; color base address
   endif
   rowbytes = $scrncols * 2            ; initialize number of bytes per row
   curoff                              ; turn the cursor off
   call opening_screen                 ; draw box with header
   fetch port comport                  ; get current port from Alt-P
   call readport with comport          ; open .PRM file/read port assignments
   call main_menu                      ; display main menu options
   call terminate                      ; terminate the script file
endproc

;* ******************************************************************* *

proc terminate
   if statsave == 1                    ; if status line was on originally
      set statline on                  ; turn it back on
   endif
   vidrest 0                           ; put the original screen back
   curon                               ; turn the cursor on
   terminal                            ; exit to terminal mode
endproc

;* ******************************************************************* *

proc opening_screen
   string boxline = "Ķ"

   box 2  6 22 73 78
   atsay  3 28 79 "P O R T   C H E C K - U P"
   atsay  4  6 78 boxline
   atsay  8  6 78 boxline
   atsay 20  6 78 boxline
   atsay 21 18 78 " to select      to view     ESC to exit"
endproc

;* ******************************************************************* *

proc menu_bar
intparm row, col, length, attrib
   integer vidoff, vidlen
   string vidline

   vidoff = (row * rowbytes) + (col << 1)
   vidlen = length << 1
   memread vidseg vidoff vidline vidlen
   length += col - 1
   for col upto length                 ; loop LENGTH number of
      putvattr row col attrib          ; times to write attr.
   endfor                              ; characters
   keyget anykey                       ; get user's menu choice
   memwrite vidseg vidoff vidline vidlen
endproc

;* ******************************************************************* *

proc main_menu
   integer row = 10

   atsay 10 30 79 "Change Current Port"
   atsay 12 30 79 "  View Registers"
   while 1                             ; equals 0 if ESC
      call menu_bar with row 28 23 112
      switch anykey
         case 0x50E0                   ; grey DOWN arrow
         case 0x5000                   ; keypad DOWN arrow
            if row == 12
               row = 10
            else
               row += 2
            endif
         endcase
         case 0x48E0                   ; grey UP arrow
         case 0x4800                   ; keypad UP arrow
            if row == 10
               row = 12
            else
               row -= 2
            endif
         endcase
         case 0x0D                     ; enter key
            if row == 10
               call port_menu
            else
               call reg_menu
            endif
         endcase
         case 0x1B                     ; ESC key
            return
         endcase
      endswitch
   endwhile
endproc

;* ******************************************************************* *

proc port_menu
   integer choice, ports, row = 9
   string biosdata, baseaddr

   vidsave 2
   box 8 32 17 43 31
   for choice = 1 upto 8
      fatsay row 36 31 "COM%i" choice
      row++
   endfor
   memread 0 0x400 biosdata 8
   for choice = ports = 0 upto 7
      strpeek biosdata choice row
      if row
         ports++
         if !(choice % 2)
            choice++
         endif
      endif
   endfor
   fatsay 19 23 79 "BIOS reports %d available COM ports" ports
   row = 9
   while 1                             ; equals 0 if ESC
      call menu_bar with row 33 10 112
      switch anykey
         case 0x50E0                   ; grey UP arrow
         case 0x5000                   ; keypad UP arrow
            if row == 16
               row = 9
            else
               row++
            endif
         endcase
         case 0x48E0                   ; grey DOWN arrow
         case 0x4800                   ; keypad DOWN arrow
            if row == 9
               row = 16
            else
               row--
            endif
         endcase
         case 0x0D                     ; enter key
            choice = row - 8
            vidrest 2
            call readport with choice
            return
         case 0x1B                     ; ESC key
            vidrest 2
            return
         endcase
      endswitch
   endwhile
endproc

;* ******************************************************************* *

proc reg_menu
   integer choice, row = 9

   vidsave 2
   scroll 0 9 7 19 72 79
   atsay  9 22 79 "Interrupt Enable Register:"
   atsay 11 22 79 "    Interrupt Id Register:"
   atsay 13 22 79 "    Line Control Register:"
   atsay 15 22 79 "   Modem Control Register:"
   atsay 17 22 79 "     Line Status Register:"
   atsay 19 22 79 "    Modem Status Register:"
   call read_reg
   call to_binary with ier_val 0
   atsay  9 49 79 result
   call to_binary with iir_val 0
   atsay 11 49 79 result
   call to_binary with lcr_val 0
   atsay 13 49 79 result
   call to_binary with mcr_val 0
   atsay 15 49 79 result
   call to_binary with lsr_val 0
   atsay 17 49 79 result
   call to_binary with msr_val 0
   atsay 19 49 79 result
   while 1
      call menu_bar with row 20 39 112
      switch anykey
         case 0x50E0                   ; grey DOWN arrow
         case 0x5000                   ; keypad DOWN arrow
            if row == 19
               row = 9
            else
               row += 2
            endif
         endcase
         case 0x48E0                   ; grey UP arrow
         case 0x4800                   ; keypad UP arrow
            if row == 9
               row = 19
            else
               row -= 2
            endif
         endcase
         case 0x0D                     ; enter key
            choice = ((row - 9) >> 1) + 1;
            call view_port with choice
         endcase
         case 0x1B                     ; ESC key
            vidrest 2
            return
         endcase
      endswitch
   endwhile
endproc

;* ******************************************************************* *

proc view_port
intparm choice
   integer value

   vidsave 1
   box 9 3 19 76 30
   switch choice
      case 1
         atsay 10 27 31 "Interrupt Enable Register"
         atsay 12  7 30 "N/A  N/A  N/A  N/A  Modem   Recv line   Trans Holding   Recv Data"
         atsay 13  7 30 "                    Status   Status     Register Empty  Available"
         atsay 14  7 30 "                              "
         atsay 15  7 30 "                               Ŀ                   "
         atsay 16  7 30 "                       Ŀ                          "
         atsay 17  7 30 "                                                         "
         value = ier_val
      endcase
      case 2
         atsay 10 24 31 "Interrupt Identification Register"
         atsay 12  7 30 "* FIFOs Enabled  N/A     N/A    Id      Id      Id      Interrupt"
         atsay 13  7 30 "               * Bit(2)  Bit(1)  Bit(0)    Pending "
         atsay 14  7 30 "                                     "
         atsay 15  7 30 "                                                           "
         atsay 16  7 30 "                                                           "
         atsay 17  7 30 "                                                         "
         value = iir_val
         box 21 8 23 55 47
         atsay 22 12 46 "* Only available on the NS16550 UART"
      endcase
      case 3
         atsay 10 29 31 "Line Control Register"
         atsay 12  7 30 " DLAB   Set   Stick     Even    Parity   Stop    Word      Word"
         atsay 13  7 30 "   Break  Parity   Parity   Enable   Bits  Length(1)  Length(0)"
         atsay 14  7 30 "                     "
         atsay 15  7 30 "                                                         "
         atsay 16  7 30 "                                                         "
         atsay 17  7 30 "                                                         "
         value = lcr_val
      endcase
      case 4
         atsay 10 29 31 "Modem Control Register"
         atsay 12  7 30 " N/A   N/A   N/A   Loopback   OUT2   OUT1   Request   Data Terminal"
         atsay 13  7 30 "                     Test           to Send       Ready    "
         atsay 14  7 30 "                                      "
         atsay 15  7 30 "                               Ŀ   Ŀ                   "
         atsay 16  7 30 "                                                            "
         atsay 17  7 30 "                                                         "
         value = mcr_val
      endcase
      case 5
         atsay 10 29 31 "Line Status Register"
         atsay 12  4 30 "N/A  Trans  Trans Holding    Break   Framing  Parity  Overrun  Recv Data"
         atsay 13  4 30 "     Empty  Register Empty  Detected  Error    Error   Error     Ready  "
         atsay 14  4 30 "                    "
         atsay 15  4 30 "       Ŀ                                            "
         atsay 16  4 30 "                                                               "
         atsay 17  4 30 "                                                              "
         value = lsr_val
      endcase
      case 6
         atsay 10 29 31 "Modem Status Register"
         atsay 12  5 30 "Carrier  Ring  Data Set  Clear To  Delta    Delta    Delta     Delta"
         atsay 13  5 30 "Detect  Detect  Ready      Send    Carrier  Ring    Data Set  Clear To "
         atsay 14  5 30 "             Detect   Detect   Ready      Send  "
         atsay 15  5 30 "                                           "
         atsay 16  5 30 "                                                       "
         atsay 17  5 30 "                                                            "
         value = msr_val
      endcase
   endswitch
   call to_binary with value 1
   atsay 18 10 31 result
   keyget
   vidrest 1
endproc

;* ******************************************************************* *

proc to_binary
intparm number, mode                   ; number passed from caller
   integer counter, x, y,remainder
   string fmtstr = "%d", str_parity

   if mode
      strcat fmtstr "       "
   endif
   y = 128
   result = ""
   for counter = 7 downto 0
      remainder = number & y
      shr remainder counter x
      strfmt str_parity fmtstr x
      strcat result str_parity
      shr y 1 y
   endfor
endproc

;* ******************************************************************* *

proc readport                          ; Note: This procedure reads data
intparm portnum                        ; from particular file offsets in
   integer hi_byte, lo_byte            ; PCPLUS.PRM; this practice,
   long position                       ; however, is not recommended
   string envstr

   fopen 0 "PCPLUS.PRM" "rb"           ; open PCPLUS.PRM for read only binary
   if failure                          ; if not in current directory
      getenv "PCPLUS" envstr           ; check PCPLUS environment variable
      strlen envstr hi_byte
      if hi_byte                       ; if environment variable exists
         hi_byte--                     ; format new path
         strpeek envstr hi_byte hi_byte
         if hi_byte != '\'
            strcat envstr "\"
         endif
         strcat envstr "PCPLUS.PRM"
      endif
      fopen 0 envstr "rb"              ; open PCPLUS.PRM for read only binary
      if failure                       ; if still unable to open file
         errormsg "Unable to open PCPLUS.PRM"
         call terminate                ; terminate script
      endif
   endif
   position = ((portnum-1) * 2) + 372  ; calculate offset of base address
   fseek 0 position 0
   fgetc 0 lo_byte
   fgetc 0 hi_byte
   position = ((portnum-1) * 2) + 388  ; calculate offset of IRQ number
   fseek 0 position 0
   fgetc 0 irq
   base_address = (hi_byte * 0x100) + lo_byte
   fclose 0
   comport = portnum
   IER = base_address + 1
   IIR = base_address + 2
   LCR = base_address + 3
   MCR = base_address + 4
   LSR = base_address + 5
   MSR = base_address + 6
   call port_update
endproc

;* ******************************************************************* *

proc read_reg
   inport IER ier_val
   inport IIR iir_val
   inport LCR lcr_val
   inport MCR mcr_val
   inport LSR lsr_val
   inport MSR msr_val
endproc

;* ******************************************************************* *

proc port_update
   integer c_databits, c_parity, c_stopbits
   integer lo_baud, hi_baud, set_dlab, unset_dlab, divisor
   long c_baudrate
   string str_parity

   call verify                         ; see if port is valid
   call read_reg                       ; read the I/O registers
   c_databits = (lcr_val & 0x03) + 5
   c_stopbits = ((lcr_val & 0x04) >> 2) + 1
   c_parity   = (lcr_val & 0x38) >> 3
   set_dlab = lcr_val | 0x80
   outport LCR set_dlab                ; raise DLAB to get baud divisor
   mspause 100
   inport base_address lo_baud         ; lo-byte of baud rate divisor
   mspause 100
   inport IER hi_baud                  ; hi-byte of baud rate divisor
   unset_dlab= lcr_val & 0x7F
   outport LCR unset_dlab              ; make sure DLAB low
   mspause 100
   divisor=(hi_baud * 0x100) + lo_baud
   switch divisor
      case 1
         c_baudrate = 115200
      endcase
      case 2
         c_baudrate = 57600
      endcase
      case 3
         c_baudrate = 38400
      endcase
      case 6
         c_baudrate = 19200
      endcase
      case 12
         c_baudrate = 9600
      endcase
      case 16
         c_baudrate = 7200
      endcase
      case 24
         c_baudrate = 4800
      endcase
      case 32
         c_baudrate = 3600
      endcase
      case 48
         c_baudrate = 2400
      endcase
      case 64
         c_baudrate = 1800
      endcase
      case 96
         c_baudrate = 1200
      endcase
      case 192
         c_baudrate = 600
      endcase
      case 384
         c_baudrate = 300
      endcase
      case 768
         c_baudrate = 150
      endcase
      case 857
         c_baudrate = 134
      endcase
      case 1047
         c_baudrate = 110
      endcase
      case 1536
         c_baudrate = 75
      endcase
      case 2304
         c_baudrate = 50
      endcase
      default
         c_baudrate = divisor
      endcase
   endswitch
   switch c_parity                     ; convert parity from number to string
      case 1
         str_parity = "Odd"
      endcase
      case 3
         str_parity = "Even"
      endcase
      case 5
         str_parity = "Mark"
      endcase
      case 7
         str_parity = "Space"
      endcase
      default
         str_parity = "None"
      endcase
   endswitch
   fatsay 5 11  79 "Current port:          Base address:           IRQ line:    "
   fatsay 5 25 112 "COM%i" comport
   fatsay 5 48 112 "0x%X" base_address
   atsay  5 68 112 irq
   fatsay 7 11  79 "Baud:         Databits:       Parity:          Stopbits:    "
   atsay  7 17 112 c_baudrate
   atsay  7 35 112 c_databits
   atsay  7 49 112 str_parity
   atsay  7 68 112 c_stopbits
endproc

;* ******************************************************************* *

proc verify
   integer org_lcr, new_lcr

   inport  LCR org_lcr                 ; Get the original LCR settings
   outport LCR 0x1A                    ; Set port to E-7-1
   outport 0xC0 0xFF                   ; Dummy OUT to "clear" bus
   inport  LCR new_lcr                 ; Read the LCR again
   outport LCR org_lcr                 ; Put the original settings back
   if new_lcr != 0x1A
      alarm 1
      box 7 18 15 60 30
      atsay  8 35 159 "E R R O R"
      atsay  9 18 30  "Ķ"
      atsay 10 21 31  "The port chosen is not valid or does"
      atsay 11 21 31  "not exist.  Verify the Base Address "
      atsay 12 21 31  "in the PORT ASSIGNMENTS under MODEM "
      atsay 13 21 31  "OPTIONS from the ALT-S SETUP menu."
      atsay 15 27 31  " Press any key to exit "
      keyget
      call terminate                   ; terminate script
   endif
endproc

