/*****************************************************************************
 * File:	searchCmnds.c
 *
 * Author:	Rhett "Jonzy" Jones
 *		jonzy@cc.utah.edu
 *
 * Date:	May 13, 1993
 *
 * Modified:	May 23, 1993 by Rhett "Jonzy" Jones.
 *		Commented out the pathway stuff with #ifdef DO_PATHWAY
 *		because this isn't ready yet.
 *		Merged this routine into jughead.
 *		Modified StillNeed2Parse() such that "?" gets mapped to
 *		"?help".
 *
 *		At the request of doylej@liberty.uc.wlu.edu gave support
 *		for the "range=start-stop what" special command.
 *
 *		April 26, 1994, by Rhett "Jonzy" Jones with
 *		code from Maes@elec.ucl.ac.BE
 *		Added the -DLOCALE to the CFLAGS to support
 *		the ISO-8859-1 character set for the European Community.
 *
 *		May 3, 1994, by Rhett "Jonzy" Jones.
 *		Moved many definitions from the Makefile into
 *		jughead.conf.h and now include jughead.conf.h.
 *
 *		May 6, 1994, by Rhett "Jonzy" Jones.
 *		Changed all calls to strcmp() to StrCmp() to support both
 *		ASCII and ISO-XXXX-Y character sets.  Removed use of LOCALE.
 *
 *		May 9, 1994, by Rhett "Jonzy" Jones.
 *		With input from Kuypers@sri.ucl.ac.BE eased the work
 *		required to localize jughead to a given spoken language
 *		by using the define JUGHEADHELPTITL, which is the
 *		title of the jughead help document.
 *		Added USAGERROR to localize the string "usage error".
 *
 *		May 22, 1994, by Rhett "Jonzy" Jones.
 *		Cleaned up the code for lint.
 *
 *		June 7, 1994, by Rhett "Jonzy" Jones.
 *		Changed JUGHEADHELP to jugheadhelp to support the new
 *		jughead.conf file.
 *
 *		July 22, 1994, by Rhett "Jonzy" Jones.
 *		Added support for the special commands "?datafile" and
 *		"?datasize" to in getting the datafile to veronica.
 *		This includes the addition of the routines:
 *		GetDateFileSize() and SendDataFile().
 *
 *		July 24, 1994, by Rhett "Jonzy" Jones.
 *		Removed inclusion of jughead.conf.h.
 *
 *		September 22, 1994, by Rhett "Jonzy" Jones.
 *		Added the routine WhichFile(), which returns the name
 *		of the veronica data file "data.veronica" if it exists
 *		otherwise it returns the name of the data file "data".
 *		and added use of the global variable 'veronica'.
 *
 * Description:	This module takes care of some special commands for use
 *		by the jughead search engine.  A special command is defined
 *		as one of the following:
 *			?all what
 *			?datafile
 *			?datasize
 *			?help [what]
 *			?limit=n what
 *			?version [what]
 *			?range=start-stop what
 *			?pathway what		NOT supported yet.
 *		where 'what' is a standard search string, anything enclosed
 *		in square brackets is optional, and all special commands must
 *		be preceeded with '?'.
 *
 *		The pathway command is currently not supported.
 *
 *		WARNING:  Any attempt to alter the words for any of these
 *		commands will result in a jughead that is not compatible with
 *		future versions of jughead.  These special commands are
 *		inherent to the jughead protocol.
 *
 * Routines:	static	short	StillNeed2Parse(char *string,short *theCommand,
 *						long *limit,long *rangeStart,
 *						long *rangeEnd);
 *		static	long	GetDateFileSize(char *fName);
 *		static	int	SendDataFile(char *fName);
 *		static	int	HandleSpecialCommand(char *string,
 *						     short theCommand);
 *			char	*SpecialCommand(char *what2find,long *limit);
 *
 * Bugs:	No known bugs.
 *
 * Copyright:	Copyright 1993, 1994, University of Utah Computer Center.
 *		JUGHEAD (TM) is a trademark of Archie Comic Publications, Inc.
 *		and is used pursuant to license.  This source may be used and
 *		freely distributed as long as this copyright notice remains
 *		intact, and is in no way used for any monetary gain, by any
 *		institution, business, person, or persons.
 *
 ****************************************************************************/

#include <ctype.h>
#include <stdio.h>
#include <string.h>

#define NIL		(void *)0

#define RANGEERROR	-4	/* Error with range usage. */
#define PATHWAYERROR	-3	/* Error with pathway usage. */
#define CMDUSEERROR	-2	/* Generic usage error. */
#define LIMITERROR	-1	/* Error with limit usage. */

/* Various jughead commands followed by syntax. */
#define ALLCMND		1	/* ?all what */
#define DATACMND	2	/* ?datafile */
#define DATSZCMND	3	/* ?datasize */
#define HELPCMND	4	/* ?help [what] */
#define LIMITCMND	5	/* ?limit=n what */
#define VERSIONCMND	6	/* ?version [what] */
#define RANGECMND	7	/* ?range=start-end what */
#ifdef DO_PATHWAY
#define PATHWAYCMND	8	/* ?pathway what */
#endif

/* This variable lists the special commands for use by the jughead search engine. */
static char	*specialCmnds[] = { "all",	/* Return all entries in excess of 1024. */
				    "datafile",	/* Return the datafile being used. */
				    "datasize",	/* The size of the datafile. */
				    "help",	/* Return a link to a help file. */
				    "limit",	/* Return at most the first n entries. */
				    "version",	/* Return the version of jughead. */
				    "range",	/* Return a range of items.*/
#ifdef DO_PATHWAY
				    "pathway",	/* Return the pathway for a selection. */
#endif
				    0 };
#ifdef DO_PATHWAY
char		*pathString;			/* The selStr to find the path of. */
#endif

extern char	*fileName,		/* Declared in "jughead.c. */
		*veronica;		/* Declared in "jughead.c". */
extern char	*jugheadhelp,		/* Declared in "search.c". */
		*usagerror,
		*hostname,
		*jugheadhelptitl;
extern int	port2use;		/* Declared in "search.c". */

extern int 	SendBuffer();		/* Defined  in "sockets.c". */
extern int 	SendString();		/* Defined  in "sockets.c". */

/*****************************************************************************
 * StillNeed2Parse determines if we have a special command and if so assigns
 * the appropriate command to 'theCommand', which includes any command errors
 * encountered.  This routine returns true if there is still information to
 * parse, and false otherwise.
 ****************************************************************************/
static short StillNeed2Parse(string,theCommand,limit,rangeStart,rangeEnd)
	char	*string;		/* The line of text given. */
	short	*theCommand;		/* Do we have a special command? */
	long	*limit,			/* The number of items to return. */
		*rangeStart,		/* The start of a range. */
		*rangeEnd;		/* The end of a range. */
{	char	*str,			/* Position in 'string'. */
		**s,			/* The special commands. */
		theSpecialCmnd[12];	/* The special command the user issued. */
	short	i;			/* A loop counter. */

	/* Initialize some variables. */
	*rangeStart = *rangeEnd = *theCommand = 0;
	theSpecialCmnd[0] = '\0';

	/* Skip any whitespace. */
	for (str = string; *str && isspace(*str); str++);

	if (*str == '?')		/* Looks like a special command. */
		{
		/* Get a copy of the special command. */
		if (!StrCmp(str,"?"))	/* Treat it like ?help. */
			(void)strcpy(theSpecialCmnd,"help");
		else
			{
			for (++str, i = 0; i < 11 && isalpha(*str); i++, str++)
				theSpecialCmnd[i] = *str;
			theSpecialCmnd[i] = '\0';
			}

		/* Locate the special command we found. */
		for (*theCommand = 1, s = specialCmnds; *s; s++, ++*theCommand)
			if (!StrCmp(*s,theSpecialCmnd))
				break;

		switch (*theCommand)
			{
			case ALLCMND:
				*limit = -1;
				break;
			case DATACMND:
			case DATSZCMND:
			case HELPCMND:
				break;
			case LIMITCMND:
				*limit = 0;
				if (*str == '=' && isdigit(*++str))
					for (*limit = 0; isdigit(*str); str++)
						*limit = *limit * 10 + *str - '0';
				else
					*theCommand = LIMITERROR;
				break;
			case VERSIONCMND:
				break;
			case RANGECMND:
				if (*str == '=' && isdigit(*++str))
					{
					for (*rangeStart = 0; isdigit(*str); str++)
						*rangeStart = *rangeStart * 10 + *str - '0';
					if (*str == '-')
						for (++str, *rangeEnd = 0; isdigit(*str); str++)
							*rangeEnd = *rangeEnd * 10 + *str - '0';
					else
						{
						*rangeStart = *rangeEnd = 0;
						*theCommand = RANGEERROR;
						}
					}
				else
					*theCommand = RANGEERROR;
				break;
#ifdef DO_PATHWAY
			case PATHWAYCMND:
				if (!*str)
					{
					*theCommand = PATHWAYERROR;
					break;
					}
				if (*str)	/* Skip any whitespace. */
					for (++str; isspace(*str); str++);

				/* Extract the pathString. */
				if (*str == '"')
					{
					for (pathString = ++str; *str && *str != '"'; str++);
					if (*str == '"')
						*str++ = '\0';
					else
						{
						*theCommand = PATHWAYERROR;
						break;
						}
					}
				else
					{
					for (pathString = str; *str && !isspace(*str); str++);
					*str = '\0';
					}
				break;
#endif
			default:	/* Oops we're in deep shit. */
				*theCommand = CMDUSEERROR;
				theSpecialCmnd[0] = '\0';
				break;
			}

		/* Shft the special command out from 'string'. */
		if (*str)
			for (++str; isspace(*str); str++);	/* Skip any whitespace. */
		for (i = 0; *str; i++, str++)
			string[i] = *str;
		string[i] = '\0';
		}

	return(*string);

}	/* StillNeed2Parse */

/*****************************************************************************
 * GetDateFileSize returns the number of bytes in the file 'fname', and -1
 * if an error occured.
 ****************************************************************************/
static long GetDateFileSize(fName)
	char	*fName;		/* Name of the datafile. */
{	FILE	*fp;		/* Pointer to the file. */
	long	bytes;		/* Number of bytes in 'fName'. */
	char	ptr[BUFSIZ];	/* Buffer with the file contents. */
	int	itemsRead;	/* The number of bytes read. */

	if (fp = fopen(fName,"r"))
		{
		for (bytes = 0; itemsRead = fread(ptr,1,BUFSIZ,fp); bytes += itemsRead);
		(void)fclose(fp);
		return(bytes);
		}
	else
		return(-1);

}	/* GetDateFileSize */


/*****************************************************************************
 * SendDataFile sends the datafile 'fName' to the client we are talking to.
 * This routine returns true is we had an error, and false otherwise.
 ****************************************************************************/
static int SendDataFile(fName)
	char	*fName;
{	FILE	*fp;		/* Pointer to the file. */
	char	ptr[BUFSIZ];	/* Buffer with the file contents. */
	int	itemsRead;	/* The number of bytes read. */

	if (fp = fopen(fName,"r"))
		{
		while (1)
			{
			itemsRead = fread(ptr,1,BUFSIZ,fp);
			if (itemsRead)
				{
				if (SendBuffer(ptr,itemsRead) != itemsRead)
					break;
				}
			else
				break;
			}
		(void)fclose(fp);
		return(0);
		}
	else
		return(1);

}	/* SendDataFile */

/*****************************************************************************
 * WhichFile returns the pathway to the 'veronica' file "data.veronica" if
 * it exists, otherwise it returns the name of the data file 'fileName'.
 ****************************************************************************/
static char *WhichFile()
{	FILE	*fp;

	if (fp = fopen(veronica,"r"))
		{
		(void)fclose(fp);
		return(veronica);
		}
	else
		return(fileName);
 
}	/* WhichFile */

/*****************************************************************************
 * HandleSpecialCommand returns true if 'theCommand' is an error or we
 * discover an error, and send the information back to the client following
 * gopher protocol.  Otherwise this routine returns false and sets some
 * variables or accomplishes the special command.
 ****************************************************************************/
static int HandleSpecialCommand(string,theCommand)
	char	*string;	/* The remainder of the input line. */
	short	theCommand;	/* The special command or error. */
{	int	error = 0;	/* Did we get an error? */
	char	help[1024];	/* Line of text adhering to gopher protocol. */

	switch (theCommand)
		{
		case RANGEERROR:
			(void)sprintf(help,"0range %s%s",usagerror,jugheadhelp);
			error = SendString(help);
			break;
		case PATHWAYERROR:
			(void)sprintf(help,"0pathway %s%s",usagerror,jugheadhelp);
			error = SendString(help);
			break;
		case CMDUSEERROR:
			(void)sprintf(help,"0jughead %s%s",usagerror,jugheadhelp);
			error = SendString(help);
			break;
		case LIMITERROR:
			(void)sprintf(help,"0limit %s%s",usagerror,jugheadhelp);
			error = SendString(help);
			break;
		case ALLCMND:
			if (!*string)
				{
				(void)sprintf(help,"0all %s%s",usagerror,jugheadhelp);
				error = SendString(help);
				}
			break;
		case DATACMND:
			error = SendDataFile(WhichFile());
			break;
		case DATSZCMND:
			(void)sprintf(help,"0%d bytes\t?datafile\t%s\t%d",GetDateFileSize(WhichFile()),hostname,port2use);
			error = SendString(help);
			break;
		case HELPCMND:
			(void)sprintf(help,"0%s%s",jugheadhelptitl,jugheadhelp);
			(void)SendString(help);
			break;
		case LIMITCMND:
			if (!*string)
				{
				(void)sprintf(help,"0limit usage error%s",jugheadhelp);
				error = SendString(help);
				}
			break;
		case VERSIONCMND:
			(void)sprintf(help,"0This version of jughead is %s%s",VERSION,jugheadhelp);
			(void)SendString(help);
			break;
		case RANGECMND:
			break;
#ifdef DO_PATHWAY
		case PATHWAYCMND:
			/* exec pathway on pathString. */
			break;
#endif
		}
	return(error);

}	/* HandleSpecialCommand */

/*****************************************************************************
 * SpecialCommand handles any special command the jughead search engine
 * might get, and returns nil if there is nothing to parse out, and the
 * string to continue parsing if there is any.
 ****************************************************************************/
char *SpecialCommand(what2find,limit,rangeStart,rangeEnd)
	char	*what2find;		/* The input line. */
	long	*limit,			/* The maximum number of items to return. */
		*rangeStart,		/* The start of a range. */
		*rangeEnd;		/* The end of a range. */
{	short	theCommand;		/* Do we have a special command? */

	if (StillNeed2Parse(what2find,&theCommand,limit,rangeStart,rangeEnd))
		{
		if (theCommand)
			if (HandleSpecialCommand(what2find,theCommand))
				return((char *)NIL);
		}
	else if (theCommand)
		{
		(void)HandleSpecialCommand(what2find,theCommand);
		return((char *)NIL);
		}

	return(what2find);

}	/* SpecialCommand */
