/*****************************************************************************
 * File:	jugheadConf.c
 *
 * Author:	Rhett "Jonzy" Jones
 *		jonzy@cc.utah.edu
 *
 * Date:	June 6, 1994
 *
 * Modifed:	June 7, 1994 by Rhett "Jonzy" Jones.
 *		Forgot to add support for new-line '\n',
 *		carriage-return '\r', and tab '\t'.  Oops.
 *
 *		July 23, 1994, by Rhett "Jonzy" Jones.
 *		Added support for the remaining defines in the jughead.conf.h
 *		file.
 *
 *		July 24, 1994, by Rhett "Jonzy" Jones.
 *		Added the routine MakeInt().
 *
 *		September 20, 1994, by Rhett "Jonzy" Jones.
 *		Gave support for the -V flag which is used in conjunction
 *		with the -b flag to build a database file for veronica
 *		which does not contain any "Disallow" gopher paths.
 *
 *		September 22, 1994, by Rhett "Jonzy" Jones.
 *		Made ReadLine() no longer be static to help consolidate
 *		routines.
 *		Added the routine NullifyDisallowList() to prevent
 *		jughead from eating memory when running as search engine
 *		by constantly parsing the "Disallow:" directive from
 *		'jughead.conf'.
 *
 *		September 23, 1994, by Rhett "Jonzy" Jones.
 *		Fixed a problem in ParseLine() where a pound sign '#' in
 *		double quotes for a value in the configuration file was
 *		interpreted incorrectly as a comment.  Yipes!
 *		Added some better error testing, and now give warnings
 *		when things look a little strange.
 *
 *		September 14, 1994, by Rhett "Jonzy" Jones.
 *		Added token values and the respective strings to support
 *		better error handling when parsing the veronica.ctl file.
 *		Who knows, may want to use these in the future.
 *
 *		September 27, 1994, by Rhett "Jonzy" Jones.
 *		Added the routine HandleVeronicaToken() to parse the
 *		veronica.ctl token values slightly different than the
 *		jughead.conf values.
 *		Removed all use of rm.
 *
 * Description:	Reads information from the jughead.conf file.
 *		The format of this file is such that any text following
 *		the pound sign "#" is a comment, blank lines are ignored,
 *		and contains a variable name followed by an assignment.
 *		A maximum of 1024 character per line is allowed, as
 *		well as embedded quotes, quoted and non-quoted strings.
 *		A sample file follows:
 *		# <- Start of a comment
 *		hostname = some.machine.name  # Not quoted
 *		hostname "some.machine.name"  # Does the same thing
 *		var = this one and that one   # var now gets [this]
 *		var = "This one and that one" # var gets [the quoted string]
 *		var "A \"quoted\" word"       # var gets [A "quoted" word]
 *
 *		V: veronica token             # V gets [veronica token]
 *
 * Notes:	When adding variables for use by ReadConfFile() you need
 *		to do the following steps:
 *		Add the #define variable_number
 *		Add the #define variable_name
 *		Add the declaration of the variables
 *		Add the else if (!strcmp(...)) in ParseLine()
 *		Add the case variable_number in ReadConfFile()
 *
 * Routines:	       int  ReadLine(FILE *fp,char *line,int lineSize);
 *		static int  HandleVeronicaToken(int token,char *s,
 *						char **varStr);
 *		static int  ParseLine(char *line,char **varStr);
 *		       char *MakeStr(char *oldStr,char *newStr);
 *		       int  MakeInt(char *str);
 *		       void NullifyDisallowList(void);
 *		       int  ReadConfFile(char *fileName);
 *
 * Bugs:	No known bugs.
 *
 * Copyright:	Copyright 1993, 1994, University of Utah Computer Center.
 *		JUGHEAD (TM) is a trademark of Archie Comic Publications, Inc.
 *		and is used pursuant to license.  This source may be used and
 *		freely distributed as long as this copyright notice remains
 *		intact, and is in no way used for any monetary gain, by any
 *		institution, business, person, or persons.
 *
 ****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include "jughead.h"

#define BUFSIZE		1024

/* The variable-number to handle. */
#define COMMENTLINE		 -1
#define HOSTNAME		  1
#define JUGHEADHELP		  2
#define ERRORHOST		  3
#define PORT2USE		  4
#define MAXPROCS		  5
#define DELIMITERS		  6
#define MAXITEMS2RETURN		  7
#define DEFAULTBOOLOP		  8
#define JUGHEADHELPTITL		  9
#define USAGERROR		 10
#define WILDCARDERR		 11
#define TOMNYPRCSERR		 12
#define NOFORKERR		 13
#define GTSTRERR		 14
#define READERR			 15
#define TMPFILENAME		 16
#define CATCOMMAND		 17
#define SORTCOMMAND		 18
#define TOUCHCOMMAND		 19
#define DISALLOWCMD		 20

#ifdef VERSION2
#define EMAIL			 21
#define HOSTSTREE		 22
#endif

/* These aren't used by jughead but exist in the veronica.ctl file. */
#define USERAGENTCMD		100
#define VERONICAINDEXCMD	101
#define VERONCIADATAFILECMD	102
#define VERONICADATAFILEPREPCMD	103
#define GOPHEREOFCOMMAND	200

/* The name of variable_name to handle. */
#define COMMENTSTR		""
#define HOSTNAMESTR		"hostname"
#define JUGHEADHELPSTR		"jugheadhelp"
#define ERRORHOSTSTR		"errorhost"
#define PORT2USESTR		"port2use"
#define MAXPROCSSTR		"maxprocs"
#define DELIMITERSSTR		"delimiters"
#define MAXITEMS2RETURNSTR	"maxitems2return"
#define DEFAULTBOOLOPSTR	"defaultboolop"
#define JUGHEADHELPTITLSTR	"jugheadhelptitl"
#define USAGERRORSTR		"usagerror"
#define WILDCARDERRSTR		"wildcarderr"
#define TOMNYPRCSERRSTR		"tomnyprcserr"
#define NOFORKERRSTR		"noforkerr"
#define GTSTRERRSTR		"gtstrerr"
#define READERRSTR		"readerr"
#define TMPFILENAMESTR		"tmpfilename"
#define CATCOMMANDSTR		"catcommand"
#define SORTCOMMANDSTR		"sortcommand"
#define TOUCHCOMMANDSTR		"touchcommand"
#define DISALLOWSTR		"Disallow:"

#ifdef VERSION2
#define EMAILSTR		"email"
#define HOSTSTREESTR		"hoststree"
#endif

/* These aren't used by jughead but exist in the veronica.ctl file. */
#define USERAGENTSTR		"User-agent:"
#define VERONICAINDEXSTR	"veronicaindex:"
#define VERONCIADATAFILESTR	"veronica-datafile:"
#define VERONICADATAFILEPREPSTR	"veronica-datafile-prep:"
#define GOPHEREOFSTR		"."


/* These variables are defined in "jughead.c". */
extern int	debug;
extern FILE	*rptPtr;
extern int	ValidVariableCheck();
extern void	PrintJugheadConf();
extern List	*CreateNode(),
		*InsertNode(),
		*RemoveNode();

/* These are defined in "jugheadVctl.c". */
extern List	*disallowHead,
		*disallowTail;
extern char	*vctlHost,
		*vctlPort;
extern void	PrintTheList();

#ifdef VERSION2
extern char	*email,
		*hoststree;
#endif

/*****************************************************************************
 * ReadLine reads from the file 'fp' a maximum of 'lineSize' characters into
 * the string 'line' upto but not including end-of-file or end-of-line.
 * This routine returns the charcter read in.
 * Note:  If more them 'lineSize' character are found to exist on a line
 * a warning message is printed only once.
 ****************************************************************************/
int ReadLine(fp,line,lineSize)
	FILE	*fp;		/* Pointer to the file we are reading. */
	char	*line;		/* Buffer for the line we read. */
	int	lineSize;	/* The size of our line buffer. */
{	int	i,c;		/* A loop counter and a character. */

	/* Read the charcters up to the limit, eof-of-file or end-of-line. */
	for (i = 0; i < lineSize && (c = fgetc(fp)) != EOF && c != '\n'; i++)
		line[i] = (char)c;
	line[i] = '\0';

	if (debug)
		(void)fprintf(rptPtr,"%s\n",line);

	if (i >= lineSize)	/* Yipes, tell the user their file is out to lunch. */
		{
		static int	messagePosted = 0;	/* Have we told them yet? */

		if (!messagePosted)
			{
			messagePosted = 1;
			(void)fprintf(rptPtr,"warning: line length exceeds %d buffer.\n",lineSize + 1);
			(void)fprintf(rptPtr,"         All charcters beyond this limit are discarded.\n");
			}
		while ((c = fgetc(fp)) != EOF && c != '\n');
		}
	else if (i && c == EOF)
		(void)fprintf(rptPtr,"warning: No EOLN found, discarding [%s]\n",line);

	return(c);

}	/* ReadLine */

/*****************************************************************************
 * HandleVeronicaToken returns the veronica token of the variable
 * encountered. This routine also checks for a pound sign '#' in the string
 * 's'.  If a pound sign is encountered it is iterpreted as a comment and
 * all white space preceeding the pound sign is set to the null character.
 * The parameter 'value' ends up pointing to 's'.  This routine was written
 * to support the "veronica.ctl" commands which do not require double
 * quotes around a string which contains white space.
 ****************************************************************************/
static int HandleVeronicaToken(token,s,varStr)
	int	token;		/* The token type. */
	char	*s,		/* The string to put into 'varStr'. */
		**varStr;	/* The value for the token 'token'. */
{       char    *pound;		/* The pound position in 's'. */ 

	/* If there is a pound null it out. */
	if (pound = strchr(s,'#'))
		*pound = '\0';

	/* Null out all trailing isspace() characters. */
	for (pound = s + strlen(s) - 1; pound != s && isspace(*pound); pound--)
		*pound = '\0';

	*varStr = s;
	return(token);

}	/* HandleVeronicaToken */ 

/*****************************************************************************
 * ParseLine returns what type of variable we have encountered within the
 * jughead.conf or veronica.ctl file.  This routine assigns to 'varStr' the
 * contents of what the encountered variable should get assigned, as well as
 * takes care of embedded quotes.  The "Get the variable contents" is kind of
 * nitty gritty but it handles quoted strings, embedded quotes, and
 * non-quoted strings.  Note: if the token is a veronica.ctl token the
 * routine HandleVeronicaToken() parses the value for the given token.
 ****************************************************************************/
static int ParseLine(line,varStr)
	char	*line,		/* The line we are parsing. */
		**varStr;	/* The string to assign the variable. */
{	int	varName = 0,	/* What variable did we encounter? */
		i, j,		/* Loop and index counters. */
		comment,	/* Are we doing a comment? */
		done,		/* Are we done with string contents? */
		escape,		/* Have we encountered an escape character? */
		quote;		/* Have we seen a quote? */
	char	*s;		/* Place holder into 'line'. */

	/* Skip any leading whitespace. */
	while (isspace(*line))
		line++;

	if (!*line || *line == '#')	/* Comment line so bail out. */
		return(COMMENTLINE);

	for (s = line; *line && !isspace(*line) && *line != '=' && *line != '#'; line++)
		/* Get the name of the variable. */;

	/* See if we got the variable name. */
	if (isspace(*line) || *line == '=')
		{
		*line++ = '\0';	/* 's' now contains the variable name. */

		/* Find out which variable we are looking at. */
		if (!strcmp(s,COMMENTSTR))
			return(COMMENTLINE);
		else if (!strcmp(s,HOSTNAMESTR))
			varName = HOSTNAME;
		else if (!strcmp(s,JUGHEADHELPSTR))
			varName = JUGHEADHELP;
		else if (!strcmp(s,ERRORHOSTSTR))
			varName = ERRORHOST;
		else if (!strcmp(s,PORT2USESTR))
			varName = PORT2USE;
		else if (!strcmp(s,MAXPROCSSTR))
			varName = MAXPROCS;
		else if (!strcmp(s,DELIMITERSSTR))
			varName = DELIMITERS;
		else if (!strcmp(s,MAXITEMS2RETURNSTR))
			varName = MAXITEMS2RETURN;
		else if (!strcmp(s,DEFAULTBOOLOPSTR))
			varName = DEFAULTBOOLOP;
		else if (!strcmp(s,JUGHEADHELPTITLSTR))
			varName = JUGHEADHELPTITL;
		else if (!strcmp(s,USAGERRORSTR))
			varName = USAGERROR;
		else if (!strcmp(s,WILDCARDERRSTR))
			varName = WILDCARDERR;
		else if (!strcmp(s,TOMNYPRCSERRSTR))
			varName = TOMNYPRCSERR;
		else if (!strcmp(s,NOFORKERRSTR))
			varName = NOFORKERR;
		else if (!strcmp(s,GTSTRERRSTR))
			varName = GTSTRERR;
		else if (!strcmp(s,READERRSTR))
			varName = READERR;
		else if (!strcmp(s,TMPFILENAMESTR))
			varName = TMPFILENAME;
		else if (!strcmp(s,CATCOMMANDSTR))
			varName = CATCOMMAND;
		else if (!strcmp(s,SORTCOMMANDSTR))
			varName = SORTCOMMAND;
		else if (!strcmp(s,TOUCHCOMMANDSTR))
			varName = TOUCHCOMMAND;
		else if (!strcmp(s,DISALLOWSTR))
			return(HandleVeronicaToken(DISALLOWCMD,line,varStr));
#ifdef VERSION2
		else if (!strcmp(s,EMAILSTR))
			varName = EMAIL;
		else if (!strcmp(s,HOSTSTREESTR))
			varName = HOSTSTREE;
#endif
		else if (!strcmp(s,USERAGENTSTR))
			return(HandleVeronicaToken(USERAGENTCMD,line,varStr));
		else if (!strcmp(s,VERONICAINDEXSTR)) 
			return(HandleVeronicaToken(VERONICAINDEXCMD,line,varStr));
		else if (!strcmp(s,VERONCIADATAFILESTR))
			return(HandleVeronicaToken(VERONCIADATAFILECMD,line,varStr));
		else if (!strcmp(s,VERONICADATAFILEPREPSTR))
			return(HandleVeronicaToken(VERONICADATAFILEPREPCMD,line,varStr));
		else if (!strcmp(s,GOPHEREOFSTR))
			return(HandleVeronicaToken(GOPHEREOFCOMMAND,line,varStr));
		else	/* Who the hell knows? */
			{
			*varStr = line;
			return(0);
			}

		/* Skip any white space or equal signs. */
		while (isspace(*line) || *line == '=')
			line++;

		/* Get the variable contents. */
		for (comment = done = escape = quote = i = j = 0, s = line; !done && line[i] && !comment; i++)
			{
			if (line[i] == '#' && !quote)
				comment++;
			if (line[i] == '\\')
				{
				escape++;
				j++;
				}
			else if (!escape && !quote && isspace(line[i]))	
				{
				done++;
				j++;
				}
			else if (line[i] == '\"')
				if (escape)
					{
					escape = 0;
					line[i - j] = line[i];
					}
				else
					{
					if (quote)
						done++;
					else
						quote++;
					j++;
					}
			else if (escape && (line[i] == 'n' || line[i] == 'r' || line[i] == 't'))
				{
				escape = 0;
				switch (line[i])
					{
					case 'n': line[i - j] = '\n';	break;
					case 'r': line[i - j] = '\r';	break;
					case 't': line[i - j] = '\t';	break;
					}
				}
			else
				line[i - j] = line[i];
			}
		line[i - j] = '\0';	/* 's' now contains the variable contents. */

		*varStr = s;
		}
	else		/* Hmm, somthing strange is going on. */
		{
		*varStr = s;
		return(0);
		}

	return(varName);

}	/* ParseLine */

/*****************************************************************************
 * MakeStr returns a copy of 'newStr' acquired in malloc space.  This
 * routine also frees up any memory occupied by 'oldStr' prior to the copy.
 ****************************************************************************/
char *MakeStr(oldStr,newStr)
	char	*oldStr,	/* The string to copy into. */
		*newStr;	/* The string we are copying. */
{

	if (oldStr)	/* Free up the memory. */
		(void)free(oldStr);

	/* Get memory and copy the sucker. */
	if (oldStr = (char *)malloc((unsigned)strlen(newStr) * sizeof(char) + 1))
		(void)strcpy(oldStr,newStr);

	return(oldStr);

}	/* MakeStr */

/*****************************************************************************
 * MakeInt returns the integer representation of 'str'.
 ****************************************************************************/
int MakeInt(str)
	char	*str;	/* The string with the integer. */
{	int	i;	/* The value to return. */

	for (i = 0; *str; str++)
		if (isdigit(*str))
			i = i * 10 + *str - '0';
	return(i);

}	/* MakeInt */

/*****************************************************************************
 * NullifyDisallowList simply frees up the memory occupied by the
 * disallow list and sets the head and tail of the list to NIL.
 * This routine was written to support the "Disallow:" directive in the
 * 'jughead.conf' file and preventing jughead from overdoseing on too
 * much memory when running as a search engine and receiving all those
 * SIGHUP signals.
 ****************************************************************************/
void NullifyDisallowList()
{	List	*l;	/* A temp node in the list. */

	if (debug)
		PrintTheList();

	for (l = disallowTail; l; l = disallowTail)
		disallowTail = RemoveNode(l);
	disallowHead = disallowTail;

	if (debug)
		PrintTheList();

}	/* NullifyDisallowList */

/*****************************************************************************
 * ReadConfFile reads from the file 'fileName', which is the jughead.conf
 * file, and initializes the variables stored within the file.  This routine
 * returns true if the file exists and false otherwise.
 ****************************************************************************/
int ReadConfFile(fileName)
	char	*fileName;	/* Name of the jughead.conf file. */
{	FILE	*fp;		/* Pointer to the conf file. */
	char	line[BUFSIZE],	/* A line from the file. */
		*varStr;	/* The string variable from the file. */
	int	command;	/* The command token ParseLine() found. */

	if (debug)
		(void)fprintf(rptPtr,"In ReadConfFile(%s)\n",fileName);

	if (fp = fopen(fileName,"r"))
		{
		while (ReadLine(fp,line,BUFSIZE - 1) != EOF)
			switch (command = ParseLine(line,&varStr))
				{
				case COMMENTLINE:								break;
				case HOSTNAME:		hostname	= MakeStr(hostname,varStr);		break;
				case JUGHEADHELP:	jugheadhelp	= MakeStr(jugheadhelp,varStr);		break;
				case ERRORHOST:		errorhost	= MakeStr(errorhost,varStr);		break;
				case PORT2USE:		port2use	= MakeInt(varStr);			break;
				case MAXPROCS:		maxprocs	= MakeInt(varStr);			break;
				case DELIMITERS:	delimiters	= MakeStr(jugheadhelp,varStr);		break;
				case MAXITEMS2RETURN:	maxitems2return	= MakeInt(varStr);			break;
				case DEFAULTBOOLOP:	defaultboolop	= MakeStr(defaultboolop,varStr);	break;
				case JUGHEADHELPTITL:	jugheadhelptitl	= MakeStr(jugheadhelptitl,varStr);	break;
				case USAGERROR:		usagerror	= MakeStr(usagerror,varStr);		break;
				case WILDCARDERR:	wildcarderr	= MakeStr(wildcarderr,varStr);		break;
				case TOMNYPRCSERR:	tomnyprcserr	= MakeStr(tomnyprcserr,varStr);		break;
				case NOFORKERR:		noforkerr	= MakeStr(noforkerr,varStr);		break;
				case GTSTRERR:		gtstrerr	= MakeStr(gtstrerr,varStr);		break;
				case READERR:		readerr		= MakeStr(readerr,varStr);		break;
				case TMPFILENAME:	tmpfilename	= MakeStr(tmpfilename,varStr);		break;
				case CATCOMMAND:	catcommand	= MakeStr(catcommand,varStr);		break;
				case SORTCOMMAND:	sortcommand	= MakeStr(sortcommand,varStr);		break;
				case TOUCHCOMMAND:	touchcommand	= MakeStr(touchcommand,varStr);		break;
				case DISALLOWCMD:
					if (!disallowHead)
						disallowHead = disallowTail = CreateNode(varStr,vctlHost,vctlPort);
					else
						disallowTail = InsertNode(disallowTail,CreateNode(varStr,vctlHost,vctlPort));
					break;
#ifdef VERSION2
				case EMAIL:		email		= MakeStr(email,varStr);		break;
				case HOSTSTREE:		hoststree	= MakeStr(hoststree,varStr);		break;
#endif
				case USERAGENTCMD:
				case VERONICAINDEXCMD:
				case VERONCIADATAFILECMD:
				case VERONICADATAFILEPREPCMD:
				case GOPHEREOFCOMMAND:
					if (debug)
						{
						(void)fprintf(rptPtr,"Found veronica.ctl directive ");
						switch (command)
							{
							case USERAGENTCMD:
								(void)fprintf(rptPtr,"[%s]",USERAGENTSTR);
								break;
							case VERONICAINDEXCMD:
								(void)fprintf(rptPtr,"[%s]",VERONICAINDEXSTR);
								break;
							case VERONCIADATAFILECMD:
								(void)fprintf(rptPtr,"[%s]",VERONCIADATAFILESTR);
								break;
							case VERONICADATAFILEPREPCMD:
								(void)fprintf(rptPtr,"[%s]",VERONICADATAFILEPREPSTR);
								break;
							case GOPHEREOFCOMMAND:
								(void)fprintf(rptPtr,"(gopher end-of-file ) [%s]",GOPHEREOFSTR);
								break;
							default:
								break;
							}
						(void)fprintf(rptPtr," - ignoring.\n\tValue is: [%s]\n",varStr);
						}
					break;
				default:
					(void)fprintf(rptPtr,"warning: ParseLine found unknown directive [%s]\n",varStr);
					break;
				}
		(void)fclose(fp);
		}
	else
		(void)fprintf(rptPtr,"Could not open %s.  Using the default values.\n",fileName);

	if (debug)
		PrintJugheadConf();

	return(ValidVariableCheck());

}	/* ReadConfFile */
