#!/bin/csh -f
#########################################################################
# File:		jughead.sample.csh
#
# Date:		May 24, 1994.
#
# Author:	Rhett "Jonzy" Jones.
#
# Description:	Builds or rebuilds the jughead database and either
#		starts jughead as a server or tells jughead to reread
#		in the database depending on the existance of the
#		file /usr/local/etc/jughead.pid, and sends mail stating
#		the database has been rebuilt with listings of any
#		problems and some statistical information.
#		This script is intended to be run via cron.
#
# Usage:	jughead.sample.csh
#
# Notes:	You will need to make the appropriate changes
#		for your particular environment and needs.
#		Inparticular you need to set the following values:
#			DEBUG		1 = debugging
#			PS		Your local ps command
#			JUGHEAD		The jughead program
#			JUGHEADPID	The jughead.pid file
#			DATANAME	Name of the database
#			REPORTNAME	Name of the report file
#			STATS		Name of the rebuild history file
#			LOG		The search engine log file.
#			DATADIR		The data directory
#			TEMPDIR		Temporary directory for building
#			MAILTO		Who gets the mail
#			MAILSUBJECT	The email subject
#
#		And then edit the COLLECT, BUILD, and SEARCH blocks
#		#########################################################
#		#  COLLECT # COLLECT # COLLECT # COLLECT # COLLECT # COLL
#		#
#		$JUGHEAD ... Your jughead -b command
#		#########################################################
#
#		#########################################################
#		# BUILD # BUILD # BUILD # BUILD # BUILD # BUILD # BUILD #
#		#
#		$JUGHEAD ... Your jughead -B command
#		#########################################################
#
#		#########################################################
#		# SEARCH # SEARCH # SEARCH # SEARCH # SEARCH # SEARCH # S
#		#
#		#	$JUGHEAD ... Your jughead -S command
#		#########################################################
#
#		And then comment out or remove the lines begining with
#			echo
#			exit
#		There are (2) two of these
#
#		You might want to rename this file to something like:
#		jughead.csh
#
# Bugs:		If jughead is not running at the time this script is
#		started and there exists a process with the same id
#		as that recorded in the /usr/local/etc/jughead.pid file
#		well, ..., huh, ..., ah-oh.  Oops.
#
# Copyright:	Copyright 1993, 1994, University of Utah Computer Center.
#		JUGHEAD (TM) is a trademark of Archie Comic Publications, Inc.
#		and is used pursuant to license.  This source may be used and
#		freely distributed as long as this copyright notice remains
#		intact, and is in no way used for any monetary gain, by any
#		institution, business, person, or persons.
#
#########################################################################

# Set to 1 if you want debugging.
set DEBUG	= 0

# This is your local `ps` command
set PS		= "ps -aux"

# This is the location of your jughead program.
set JUGHEAD	= "/usr/local/etc/jughead"

# The jughead.pid file with pathway
set JUGHEADPID	= "/usr/local/etc/jughead.pid"

# This is the name of your jughead database.
set DATANAME	= "data"

# This is the name of your jughead rebuild report.
set REPORTNAME	= "rebuild.log"

# This is the name of your jughead record of rebuilds.
set STATS	= "/jughead.rebuild.stats"

# This is the name of the jughead search engine log.
set LOG		= "/cwis/info/gopher/gopherLog/jughead.log"

# This is where your jughead database resides.
set DATADIR	= "/jugdata"

# This is the location for rebuilding the jughead databases.
set TEMPDIR	= "/newdat"

# The rfc 822 email address of who gets the report.
set MAILTO	= "yourUserid@your.site"

# The subject for the email message.
set MAILSUBJECT	= "jughead rebuild info"


#########################################################################
# Comment the next 2 lines out after you have made the required changes
# for this script to funtion in your environment.
echo You need to read and edit this script for it to work.
exit
#########################################################################

#########################################################################
#   Edit the COLLECT BUILD and SEARCH blocks below to your particular   #
#               environment specifications and needs.                   #
#########################################################################

# Lets make sure jughead is even compiled and in place.
if (!( -x "$JUGHEAD")) then
  echo $JUGHEAD does not exist or permissions are wrong.
  exit
endif

# This step rebuilds the database.
if ($DEBUG == 1) then
  echo rebuilding the database
endif
#########################################################################
#  COLLECT # COLLECT # COLLECT # COLLECT # COLLECT # COLLECT # COLLECT  #
#
# This is the jughead call to collect your data
# My call looks like
# $JUGHEAD -Vb "$TEMPDIR/$DATANAME" -l "$TEMPDIR/$REPORTNAME" -X "ftp:*" .utah.edu gopher.utah.edu
# Make sure you comment out the next 2 lines once you set your call up.
echo You need to adjust the jughead COLLECT call
exit
$JUGHEAD -Vb "$TEMPDIR/$DATANAME" -l "$TEMPDIR/$REPORTNAME" -X "ftp:*" .your.domain gopher.your.domain
#########################################################################
if ($DEBUG == 1 && $status != 0) then
  echo error: jughead -b failed
  exit
endif


# This step rebuilds the tables.
if ($DEBUG == 1) then
  echo rebuilding the tables
endif
#########################################################################
# BUILD # BUILD # BUILD # BUILD # BUILD # BUILD # BUILD # BUILD # BUILD #
#
# This is the jughead call to Build your data tables
$JUGHEAD -Bml "$TEMPDIR/$REPORTNAME" "$TEMPDIR/$DATANAME"
#########################################################################
if ($DEBUG == 1 && $status != 0) then
  echo error: jughead -Bml failed
  exit
endif


# This step moves the databases into place.
if ($DEBUG == 1) then
  echo doing mv "$TEMPDIR/$DATANAME*" "$DATADIR/"
endif
/bin/mv $TEMPDIR/$DATANAME* "$DATADIR/"
if ($DEBUG == 1 && $status != 0) then
  echo error: /bin/mv failed
  exit
endif


# Append the report to the statistical file for history of jughead rebuilds.
if ($DEBUG == 1) then
  echo cating to $STATS
endif
/bin/cat "$TEMPDIR/$REPORTNAME" >> "$STATS"
if ($DEBUG == 1 && $status != 0) then
  echo error: /bin/cat failed
  exit
endif

# Either start jughead from scratch or kill HUP the sucker.
# See if the jughead.pid file exists.
if ( -f $JUGHEADPID ) then	# Yep it does.

  if ($DEBUG == 1) then
    echo the jughead.pid file exists
  endif

  # This is a kludge to see if jughead is currently running
  # with the process as specified on the first line of the
  # jughead.pid file.

  $PS > /tmp/jtmp1
  if ($DEBUG == 1) then
    echo greping for the process
    ls /tmp/jtmp1
  endif

  /bin/grep `/usr/ucb/head -1 $JUGHEADPID` /tmp/jtmp1 > /tmp/jtmp2

  # Now if /tmp/jtmp2 is 0 bytes jughead is not running
  # or if it is jughead did not create the file
  # /usr/local/etc/jughead.pid.  If jughead is not
  # running and there exists another process with the
  # pid as recoreded in the jughead.pid, well ah-oh.

  # Check file size of /tmp/jtmp2.
  if ($DEBUG == 1) then
    echo checking if the jughead process is running
  endif
  if ( -z /tmp/jtmp2 ) then	# No process running with the jughead-pid.

    if ($DEBUG == 1) then
      echo jughead is NOT running so starting jughead per jughead.pid
    endif

    # Make note that jughead is dead for some reason.
    echo jughead is NOT running so starting jughead per jughead.pid >> "$TEMPDIR/$REPORTNAME"

    # Restart jughead like the last time it was started.
    eval `/usr/ucb/tail -1 $JUGHEADPID`

  else	# jughead running or another process with the recorded jughead-pid is.

    # Check if we can SIGHUP the sucker
    $PS | grep $JUGHEAD | grep `whoami` >& /dev/null
    if ($status == 1) then
      if ($DEBUG == 1) then
        echo error: cannot SIGHUP jughead - does not look like your the owner
      endif
      echo not owner cannot SIGHUP jughead  >> "$TEMPDIR/$REPORTNAME"
    else
      if ($DEBUG == 1) then
        echo jughead is running so sending it SIGHUP
      endif

      kill -HUP `/usr/ucb/head -1 $JUGHEADPID`
    endif

  endif	# Checking file size of /tmp/jtmp2.

  # Get rid of the temp files for our kludge.
  /bin/rm -f /tmp/jtmp1 /tmp/jtmp2
else	# No jughead.pid file

  # Make note that jughead was started from scratch.
  if ($DEBUG == 1) then
    echo starting jughead from scratch
  endif
  echo starting jughead from scratch >> "$TEMPDIR/$REPORTNAME"
#########################################################################
# SEARCH # SEARCH # SEARCH # SEARCH # SEARCH # SEARCH # SEARCH # SEARCH #
#
# This is the call to start jugead as a search engine from scratch
$JUGHEAD -Sl "$LOG" "$DATADIR/$DATANAME"
#########################################################################

endif	# Existance of the jughead.pid file.

# Send off a note stating what happened when rebuilding jughead.

# Mail the results to the person concerned.
if ($DEBUG == 1) then
  echo mailing results to $MAILTO
endif
/usr/ucb/mail -s "$MAILSUBJECT" "$MAILTO" < "$TEMPDIR/$REPORTNAME"
if ($DEBUG == 1 && $status != 0) then
  echo error: /usr/ucb/mail failed
  exit
endif

# Now deleted the old report so it doesn't grow and get mailed.
if ($DEBUG == 1) then
  echo cleaning up
endif
/bin/rm -f "$TEMPDIR/$REPORTNAME"
if ($DEBUG == 1 && $status != 0) then
  echo error: /bin/rm failed
  exit
endif
