/*
 *  XmNap  A Motif napster client
 *  
 *  Copyright (C) 2000 Mats Peterson
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; see the file COPYING.  If not, write to
 *  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 *  
 *  Please send any comments/bug reports to
 *  matsp888@yahoo.com  (Mats Peterson)
 */

#include <Xm/Xm.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "main.h"
#include "connect.h"
#include "message.h"
#include "search.h"
#include "input.h"
#include "msgbox.h"
#include "hotlist.h"
#include "configwin.h"
#include "listwin.h"
#include "util.h"


HOTLIST *hotList = NULL;
HOTLIST *onLineList = NULL;

static Widget listWin = NULL;


void DestroyOnLineListWin(void)
{
    if (listWin) {
	DestroyWin(XtParent(listWin));
	listWin = NULL;
    }
}


void AddHotList(String nick, int manual)
{
    HOTLIST *newHot, *hotPtr, *prevPtr = NULL;
    char tmp[256];
    
    if (manual) {
	for (hotPtr = hotList; hotPtr; hotPtr = hotPtr->next) {
	    if (! strcmp(hotPtr->nick, nick))
		break;
	}
	if (hotPtr) {
	    sprintf(tmp, "User %s is already in your hotlist", nick);
	    ErrMsg(tmp);
	    return;
	}

	newHot = XtNew(HOTLIST);
	newHot->nick = XtNewString(nick);
    
	for (hotPtr = hotList; hotPtr; hotPtr = hotPtr->next) {
	    if (strcasecmp(hotPtr->nick, newHot->nick) > 0)
		break;
	    prevPtr = hotPtr;
	}

	if (prevPtr) {
	    newHot->next = prevPtr->next;
	    prevPtr->next = newHot;
	} else {
	    newHot->next = hotList;
	    hotList = newHot;
	}
    }

    if (srvConn) {
	if (SendMsg(MSG_CLIENT_ADD_HOTLIST, nick))
	    Disconnect(strerror(errno));
    }
}


void RemoveHotList2(int n)
{
    HOTLIST *hot, *prevHot = NULL;
    int i;
    
    for (i = 0, hot = hotList; i < n; i++, hot = hot->next)
	prevHot = hot;

    if (srvConn) {
	if (SendMsg(MSG_CLIENT_REMOVE_HOTLIST, hot->nick))
	    Disconnect(strerror(errno));
    }

    ClearOnLine(hot->nick);
    
    if (prevHot)
	prevHot->next = hot->next;
    else 
	hotList = hotList->next;
    XtFree(hot->nick);
    XtFree((char*)hot);
}


void RemoveHotList(String nick)
{
    HOTLIST *hot, *prevHot = NULL;
    char tmp[256];

    for (hot = hotList; hot; hot = hot->next) {
	if (! strcmp(hot->nick, nick))
	    break;
	prevHot = hot;
    }

    if (! hot) {
	sprintf(tmp, "User %s is not in your hotlist", nick);
	ErrMsg(tmp);
	return;
    }

    if (srvConn) {
	if (SendMsg(MSG_CLIENT_REMOVE_HOTLIST, hot->nick))
	    Disconnect(strerror(errno));
    }

    ClearOnLine(hot->nick);
    
    if (prevHot)
	prevHot->next = hot->next;
    else
	hotList = hotList->next;
    XtFree(hot->nick);
    XtFree((char*)hot);
}


void UserSignOn(String data)
{
    HOTLIST *newOnLine, *onLinePtr, *prevPtr = NULL;
    char tmp[256];
    
    newOnLine = XtNew(HOTLIST);
    newOnLine->nick = XtNewString((String)strtok(data, " "));
    newOnLine->next = NULL;
    
    for (onLinePtr = onLineList; onLinePtr; onLinePtr = onLinePtr->next) {
	if (strcasecmp(onLinePtr->nick, newOnLine->nick) > 0)
	    break;
	prevPtr = onLinePtr;
    }

    if (prevPtr) {
	newOnLine->next = prevPtr->next;
	prevPtr->next = newOnLine;
    } else {
	newOnLine->next = onLineList;
	onLineList = newOnLine;
    }

    sprintf(tmp, "%s has signed on", newOnLine->nick);
    ShowMiscInfo(tmp, 0);
}


void UserSignOff(String data)
{
    char tmp[256];

    ClearOnLine(data);
    sprintf(tmp, "%s has signed off", data);
    ShowMiscInfo(tmp, 0);
}


static void OnLineListCB(Widget w, XtPointer clientData,
	XmSelectionBoxCallbackStruct *cbs)
{
    String nick, msg, tmp = XtMalloc(1024);
    
    switch (cbs->reason) {
	case XmCR_OK:
	    XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &nick);
	    Browse(nick);
	    break;
	case XmCR_APPLY:
	    XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &nick);
	    if (! strlen(nick))
		goto end;
	    msg = GetInput("Message", "", 40);
	    if (! strlen(msg))
		goto end;
	    sprintf(tmp, "%s %s", nick, msg);
	    if (SendMsg(MSG_CLIENT_PRIVMSG, tmp))
		Disconnect(strerror(errno));
	    break;
	case XmCR_CANCEL:
	case XmCR_PROTOCOLS:
	    DestroyOnLineListWin();
    }

end:
    XtFree(nick);
    XtFree(tmp);
}


void ListOnLine(void)
{
    HOTLIST *onLine;
    int i, numOnLine;
    String tmp = XtMalloc(8192);
    XmString *str;

    DestroyOnLineListWin();
    if (! onLineList) {
	InfoMsg("No hotlist users online");
	goto end;
    }
 
    listWin = ListWin("onLineList", (XtCallbackProc)OnLineListCB, NULL);
    
    for (numOnLine = 0, onLine = onLineList; onLine;
	 numOnLine++, onLine = onLine->next);
    
    str = (XmString*)XtMalloc(numOnLine * sizeof(XmString));
    for (i = 0, onLine = onLineList; i < numOnLine;
	    i++, onLine = onLine->next) {
        str[i] = XmStringCreateLocalized(onLine->nick);
    }

    XtVaSetValues(listWin,
	    XmNlistItems, str,
	    XmNlistItemCount, numOnLine,
	    NULL);
    
    for (i = 0; i < numOnLine; i++)
        XmStringFree (str[i]);
    XtFree((String)str);

end:
    XtFree(tmp);
}


void ClearOnLine(String nick)
{
    HOTLIST *onLine, *prevOnLine = NULL;

    for (onLine = onLineList; onLine; onLine = onLine->next) {
	if (! strcmp(onLine->nick, nick))
	    break;
	prevOnLine = onLine;
    }
    if (! onLine)
	return;

    if (prevOnLine)
	prevOnLine->next = onLine->next;
    else 
	onLineList = onLineList->next;
    XtFree(onLine->nick);
    XtFree((char*)onLine);
}


void ClearOnLineList(void)
{
    String p;
    
    while (onLineList) {
	XtFree(onLineList->nick);
	p = (String)onLineList;
	onLineList = onLineList->next;
	XtFree(p);
    }
}


void SendHotList(void)
{
    HOTLIST *hot;

    for (hot = hotList; hot; hot = hot->next)
	AddHotList(hot->nick, 0);
}


void ReadHotList(FILE *fd)
{
    char tmp[80];
    int i, numHot;
    
    fscanf(fd, "%d\n", &numHot);
    if (! numHot)
	return;
    
    for (i = 0; i < numHot; i++) {
	fgets(tmp, 80, fd);
	tmp[strlen(tmp) - 1] = 0;
	AddHotList(tmp, 1);
    }
}


void SaveHotList(FILE *fd)
{
    HOTLIST *hot;
    int i;
    
    for (hot = hotList, i = 0; hot; hot = hot->next, i++);
    fprintf(fd, "%d\n", i);
    if (! i)
	return;
     
    for (hot = hotList; hot; hot = hot->next)
	fprintf(fd, "%s\n", hot->nick);
}
