/**
 * Juke  A simple jukebox for midi and rmf files.
 */

import java.awt.Color;
import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.filechooser.*;

import javax.sound.midi.*;
import java.io.File;
import java.io.InputStream;
import java.io.FileInputStream;
import java.io.BufferedInputStream;
import java.util.Arrays;
import java.util.Vector;
import java.net.URL;


public class Juke extends JFrame implements Runnable, ActionListener,
         MetaEventListener {
    final String version = "0.1";
    JLabel text, clicked;
    JButton [] buttons;
    JPanel panel, buttonPanel;
    JFileChooser chooser;
    Vector files = new Vector();
    Sequencer sequencer;
    boolean midiEOM;
    Thread thread = null;

    
    public Juke() {
	setTitle("Juke v" + version);

	URL url = Juke.class.getResource("Juke.gif");
	ImageIcon image = new ImageIcon(url);
	setIconImage(image.getImage());

	buttons = new JButton[4];
	String [] buttonText = {"Play", "Play Dir", "Stop", "Quit"};
	int [] keyEvent = {
	    KeyEvent.VK_P, KeyEvent.VK_D, KeyEvent.VK_S, KeyEvent.VK_Q};
	for (int i = 0; i < 4; i++) {
	    buttons[i] = new JButton(buttonText[i]);
	    buttons[i].setMnemonic(keyEvent[i]);
	    buttons[i].addActionListener(this);
	}

	panel = new JPanel();
	panel.setLayout(new GridLayout(1, 0, 10, 0));
	panel.setBorder(BorderFactory.createEmptyBorder(
	    10, //top
	    10, //left
	    10, //bottom
	    10) //right
	    );

	for (int i = 0; i < 4; i++)
	    panel.add(buttons[i]);

	getContentPane().add(panel);

	chooser = new JFileChooser();
	String [] extensions = {"mid", "rmf"};
	JukeFileFilter filter =
	    new JukeFileFilter(extensions, "MIDI & RMF Files");
	chooser.setFileFilter(filter);

	initJuke();
    }


    private String getFile(int mode) {
	int retval = chooser.showOpenDialog(this);
	if(retval == JFileChooser.APPROVE_OPTION) {
	    if (mode == 0) {
		File theFile = chooser.getSelectedFile();
		return(theFile.getAbsolutePath());
	    } else {
		return((chooser.getCurrentDirectory()).getAbsolutePath());
	    }
	} else
	    return null;
    }

    
    public void actionPerformed(ActionEvent event) {
        JButton button = (JButton)event.getSource();
				
	if (button == buttons[0]) {
	    String fileName = getFile(0);
	    if (fileName != null) {
		if (thread != null) {
		    paintAll(getGraphics());
		    stop();
		    try {
			Thread.sleep(1000);
		    } catch (Exception e) {}
		}
		loadJuke(fileName);
		start();
	    }
	} else if (button == buttons[1]) {
	    String fileName = getFile(1);
	    if (fileName != null) {
		if (thread != null) {
		    paintAll(getGraphics());
		    stop();
		    try {
			Thread.sleep(1000);
		    } catch (Exception e) {}
		}
		loadJuke(fileName);
		start();
	    }
	} else if (button == buttons[2]) {
	    buttons[2].paint(buttons[2].getGraphics());
	    if (thread != null)
		stop();
	} else {
	    System.exit(0);
	}
    }
    
    
    private void initJuke() {
        try {
            sequencer = MidiSystem.getSequencer();
        } catch (Exception ex) { ex.printStackTrace(); return; }
        sequencer.addMetaEventListener(this);
    }

    
    private void loadJuke(String name) {
	files.clear();
        try {
	    File file = new File(name);
            if (file != null && file.isDirectory()) {
                String files[] = file.list();
		Arrays.sort(files);
                for (int i = 0; i < files.length; i++) {
                    File leafFile = new File(file.getAbsolutePath(), files[i]);
                    if (leafFile.isDirectory()) {
                        loadJuke(leafFile.getAbsolutePath());
                    } else {
                        addFile(leafFile);
                    }
                }
            } else if (file != null && file.exists()) {
                addFile(file);
            }
        } catch (SecurityException ex) {
            System.out.println(ex.toString());
        } catch (Exception ex) {
            System.out.println(ex.toString());
        }
    }

    
    private void addFile(File file) {
	String s = file.getName().toLowerCase();
	if (s.endsWith(".mid") || s.endsWith(".midi") ||
		s.endsWith(".smf") || s.endsWith(".mff") ||
		s.endsWith(".rmf"))
	    files.add(file);
    }

    
    private boolean loadFile(File file) {
	BufferedInputStream stream;

//	System.out.println(file.getName());
	
	try { 
	    FileInputStream is = new FileInputStream(file);
	    stream = new BufferedInputStream(is, 1024);
	} catch (Exception e3) { 
	    e3.printStackTrace(); 
	    return false;
	}

	try {
	    sequencer.open();
	    sequencer.setSequence(stream);
	} catch (InvalidMidiDataException imde) { 
	    System.out.println("Unsupported audio file.");
	    return false;
	} catch (Exception ex) {
	    ex.printStackTrace();
	    return false;
	}
	
	return true;
    }


    private boolean playFile() {
	boolean retStatus = true;
	
	midiEOM = false;
	sequencer.start();
	while (! midiEOM) {
	    try {thread.sleep(99);} catch (Exception e) {
		retStatus = false;
		break;
	    }
	    if (thread == null) {
		retStatus = false;
		break;
	    }
	}
	sequencer.stop();
	sequencer.close();
	return retStatus;
    }
    
    
    public void meta(MetaMessage message) {
	if (message.getType() == 47) {  // 47 is end of track
             midiEOM = true;
        }
    }


    public void start() {
 	thread = new Thread(this);
	thread.start();
    }


    public void stop() {
	if (thread != null) {
            thread.interrupt();
	    thread = null;
	}
    }


    public void run() {
	while (true) {
	    for (int num = 0; num < files.size(); num++) {
		if(loadFile((File)files.get(num))) {
		    if (! playFile())
			return;
		}
		try {thread.sleep(1000);} catch (Exception e) {return;}
	    }
	}
    }


    public static void main(String[] args){
        try {
            UIManager.setLookAndFeel(
                UIManager.getCrossPlatformLookAndFeelClassName());
        } catch (Exception e) { }

	Juke frame = new Juke();
	WindowListener l = new WindowAdapter() {
		public void windowClosing(WindowEvent e) {
		    System.exit(0);
		}
	};
	frame.addWindowListener(l);
	frame.pack();
	frame.setLocation(100, 100);
	frame.setVisible(true);
    }
}
