/******************************************************************************/
/* dtm2xml.c                                                                 */
/* Copyright (C) 2003, S.C. Kremer <stefan@kremer.ca>                         */
/* Copyright (C) 2004, H.N. Schaller <hns@computer.org> for bug fixing and extensions */
/*                                                                            */
/* This program is free software; you can redistribute it and/or modify       */
/* it under the terms of the GNU General Public License as published by       */
/* the Free Software Foundation; either version 2 of the License, or          */
/* (at your option) any later version.  This program is distributed in        */
/* the hope that it will be useful, but WITHOUT ANY WARRANTY; without         */
/* even the implied warranty of MERCHANTABILITY or FITNESS FOR A              */
/* PARTICULAR PURPOSE.  See the GNU General Public License for more           */
/* details; it is available at <http://www.fsf.org/copyleft/gpl.html>,        */
/* or by writing to the Free Software Foundation, Inc., 59 Temple Place       */
/* - Suite 330, Boston, MA 02111-1307, USA.                                   */
/******************************************************************************/

/******************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <strings.h>
#include <ctype.h>
#include <stdio.h>
#define  __USE_POSIX	// load PATH_MAX etc.
#include <limits.h>
#include <time.h>

#include <dtmlib.h>

#if defined(__cplusplus)
extern "C" {
#endif
	extern "C" int dtmreset(void);
#if defined(__cplusplus)
}
#endif

#define  BUFFERSIZE (10*1024)	// 10k standard limit

#define  INCLUDE_STRPTIME 1
// taken from NetBSD

#define  TM_YEAR_BASE 1900	// defined by POSIX

#if INCLUDE_STRPTIME

/*-
* Copyright (c) 1997, 1998 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code was contributed to The NetBSD Foundation by Klaus Klein.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by the NetBSD
 *        Foundation, Inc. and its contributors.
 * 4. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
						  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
						  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * We do not implement alternate representations. However, we always
 * check whether a given modifier is allowed for a certain conversion.
 */
#define ALT_E			0x01
#define ALT_O			0x02
#define LEGAL_ALT(x)		{ if (alt_format & ~(x)) return (0); }


static	int conv_num(const char **, int *, int, int);

static const char *day[7] = {
	"Sunday", "Monday", "Tuesday", "Wednesday", "Thursday",
	"Friday", "Saturday"
};
static const char *abday[7] = {
	"Sun","Mon","Tue","Wed","Thu","Fri","Sat"
};
static const char *mon[12] = {
	"January", "February", "March", "April", "May", "June", "July",
	"August", "September", "October", "November", "December"
};
static const char *abmon[12] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};
static const char *am_pm[2] = {
	"AM", "PM"
};


char *
strptime(const char *buf, const char *fmt, struct tm *tm)
{
	char c;
	const char *bp;
	size_t len = 0;
	int alt_format, i, split_year = 0;
	
	bp = buf;
	
	while ((c = *fmt) != '\0') {
		/* Clear `alternate' modifier prior to new conversion. */
		alt_format = 0;
		
		/* Eat up white-space. */
		if (isspace(c)) {
			while (isspace(*bp))
				bp++;
			
			fmt++;
			continue;
		}
		
		if ((c = *fmt++) != '%')
			goto literal;
		
		
again:		switch (c = *fmt++) {
	case '%':	/* "%%" is converted to "%". */
		literal:
		if (c != *bp++)
			return (0);
		break;
		
		/*
		 * "Alternative" modifiers. Just set the appropriate flag
		 * and start over again.
		 */
	case 'E':	/* "%E?" alternative conversion modifier. */
		LEGAL_ALT(0);
		alt_format |= ALT_E;
		goto again;
		
	case 'O':	/* "%O?" alternative conversion modifier. */
		LEGAL_ALT(0);
		alt_format |= ALT_O;
		goto again;
		
		/*
		 * "Complex" conversion rules, implemented through recursion.
		 */
	case 'c':	/* Date and time, using the locale's format. */
		LEGAL_ALT(ALT_E);
		if (!(bp = strptime(bp, "%x %X", tm)))
			return (0);
			break;
		
	case 'D':	/* The date as "%m/%d/%y". */
		LEGAL_ALT(0);
		if (!(bp = strptime(bp, "%m/%d/%y", tm)))
			return (0);
			break;
		
	case 'R':	/* The time as "%H:%M". */
		LEGAL_ALT(0);
		if (!(bp = strptime(bp, "%H:%M", tm)))
			return (0);
			break;
		
	case 'r':	/* The time in 12-hour clock representation. */
		LEGAL_ALT(0);
		if (!(bp = strptime(bp, "%I:%M:%S %p", tm)))
			return (0);
			break;
		
	case 'T':	/* The time as "%H:%M:%S". */
		LEGAL_ALT(0);
		if (!(bp = strptime(bp, "%H:%M:%S", tm)))
			return (0);
			break;
		
	case 'X':	/* The time, using the locale's format. */
		LEGAL_ALT(ALT_E);
		if (!(bp = strptime(bp, "%H:%M:%S", tm)))
			return (0);
			break;
		
	case 'x':	/* The date, using the locale's format. */
		LEGAL_ALT(ALT_E);
		if (!(bp = strptime(bp, "%m/%d/%y", tm)))
			return (0);
			break;
		
		/*
		 * "Elementary" conversion rules.
		 */
	case 'A':	/* The day of week, using the locale's form. */
	case 'a':
		LEGAL_ALT(0);
		for (i = 0; i < 7; i++) {
			/* Full name. */
			len = strlen(day[i]);
			if (strncasecmp(day[i], bp, len) == 0)
				break;
			
			/* Abbreviated name. */
			len = strlen(abday[i]);
			if (strncasecmp(abday[i], bp, len) == 0)
				break;
		}
			
			/* Nothing matched. */
			if (i == 7)
				return (0);
			
			tm->tm_wday = i;
		bp += len;
		break;
		
	case 'B':	/* The month, using the locale's form. */
	case 'b':
	case 'h':
		LEGAL_ALT(0);
		for (i = 0; i < 12; i++) {
			/* Full name. */
			len = strlen(mon[i]);
			if (strncasecmp(mon[i], bp, len) == 0)
				break;
			
			/* Abbreviated name. */
			len = strlen(abmon[i]);
			if (strncasecmp(abmon[i], bp, len) == 0)
				break;
		}
			
			/* Nothing matched. */
			if (i == 12)
				return (0);
			
			tm->tm_mon = i;
		bp += len;
		break;
		
	case 'C':	/* The century number. */
		LEGAL_ALT(ALT_E);
		if (!(conv_num(&bp, &i, 0, 99)))
			return (0);
			
			if (split_year) {
				tm->tm_year = (tm->tm_year % 100) + (i * 100);
			} else {
				tm->tm_year = i * 100;
				split_year = 1;
			}
				break;
		
	case 'd':	/* The day of month. */
	case 'e':
		LEGAL_ALT(ALT_O);
		if (!(conv_num(&bp, &tm->tm_mday, 1, 31)))
			return (0);
			break;
		
	case 'k':	/* The hour (24-hour clock representation). */
		LEGAL_ALT(0);
		/* FALLTHROUGH */
	case 'H':
		LEGAL_ALT(ALT_O);
		if (!(conv_num(&bp, &tm->tm_hour, 0, 23)))
			return (0);
			break;
		
	case 'l':	/* The hour (12-hour clock representation). */
		LEGAL_ALT(0);
		/* FALLTHROUGH */
	case 'I':
		LEGAL_ALT(ALT_O);
		if (!(conv_num(&bp, &tm->tm_hour, 1, 12)))
			return (0);
			if (tm->tm_hour == 12)
				tm->tm_hour = 0;
				break;
		
	case 'j':	/* The day of year. */
		LEGAL_ALT(0);
		if (!(conv_num(&bp, &i, 1, 366)))
			return (0);
			tm->tm_yday = i - 1;
		break;
		
	case 'M':	/* The minute. */
		LEGAL_ALT(ALT_O);
		if (!(conv_num(&bp, &tm->tm_min, 0, 59)))
			return (0);
			break;
		
	case 'm':	/* The month. */
		LEGAL_ALT(ALT_O);
		if (!(conv_num(&bp, &i, 1, 12)))
			return (0);
			tm->tm_mon = i - 1;
		break;
		
	case 'p':	/* The locale's equivalent of AM/PM. */
		LEGAL_ALT(0);
		/* AM? */
		if (strcasecmp(am_pm[0], bp) == 0) {
			if (tm->tm_hour > 11)
				return (0);
			
			bp += strlen(am_pm[0]);
			break;
		}
			/* PM? */
			else if (strcasecmp(am_pm[1], bp) == 0) {
				if (tm->tm_hour > 11)
					return (0);
				
				tm->tm_hour += 12;
				bp += strlen(am_pm[1]);
				break;
			}
			
			/* Nothing matched. */
			return (0);
		
	case 'S':	/* The seconds. */
		LEGAL_ALT(ALT_O);
		if (!(conv_num(&bp, &tm->tm_sec, 0, 61)))
			return (0);
			break;
		
	case 'U':	/* The week of year, beginning on sunday. */
	case 'W':	/* The week of year, beginning on monday. */
		LEGAL_ALT(ALT_O);
		/*
		 * XXX This is bogus, as we can not assume any valid
		 * information present in the tm structure at this
		 * point to calculate a real value, so just check the
		 * range for now.
		 */
		if (!(conv_num(&bp, &i, 0, 53)))
			return (0);
			break;
		
	case 'w':	/* The day of week, beginning on sunday. */
		LEGAL_ALT(ALT_O);
		if (!(conv_num(&bp, &tm->tm_wday, 0, 6)))
			return (0);
			break;
		
	case 'Y':	/* The year. */
		LEGAL_ALT(ALT_E);
		if (!(conv_num(&bp, &i, 0, 9999)))
			return (0);
			
			tm->tm_year = i - TM_YEAR_BASE;
#if 0
		printf("Year=%d\n", tm->tm_year);
#endif
		break;
		
	case 'y':	/* The year within 100 years of the epoch. */
		LEGAL_ALT(ALT_E | ALT_O);
		if (!(conv_num(&bp, &i, 0, 99)))
			return (0);
			
			if (split_year) {
				tm->tm_year = ((tm->tm_year / 100) * 100) + i;
				break;
			}
				split_year = 1;
		if (i <= 68)
			tm->tm_year = i + 2000 - TM_YEAR_BASE;
		else
			tm->tm_year = i + 1900 - TM_YEAR_BASE;
		break;
		
		/*
		 * Miscellaneous conversions.
		 */
	case 'n':	/* Any kind of white-space. */
	case 't':
		LEGAL_ALT(0);
		while (isspace(*bp))
			bp++;
			break;
		
		
	default:	/* Unknown/unsupported conversion. */
		return (0);
}


	}

/* LINTED functional specification */
return ((char *)bp);
}


static int
conv_num(const char **buf, int *dest, int llim, int ulim)
{
	int result = 0;
	
	/* The limit also determines the number of valid digits. */
	int rulim = ulim;
	
	if (**buf < '0' || **buf > '9')
		return (0);
	
	do {
		result *= 10;
		result += *(*buf)++ - '0';
		rulim /= 10;
	} while ((result * 10 <= ulim) && rulim && **buf >= '0' && **buf <= '9');
	
	if (result < llim || result > ulim)
		return (0);
	
	*dest = result;
	return (1);
}

#endif

/******************************************************************************/
/* structure mapping a DTM Parameter ITEMNAME to a function that pretty       */
/* prints its contents                                                        */

struct PrintMethod
{
  char *name;                                       /* ITEMNAME */
  void (*fn_ptr)( unsigned long len, char *data );  /* point to print fn */
};

/******************************************************************************/
/* pretty print function prototypes (need these here to forward refernce      */
/* mapping, below)                                                            */

void print_unsupported( unsigned long len, char *data );
void print_bit        ( unsigned long len, char *data );
void print_time       ( unsigned long len, char *data );
void print_ulong      ( unsigned long len, char *data );
void print_barray     ( unsigned long len, char *data );
void print_uchar      ( unsigned long len, char *data );
void print_word       ( unsigned long len, char *data );
void print_utf8       ( unsigned long len, char *data );
void print_category   ( unsigned long len, char *data );

/******************************************************************************/
/* array of mappings                                                          */
/* for efficency this could be sorted alphabetically and then searched using  */
/* bsearch, but it seems to run fast enough as is                             */

struct PrintMethod print_method[] =
{
  { "ATTR", print_bit    }, /* common */
  { "CTTM", print_time   }, /* common */
  { "MDTM", print_time   }, /* common */
  { "SYID", print_ulong  }, /* common */
  { "CTGR", print_category }, /* Category list/bits - ToDo, Calendar, Addressbook */
  { "ETDY", print_time   }, /* ToDo */
  { "LTDY", print_time   }, /* ToDo */
  { "FNDY", print_time   }, /* ToDo */
  { "MARK", print_uchar  }, /* ToDo */
  { "PRTY", print_uchar  }, /* ToDo */
  { "TITL", print_utf8   }, /* ToDo */
  { "MEM1", print_utf8   }, /* ToDo, Calendar */
  { "DSRP", print_utf8   }, /* Calendar */
  { "PLCE", print_utf8   }, /* Calendar */
  { "TIM1", print_time   }, /* Calendar */
  { "TIM2", print_time   }, /* Calendar */
  { "ADAY", print_uchar  }, /* Calendar */
  { "ARON", print_uchar  }, /* Calendar */
  { "ARSD", print_uchar  }, /* Calendar */
  { "ARMN", print_word   }, /* Calendar */
  { "RTYP", print_uchar  }, /* Calendar */
  { "RFRQ", print_word   }, /* Calendar */
  { "RPOS", print_word   }, /* Calendar */
  { "RDYS", print_uchar  }, /* Calendar */
  { "REND", print_uchar  }, /* Calendar */
  { "REDT", print_time   }, /* Calendar */
  { "ALSD", print_time   }, /* Calendar */
  { "ALED", print_time   }, /* Calendar */
  { "MDAY", print_uchar  }, /* Calendar */
  { "FULL", print_utf8   }, /* Addressbook */
  { "NAPR", print_utf8   }, /* Addressbook */
  { "TITL", print_utf8   }, /* Addressbook */
  { "LNME", print_utf8   }, /* Addressbook */
  { "FNME", print_utf8   }, /* Addressbook */
  { "MNME", print_utf8   }, /* Addressbook */
  { "SUFX", print_utf8   }, /* Addressbook */
  { "FLAS", print_utf8   }, /* Addressbook */
  { "LNPR", print_utf8   }, /* Addressbook */
  { "FNPR", print_utf8   }, /* Addressbook */
  { "CPNY", print_utf8   }, /* Addressbook */
  { "CPPR", print_utf8   }, /* Addressbook */
  { "SCTN", print_utf8   }, /* Addressbook */
  { "PSTN", print_utf8   }, /* Addressbook */
  { "TEL2", print_utf8   }, /* Addressbook */
  { "FAX2", print_utf8   }, /* Addressbook */
  { "CPS2", print_utf8   }, /* Addressbook */
  { "BSTA", print_utf8   }, /* Addressbook */
  { "BCTY", print_utf8   }, /* Addressbook */
  { "BSTR", print_utf8   }, /* Addressbook */
  { "BZIP", print_utf8   }, /* Addressbook */
  { "BCTR", print_utf8   }, /* Addressbook */
  { "BWEB", print_utf8   }, /* Addressbook */
  { "OFCE", print_utf8   }, /* Addressbook */
  { "PRFS", print_utf8   }, /* Addressbook */
  { "ASST", print_utf8   }, /* Addressbook */
  { "MNGR", print_utf8   }, /* Addressbook */
  { "PRFS", print_utf8   }, /* Addressbook */
  { "ASST", print_utf8   }, /* Addressbook */
  { "MNGR", print_utf8   }, /* Addressbook */
  { "BPGR", print_utf8   }, /* Addressbook */
  { "CPS1", print_utf8   }, /* Addressbook */
  { "TEL1", print_utf8   }, /* Addressbook */
  { "FAX1", print_utf8   }, /* Addressbook */
  { "HSTA", print_utf8   }, /* Addressbook */
  { "HCTY", print_utf8   }, /* Addressbook */
  { "HSTR", print_utf8   }, /* Addressbook */
  { "HZIP", print_utf8   }, /* Addressbook */
  { "HCTR", print_utf8   }, /* Addressbook */
  { "HWEB", print_utf8   }, /* Addressbook */
  { "DMAL", print_utf8   }, /* Addressbook */
  { "MAL1", print_utf8   }, /* Addressbook */
  { "SPUS", print_utf8   }, /* Addressbook */
  { "GNDR", print_utf8   }, /* Addressbook */
  { "BRTH", print_utf8   }, /* Addressbook */
  { "ANIV", print_utf8   }, /* Addressbook */
  { "NCNM", print_utf8   }, /* Addressbook */
  { "CLDR", print_utf8   }, /* Addressbook */
  { "MEM1", print_utf8   }, /* Addressbook */
  { "GRPS", print_utf8   }, /* Addressbook */
  { "SCTG", print_uchar	}, /* Category */
  { "TYPE", print_uchar }, /* Category */
  { "COLR", print_ulong }, /* Category */
  { "CLAS", print_utf8 }, /* Memo */
  { "DATE", print_time }, /* Memo */
/* all for EMail (outbox)
Don't know how to print MLID
Don't know how to print SIZE
Don't know how to print RDCK
Don't know how to print DLCK
Don't know how to print DECK
Don't know how to print SVID
Don't know how to print SDCK
Don't know how to print MDCK
Don't know how to print DRFT
Don't know how to print SLCK
Don't know how to print ACNM
Don't know how to print MLBX
Don't know how to print RVTM
Don't know how to print SDTM
Don't know how to print TMZN
Don't know how to print FMAC
Don't know how to print FMNM
Don't know how to print FMAD
Don't know how to print MLTO
Don't know how to print MLCC
Don't know how to print MLBC
Don't know how to print MLRP
Don't know how to print SBJT
Don't know how to print BODY
Don't know how to print HTML
Don't know how to print ENNM
Don't know how to print ENAT
Don't know how to print ENTP
Don't know how to print ENFL
Don't know how to print DMMY
*/
  { NULL, NULL }
};

/******************************************************************************/
/* this function supports the printing of unsupported data, it is used for    */
/* development/debugging purposes only                                        */
/* it prints out the data as series of bits (with blanks after every 8)       */

void print_unsupported( unsigned long len, char *data )
{
  unsigned long i;
  int bit, val;
  unsigned char x;

  for (i=0;i<len;i++)
  {
    if (i) 
      printf(" ");
    for (bit=7;bit>=0;bit--)
    {
      x = ((unsigned char *)data)[i];
      val = (x>>bit) & 1;
      printf("%d",val);
    }
  }
}

/******************************************************************************/
/* this function is designed to print data of type DATA_ID_BIT                */
/* it prints the first byte of data as an usigned char                        */

void print_bit( unsigned long len, char *data )
{
  printf( "%u", *((unsigned char *)data) );
}


/******************************************************************************/
/* this funciton is designed to print data of type DATA_ID_TIME               */
/* it uses a bitfield structure to identify all components of time            */
/* NOTE: this code is compiler implementation specific - its is only known to */
/* work with the arm-linux-gcc compiler                                       */
/* This is because bitfields are implementation specific.  In particulur, on  */
/* the arm compiler they seem to be packed tightly, whereas on the x86        */
/* compiler most items are byte aligned.                                      */

struct timestruct
{
    unsigned int junk:   2;   /* can anyone tell me what the 2 bits are for? */
    unsigned int sec:    6;
    unsigned int min:    6;
    unsigned int hour:   5;
	
    unsigned int mday:   5;
    unsigned int mon:    4;
	// unsigned int fill:  4;
    unsigned int year:   8;
};

#define TIME_STRUCT_SIZE 5	// NOTE: this struct is just 5 bytes written, not 8

void print_time( unsigned long len, char *data )
{
  static char buffer[256];    /* string to hold ASCII representation of time  */
  struct timestruct *ts;
  static struct tm tm;

  if (len==0)
  {
    printf( "NULL" );
  }
  else
  {
    ts = (struct timestruct *)data; /* point to the data */

    tm.tm_sec  = ts->sec;           /* transfer packed data into standard     */
    tm.tm_min  = ts->min;           /*    unix time structure                 */
    tm.tm_hour = ts->hour;
    tm.tm_mday = ts->mday;
    tm.tm_mon  = ts->mon-1;
    tm.tm_year = ts->year;

//    strftime( buffer, 255, "%a %b %e %H:%M:%S %Z %Y", &tm );
    strftime( buffer, sizeof(buffer)-1, "%Y%m%dT%H%M%S", &tm );	// like RFC822 / vCalendar DATETIME 20040927T212943
//	printf("%d-", ts->junk);
    print_utf8( strlen(buffer), buffer );
  }
}



/******************************************************************************/
/* this function is designed to print data of type DATA_ID_ULONG              */
/* it prints data as an unsigned long int                                     */

void print_ulong( unsigned long len, char *data )
{
  printf( "%lu", *((unsigned long *)data) );
}



/******************************************************************************/
/* this function is designed to print data of type DATA_ID_BARRAY             */
/* as a bit array															  */

void print_barray( unsigned long len, char *data )
{
	unsigned long i;
	int bit, val;
	unsigned char x;
	
	for (i=0;i<len;i++)
		{
		if (i) 
			printf(" ");
		for (bit=7;bit>=0;bit--)
			{
			x = ((unsigned char *)data)[i];
			val = (x>>bit) & 1;
			printf("%d",val);
			}
		}
}

// translate bit array to Category list

void print_category( unsigned long len, char *data )
{
	unsigned long i;
	int bit, val;
	unsigned char x;
	
	for (i=0;i<len;i++)
		{
		if (i) 
			printf(" ");
		for (bit=7;bit>=0;bit--)
			{
			x = ((unsigned char *)data)[i];
			val = (x>>bit) & 1;
			printf("%d",val);
			}
		}
}



/******************************************************************************/
/* this function is designed to print data of type DATA_ID_UCHAR              */
/* it prints the data as a single unsigned char                               */

void print_uchar( unsigned long len, char *data )
{
  printf( "%u", *((unsigned char *)data) );
}

/******************************************************************************/
/* this function is designed to print data of type DATA_ID_WORD               */

void print_word( unsigned long len, char *data )
{
  printf( "%u", *((unsigned short *)data) );
}

/******************************************************************************/
/* this function is designed to print data of type DATA_ID_UTF8               */
/* it prints data as an ASCII string, which should be ok for most purposes    */
/* should really replace < and > signs here when using XML output!            */

void print_utf8( unsigned long len, char *data )
{
	while(*data && len-->0)
		{
		switch(*data)
			{
			case '&':
				printf("&amp;");
				break;
			case '<':
				printf("&lt;");
				break;
			case '>':
				printf("&gt;");
				break;
			case '\t':
			case '\n':
				putchar(*data);
				break;
			default:
				if((*data&0xff) >= 0x20)
					putchar(*data);	// assumed to be UTF8 if MSB is set!
				// else - othes are disallowed even as character entities
		//			printf("&#x%04X;", *data);	// embedded control character
			}
		data++;
		}
}

/******************************************************************************/
/* this function is used to pretty print the given data of type item and      */
/* length len.                                                                */
/* it selects the appropriate function based on the contents of print_method  */
/* This uses a for loop, rather than a binary search which might be           */
/* marginally faster if the size of print_method is large enough.             */

void pp( char item[4], unsigned long len, UCHAR *data )
{
  struct PrintMethod *pm;

  for ( pm = print_method; pm->name!=NULL; pm++ )
  {
    if ( !strncmp( (char *)pm->name, (char *)item, 4 ) )
    {
      printf( "  <%.4s>", pm->name );   /* print itemname                     */
      pm->fn_ptr( len, (char *)data );  /* print data using appropriate fn    */
      printf( "</%.4s>\n", pm->name );
      return;
    }
  }

  fprintf( stderr, "Don't know how to print %.4s\n", item );
               /* this line should not be reached if print_method is complete */
}

/******************************************************************************/
/* This is an incomplete function to print out error messages.  The rest      */
/* should be added based on the pdf document.                                 */
char *err_name( DTMERR retval )
{
	struct { DTMERR code; char *string; } errtable[]={
		{	0x00, "Closed Correctly" },
		{	0x20, "Cannot write to the designated drive" },
		{	0x21, "Drive specified incorrectly" },
		{	0x23, "Incorrectly arguments" },
		{	0x40, "Specified Box File does not exist" },
		{	0x41, "Cannot delete the existing table" },
		{	0x42, "Specified card does not exist" },
		{	0x43, "Same item identifier already exists" },
		{	0x44, "Lack of memory" },
		{	0x45, "Incorrect conditional expression" },
		{	0x46, "Exceeds interim buffer size" },
		{	0x47, "Exceeds the stack" },
		{	0x48, "Same item and data already exists" },
		{	0x49, "Identifier includes bad character(s)" },
		{	0xe1, "Data does not exist" },
		{	0xfe, "Process aborted due to system abort" },
		{	0xff, "System error" },
		{ DTM_VERSION_ERR, "Version error" },
		{ DTM_SIZE_ERR, "memory_block size contradiction" },
		{ DTM_STRUCT_ERR, "Pointer contradiction" },
		{ DTM_ITEM_STRUCT_ERR, "ITEM infoirmation error" },
		{ DTM_CARD_STRUCT_ERR, "Card struct error" },
		{ DTM_INFO_STRUCT_ERR, "INDEX information error" },
		{ DTM_ID_TABLE_ERR, "INDEX ID Table error" },
		{ DTM_LANGUAGE_ERR, "Language mode error" },
		{ DTM_INDEX_NO_ERR, "Number of Card ID table is 00" },
		{ DTM_NO_IDX_ERR, "No Index" },
		{ DTM_ALREADY_CONVERTED, "Already converted file" },
		{ DTM_ABORTED_FILE, "Aborted convertion file" },
		{ DTM_OLD_VERSION, "Old version file" },
		};
		unsigned int i;
	static char bfr[50];
	for(i=0; i<sizeof(errtable)/sizeof(errtable[0]); i++)
		{
		if(errtable[i].code == retval)
			return errtable[i].string;
		}
	sprintf(bfr, "Unknown error code %02x", retval);
	return bfr;
}

char *ref;			  // value of <REF> tag

/******************************************************************************/
/* this function prints out the contents of the card given by "card",         */
/* assuming that each card contains "itemNum" numbers of items and that       */
/* the item names are given in "items". "BoxFHd" is an opened boxhandle, and  */
/* "itemname" is the attribute of the outermost XML tag to use.               */

DTMERR print_card( int card, UCHAR itemNum, ITEMNAME *items, 
                BOXHANDLE BoxFHd,
                char *itemname, UCHAR* Data, DATALEN LenMax )
{
	CARD Card;          /* data structure holding desired card no, and itemname */
	/* for retrieval using _CardRead                        */
	
	UCHAR item;         /* counter over the items                               */
	
	DATALEN Len;        /* max len of item retrieved                            */
	
	DTMERR retval;      /* return value result                                  */
	
	int first_item;     /* flag indicating first item from a given card         */
	
	Card.CardID = card; /* CardID numbered from 1								  */
	
	first_item = 1;     /* assume first item from card                          */
	
	for (item=0;item<itemNum;item++)  /* loop over all items                    */
		{
		strncpy( (char *)Card.ItemName, (char *)items[item], 4 );
		/* copy item name into datastructure                    */
		
		Len = LenMax;     /* 4th argument of _CardRead served two purposes:       */
		/* 1 - it specified the size of memory allocated to     */
		/*    argument 3 to prevent buffer overflow             */
		/* 2 - it is used to return the actual number read      */
		/*    it is important to not just reuse the 4th arg     */
		/*    without resetting, or odd things will happen      */
		
		if ( (retval = _CardRead( BoxFHd, &Card, Data, &Len )) != 0 )
			{
			return retval;  /* if something goes wrong, return */
			}
		else
			{
			if (first_item) /* if its the first item on the card */
				{
				printf( "<%s card=\"%d\">\n", itemname, card ); 
				/* wrap it in a tag for the card */
				
				first_item = 0;
				}
			
			pp( (char *)items[item], Len, Data );   /* pretty print it */
			}
		}
	if(ref)
		{ // echo back contents of ref value (if present)
		printf("  <REF>");
		print_utf8(strlen(ref), ref);
		printf("</REF>\n");
		free(ref);
		ref=NULL;	// has been written
		}
	printf( "</%s>\n", itemname );    /* close tag for card */
			
	return 0;
}

int boxtable(char *tag, char **itemnamep, char **itemsnamep, char ** BoxFNamep, char **idxfp)
{
	char *itemsname;
	if(strcmp(tag, itemsname="Tasks") == 0)
		{
		*itemnamep="Task",
			*BoxFNamep="/home/zaurus/Applications/dtm/SLTODO.BOX",
			*idxfp="SLTODO.IDX:SLTODO1.IDX:SLTODO2.IDX:SLTODO3.IDX:SLTODO4.IDX";
		}
	else if(strcmp(tag, itemsname="Events") == 0)
		{
		*itemnamep="Event",
			*BoxFNamep="/home/zaurus/Applications/dtm/SLDATE.BOX",
			*idxfp="SLDATE.IDX:SLDATE1.IDX:SLDATE2.IDX:SLDATE3.IDX:SLDATE4.IDX";
		}
	else if(strcmp(tag, itemsname="Contacts") == 0)
		{
		*itemnamep="Contact",
			*BoxFNamep="/home/zaurus/Applications/dtm/SLADRS.BOX",
			*idxfp="SLADRS.IDX";
		}
	else if(strcmp(tag, itemsname="Memos") == 0)
		{
		*itemnamep="Memo",
			*BoxFNamep="/home/zaurus/Applications/dtm/SLMEMO.BOX",
			*idxfp="SLMEMO.IDX";
		}
	else if(strcmp(tag, itemsname="Categories") == 0)
		{
		*itemnamep="Category",
			*BoxFNamep="/home/zaurus/Applications/dtm/SLFLER.BOX",
			*idxfp="SLFLER.IDX";
		}
	else if(strcmp(tag, itemsname="Inbox") == 0)
		{
		*itemnamep="Email",
			*BoxFNamep="/home/zaurus/Applications/dtm/slinbox.box",
			*idxfp="slinbox.idx:slinunrd.idx";
		}
	else if(strcmp(tag, itemsname="Outbox") == 0)
		{
		*itemnamep="Email",
			*BoxFNamep="/home/zaurus/Applications/dtm/sloutbox.box",
			*idxfp="sloutbox.idx";
		}
	else if(strcmp(tag, itemsname="Trash") == 0)
		{
		*itemnamep="Email",
			*BoxFNamep="/home/zaurus/Applications/dtm/slmtrash.box",
			*idxfp="slmtrash.idx";
		}
	else
		return 0;	// no match
	*itemsnamep=itemsname;
	return 1;
}

/******************************************************************************/
/* some global variables   												    */

BOXHANDLE BoxFHd;     /* handle to box file									*/
char *itemsname,      /* outermost XML tag                                  */
	 *itemname;       /* XML tag for a particular card                      */
UCHAR itemNum;        /* total number of items on all cards                 */
ITEMNAME *items;      /* names of all the items                             */
DATALEN LenMax;       /* max size of item retrieved                         */
UCHAR* Data;		  /* item data is retrieved to this buffer			    */
char *arg0;
char *BoxFName,       /* pointer to name of boxfile                         */
filename[PATH_MAX];	  /* name of boxfile without extension					*/
char *idxlist;		// list of index files to update
char eflag;			// -e


/******************************************************************************/
/* process tags																  */

#define VERBOSE 0

struct timestruct convert_time(struct tm *tmp)
{ // convert to DTM format
	struct timestruct ts;
	memset(&ts, 0, sizeof(ts));	// clear all bits
	ts.sec=tmp->tm_sec;
	ts.min=tmp->tm_min;
	ts.hour=tmp->tm_hour;
	ts.mday=tmp->tm_mday;
	ts.mon=tmp->tm_mon+1;
	ts.year=tmp->tm_year;
	return ts;
}

int handletag(char *tag, char *value)
{ // process XM tags
	int i;
	static CARD Card;			// current card
	static CARDID CardID=DTM_FROM_LAST_CARDID;	// current card ID (-1 if no card, 0 if new card)
	DTMERR error;
#if VERBOSE
	fprintf(stderr, "%s<%s>\n", value, tag);
#endif
	char *args=strchr(tag, ' ');
	if(args)
		*args++=0;
	else
		args="";	// no arguments
	if(!BoxFHd && (strcmp(tag, "?xml") == 0 || strcmp(tag, "?xml?") == 0))
		{
		return 0;	// ignore
		}
	if(!BoxFHd && strcmp(tag, "!DOCTYPE") == 0)
		{
		return 0;	// ignore
		}
	if(strcmp(tag, "DTM") == 0 || strcmp(tag, "/DTM") == 0)
		return 0;	// ignore
	if(BoxFHd && strcmp(tag, "delete") == 0)
		return 0;	// ignore
	if(BoxFHd && strcmp(tag, "/delete") == 0)
		{
		if(CardID != 0 && CardID != DTM_FROM_LAST_CARDID)
			{
#if VERBOSE
			fprintf(stderr, "delete card %d\n", CardID);
#endif
			error=_CardDelete( BoxFHd, CardID );
			if(error)
				fprintf(stderr, "can't delete card %d due to %s\n", CardID, err_name(error));
			CardID=DTM_FROM_LAST_CARDID;	// don't write anything any more!
			return error;
			}
		return 0;
		}
	if(boxtable(tag, &itemname, &itemsname, &BoxFName, &idxlist))
		{ // open new database
		UCHAR item;           /* item iterator                                      */
#if VERBOSE
		fprintf(stderr, "Start database %s <%s>/<%s> %s\n", BoxFName, itemsname, itemname, idxlist);
#endif
		if(BoxFHd)
			{
			fprintf(stderr, "%s: There is already a box file open (%s)\n", arg0, BoxFName);
			return 1;
			}
		BoxFHd = _BoxOpen( (unsigned char *)BoxFName );
		
		if (!BoxFHd)
			{
			fprintf( stderr, "%s: BoxOpen(\"%s\") failed\n", arg0, BoxFName );
			return 1;
			}
		
		/************************************************************************/
		/* compute number of items */
		itemNum = _GetItemNum( BoxFHd );
		
		/* get the item names */
		
		items = (ITEMNAME *)malloc( sizeof(ITEMNAME) * itemNum );
		
		for ( item=0; item<itemNum; item++ ) 
			{
			_GetItemName( (unsigned char *)BoxFName, (item+1), items[item] );
			}
		
		Data = (UCHAR *)calloc( LenMax=BUFFERSIZE, 1 );
		if(eflag)
			printf("<%s>\n", itemsname);
		return Data == NULL;	// 1=error if not allocated
		}
	if(BoxFHd && strcmp(tag, itemname) == 0)
		{ // new item
		if(strncmp(args, "card=\"", 6) == 0)
			CardID=atoi(args+6);	// just save!
		else
			{ // expects <tag card="123">
			fprintf(stderr, "<%s> invalid card number attribute: %s\n", tag, args);
			return 1;
			}
#if VERBOSE
		fprintf(stderr, "create new card %d\n", CardID);
#endif
		Card.CardID = DTM_SYSTEM_CARD_ID;    /* Delete System Card (just in case) */
		error=_CardDelete( BoxFHd, Card.CardID );
		if(error == S_DATA_NF)
			error=0;	// ignore
		if(error)
			fprintf(stderr, "can't delete card %d due to %s\n", Card.CardID, err_name(error));
		/* Create a System Card */
		error=_CardCreate( BoxFHd, &Card.CardID );  // create new system card
		if(error)
			fprintf(stderr, "can't create card %d due to %s\n", Card.CardID, err_name(error));
		if(1 || Card.CardID == 0)
			{ // preassign CTTM, MDTM, SYID unless explicitly overwritten
			struct timestruct ts;
			struct tm *tm;
			static long syid;
			time_t now;
			time(&now);
			tm=gmtime(&now);
			ts=convert_time(tm);
			memcpy(Card.ItemName, "CTTM", 4);	// set item name
			error=_CardWrite(BoxFHd, &Card, (UCHAR *) &ts, sizeof(ts));
			memcpy(Card.ItemName, "MDTM", 4);	// set item name
			error=_CardWrite(BoxFHd, &Card, (UCHAR *) &ts, sizeof(ts));
			memcpy(Card.ItemName, "SYID", 4);	// set item name
			syid++;	// sequential
			error=_CardWrite(BoxFHd, &Card, (UCHAR *) &syid, sizeof(syid));
			}
		return error;
		}
	if(BoxFHd && tag[0] == '/')
		{ // closing tag
		if(strcmp(tag+1, itemsname) == 0)
			{ // close database file
#if VERBOSE
			fprintf(stderr, "Database done\n");
#endif
			if(items)
				free( items );
			if(Data)
				free( Data );
			if(BoxFHd)
				_BoxClose( BoxFHd );
			items=NULL;
			Data=NULL;
			BoxFHd=NULL;
			itemname=NULL;
			if(eflag)
				printf("</%s>\n", itemsname);
			return 0;
			}
		if(strcmp(tag+1, itemname) == 0)
			{ // close item
			if(CardID != DTM_FROM_LAST_CARDID)
				{
#if VERBOSE
				fprintf(stderr, "write new card %d\n", CardID);
#endif
				error=_CardCopy( BoxFHd, Card.CardID, &CardID );
				if(error)
					fprintf(stderr, "can't write card %d due to %s\n", Card.CardID, err_name(error));
				if(error)
					_CardDelete( BoxFHd, Card.CardID );
				else if(0)
					{
					INDEXNO idx;
					UCHAR IdxFName[256];
					char *r=strrchr(BoxFName, '/')+1;
					char *i=idxlist;
					UCHAR *s;	// where to insert new file name
#if VERBOSE
					fprintf(stderr, "write index %s\n", idxlist);
#endif
					memcpy(IdxFName, BoxFName, r-BoxFName);	// copy base path
					s=IdxFName+(r-BoxFName);
					while(*i)
						{ // update all indexes
						char *j=strchr(i, ':');	// get end of this field
						if(!j)
							j=i+strlen(i);	// up to end of string
						memcpy(s, i, j-i);	// append
						s[j-i]=0;	// 0-terminate
						i=j;
						if(*i == ':')
							i++;
#if VERBOSE
						fprintf(stderr, "append index to box %s\n", IdxFName);
#endif
						error=_IndexNoAppend((unsigned char *)IdxFName, BoxFHd, CardID, &idx);
						if(error == S_SAME_DATA)
							error=0;	// ignore
						if(error)
							fprintf(stderr, "can't create index in %s for card %d due to %s\n", IdxFName, Card.CardID, err_name(error));
						}
					}
				if(!error && eflag)
					{ // echo result (with new CARDID!)
					error=print_card( CardID, itemNum, items, 
								BoxFHd,
								itemname, Data, LenMax);
					}
				CardID=DTM_FROM_LAST_CARDID;	// no current card
				return error;
				}
			return 0;	// ignore
			}
		if(strlen(tag+1) == 4)
			{
			for(i=0; i<itemNum; i++)
				{
				if(strncmp(tag+1, (char *) items[i], 4) == 0)
					{ // item found!
					struct PrintMethod *pm;
#if VERBOSE
					fprintf(stderr, "%s:=%s\n", tag+1, value);
#endif
					memcpy(Card.ItemName, tag+1, 4);	// set item name
					for ( pm = print_method; pm->name!=NULL; pm++ )
						{ // scan print methods
						if(!strncmp( (char *)pm->name, (char *)Card.ItemName, 4 ) )
							{
							// add other if's to decode formats
							// do proper conversions
							error=0;
							if(pm->fn_ptr == print_utf8)
								error=_CardWrite( BoxFHd, &Card, (UCHAR *) value, strlen((const char*)value));
							else if(pm->fn_ptr == print_uchar)
								{
								unsigned char x=atoi(value);
								error=_CardWrite( BoxFHd, &Card, (UCHAR *) &x, sizeof(x));
								}
							else if(pm->fn_ptr == print_word)
								{
								unsigned short x=atoi(value);
								error=_CardWrite( BoxFHd, &Card, (UCHAR *) &x, sizeof(x));
								}
							else if(pm->fn_ptr == print_ulong)
								{
								unsigned long x=atoi(value);
								error=_CardWrite( BoxFHd, &Card, (UCHAR *) &x, sizeof(x));
								}
							else if(pm->fn_ptr == print_bit)
								{
								unsigned char x=atoi(value);
								error=_CardWrite( BoxFHd, &Card, (UCHAR *) &x, sizeof(x));
								}
							else if(pm->fn_ptr == print_time)
								{
								if(strcmp(value, "NULL") == 0 || *value == 0)
									{ // NULL time
								//	error=_CardWrite( BoxFHd, &Card, NULL, 0);	// write empty record
									error=0;	// don't write
									}
								else
									{
									struct timestruct ts;
									struct tm tm;
									char *r=strptime(value, "%Y%m%dT%H%M%S", &tm);
									if(!r)
										{
										fprintf(stderr, "time value expected for item %s of card %d: %s\n", tag+1, Card.CardID, value);
										return 1;
										}
									if(*r != 0)
										fprintf(stderr, "extra characters ignored for item %s of card %d: %s\n", tag+1, Card.CardID, r);
									ts=convert_time(&tm);
									error=_CardWrite( BoxFHd, &Card, (UCHAR *) &ts, TIME_STRUCT_SIZE);
#if 0
									print_time(sizeof(ts), (char *) &ts);	// echo back
									printf("\n");
#endif
									}
								}
							else if(pm->fn_ptr == print_barray)
								{
								UCHAR buffer[256];
								int i=0;
								int b=0;
								char bits;
								char *c;
								for(c=value; *c; c++)
									{ // get next bit (LSB first)
									if(*c == ' ')
										continue;
									bits>>=1;
									if(*c == '1')
										bits |= 0x80;
									else if(*c != 0)
										{
										fprintf(stderr, "binary data expected for item %s of card %d\n", tag+1, Card.CardID);
										return 1;
										}
									if(++b == 8)
										buffer[i++]=bits, b=0;	// one byte every 8 bits
									}
								error=_CardWrite( BoxFHd, &Card, buffer, i);
								}
							else if(pm->fn_ptr == print_category)
								{
								if(strlen(value) != 0)
									{ // not empty
									fprintf(stderr, "can't write category (%s) in this version\n", value);							
									return 1;
									}
								}
							else
								break;	// unrecognized
							if(error)
								fprintf(stderr, "can't write item %s of card %d due to %s\n", tag+1, Card.CardID, err_name(error));							
							return error;
							}
						}
					fprintf( stderr, "Don't know how to parse %.4s\n", (char *)Card.ItemName );
					return 1;
					}
				}
			}
		else if(strcmp(tag, "/REF") == 0)
			{
			if(ref)
				free(ref);
			ref=(char *) malloc(strlen(value)+1);
			strcpy(ref, value);	// save
			return 0;
			}
		else
			;	// unrecognized
		}
	if(BoxFHd && strcmp(tag, "REF") == 0)
		return 0;	// accept
	if(BoxFHd && strlen(tag) == 4)
		{
		for(i=0; i<itemNum; i++)
			{
//			fprintf(stderr, "check %s\n", (char *) items[i]);
			if(strncmp(tag, (char *) items[i], 4) == 0)
				return 0;	// valid opening tag
			}
		}
	fprintf(stderr, "unrecognized tag <%s %s>\n", tag, args);
	return 1;
}

/******************************************************************************/
/* XML parser by hns@computer.org                                             */

int write(FILE *fp)
{ // parse XML input
	int c;
	// FIXME: not buffer overflow safe!!!
	char tag[256], *tp=NULL;
	static char *value;
	char *vp=NULL;
	char escchar[256], *ep=NULL;
	if(!value)
		value=(char *) malloc(BUFFERSIZE);	// let's hope this is enough!!!!!
	value[0]=0;
	while((c = getc(fp)) != EOF)
		{
		switch(c)
			{
			case '<':
				if(tp)
					break;	// already within tag
				tp=tag;	// open tag
				continue;
			case '>':
				{
					int taglen;
					if(!tp)
						break;	// was not opened!
					*tp=0;
					taglen=tp-tag;
					if(vp)
						*vp=0;
					if(taglen > 1 && tag[taglen-1] == '/')
						{ // <tag/>
						tp[-1]=0;	// remove closing /
						if(handletag(tag, ""))	// fake opening tag
							return 1;
						memmove(tag+1, tag, taglen+1);	// make room
						tag[0]='/';	// make it a closing tag
						}
					if(handletag(tag, value))
						return 1;
					vp=value;	// start over
					tp=NULL;	// done
					continue;
				}
			case '&':
				if(vp)
					{
					ep=escchar;
					continue;
					}
				break;	// take as & character
			case ';':
				if(ep)
					{ // replace ; by decoded character
					*ep=0;
					ep=NULL;	// done
#if 0
					fprintf(stderr, "esc=%s\n", escchar);
#endif
					if(strcmp(escchar, "lt") == 0)
						c='<';
					else if(strcmp(escchar, "gt") == 0)
						c='>';
					else if(strcmp(escchar, "amp") == 0)
						c='&';
					else if(isdigit(escchar[0]))
						c=atoi(escchar);	// &num;
					else
						;	// keep ';'
					}
				break;
			}
		if(ep)
			*ep++=c;	// store in escape character
		else if(tp)
			*tp++=c;	// store in tag
		else if(vp)
			*vp++=c;	// store in value
		else
			return -1;	// invalid! i.e. not initialized
	}
	return 0;	// OK
}

void boxname(char *application)
{ // translate and set box file name
    if(!strcasecmp(application,"ToDo"))
		BoxFName = "/home/zaurus/Applications/dtm/SLTODO.BOX";
    else if (!strcasecmp(application,"Calendar"))
		BoxFName = "/home/zaurus/Applications/dtm/SLDATE.BOX";
    else if (!strcasecmp(application,"Addressbook"))
		BoxFName = "/home/zaurus/Applications/dtm/SLADRS.BOX";
    else if (!strcasecmp(application,"Memo"))
		BoxFName = "/home/zaurus/Applications/dtm/SLMEMO.BOX";
    else if (!strcasecmp(application,"Category"))
		BoxFName = "/home/zaurus/Applications/dtm/SLFLER.BOX";
	else if (!strcasecmp(application,"Inbox"))
		BoxFName = "/home/zaurus/Applications/dtm/slinbox.box";
	else if (!strcasecmp(application,"Outbox"))
		BoxFName = "/home/zaurus/Applications/dtm/sloutbox.box";
	else if (!strcasecmp(application,"Trash"))
		BoxFName = "/home/zaurus/Applications/dtm/slmtrash.box";
	else
		{
		fprintf( stderr, "%s:  Unsupported application \"%s\" (ToDo, Calendar, Addressbook, Memo, Category, Inbox, Outbox, Trash).\n", 
				 arg0, application);
		exit(1);
		}
}

/****************************************************************************/
/* figure out appropriate XML tag names                                     */

void tagnames(void)
{ // get basename
	char *basename=strrchr(BoxFName, '/')+1;	// !! assumes valid path with / !!
	char *suffix=strchr(basename, '.');
	int len;
	if(suffix)
		len=suffix-basename;
	else
		len=strlen(basename);	// take all
	strncpy( filename, basename, len);	// copy valid part
	filename[len] = '\0';
	//	  fprintf(stderr, "filename=\"%s\"\n", filename);
	if ( !strcmp( filename, "SLTODO" ) )
		{
		itemsname = "Tasks";
		itemname  = "Task";
		}
	else if ( !strcmp( filename, "SLDATE" ) )
		{
		itemsname = "Events";
		itemname  = "Event";
		}
	else if ( !strcmp( filename, "SLADRS" ) )
		{
		itemsname = "Contacts";
		itemname  = "Contact";
		}
	else if ( !strcmp( filename, "SLMEMO" ) )
		{
		itemsname = "Memos";
		itemname  = "Memo";
		}
	else if ( !strcmp( filename, "SLFLER" ) )
		{
		itemsname = "Categories";
		itemname  = "Category";
		}
	else if ( !strcmp( filename, "slinbox" ) )
		{
		itemsname = "Inbox";
		itemname  = "Email";
		}
	else if ( !strcmp( filename, "sloutbox" ) )
		{
		itemsname = "Outbox";
		itemname  = "Email";
		}
	else if ( !strcmp( filename, "slmtrash" ) )
		{
		itemsname = "Trash";
		itemname  = "Email";
		}
	else
		{
		fprintf( stderr, "%s:  error \"%s\" not supported\n", arg0, filename );
		exit(-1);
		}
}	

void createbox(void)
{ // try to create box if it was deleted
	APLID aplid='MAIL';	// 4 letters application ID - ADRS, CSDL, FLER, TEDT, TODO, MAIL
	DTMERR retval;        /* return value of print_card                         */
	retval=_BoxCreate((unsigned char *)BoxFName, aplid);
	if(retval)
		{
		fprintf( stderr, "%s: BoxCreate(\"%s\") failed\n", arg0, BoxFName);
		exit(-1);
		}
	// should somehow create data fields!
//	DTMERR BoxItemAppend(BoxFName,Item Data,ItemNum,IsNewItemAdded) 
}

DTMERR testbox(void)
{
	DTMERR retval;        /* return value of print_card                         */
	retval = _DTMFileCheckLevel((unsigned char *)BoxFName, DTM_CHECK_FULL);
	if(retval)
		fprintf( stderr, "%s: DTMFileCheckLevel(\"%s\") failed: %s\n", arg0, BoxFName, err_name(retval));
	return retval;
}	
	
DTMERR printbox(void)
{
	ULONG size;           /* size in bytes of the box file - not actually used  */
	USHORT card,          /* iterator over cards                                */
		indexNo,		  /* iterator through index								*/
		num;              /* number of undeleted cards (NOT the actual number   */
						  /* of entries in the box file)                        */
	
	UCHAR item;           /* item iterator                                      */
	DTMERR retval;        /* return value of print_card                         */

	tagnames();	// get tag names for this box file
	
	/************************************************************************/
	/* open file */
	BoxFHd = _BoxOpen( (unsigned char *)BoxFName );
	
	if (!BoxFHd)
		{ // try to create
//		createbox();	// try to create a box file
		if (!BoxFHd)
			{
			fprintf( stderr, "%s: BoxOpen(\"%s\") failed even after creating new Box\n", arg0, BoxFName );
			exit(-1);
			}
		}

	printf( "<%s>\n", itemsname );	// new range to read

	/************************************************************************/
	/* compute number of items */
	itemNum = _GetItemNum( BoxFHd );
	
	/* get the item names */
	
	items = (ITEMNAME *)malloc( sizeof(ITEMNAME) * itemNum );
	
	for ( item=0; item<itemNum; item++ ) 
		{
		_GetItemName( (unsigned char *)BoxFName, (item+1), items[item] );
		}
	
	Data = (UCHAR *)calloc( LenMax=BUFFERSIZE, 1 );
	
	/************************************************************************/
	/* compute card number (and box file size) */
	_BoxCardTotal( (unsigned char *)BoxFName, &size, &num );
	
	/************************************************************************/
	
	retval=0;
	for (indexNo=0, card=1; indexNo < num;card++)
		/* don't loop to num here, it doesn't work */
		// that one doesn't either - card numbers are not necessarily without holes after deleting one!
		{
		retval = print_card( card, itemNum, items, BoxFHd, itemname, Data, LenMax );
		if(retval == 0x42)
			continue;	// card number does not exist, try next one
		if(retval)
			{ // other error
			fprintf( stderr, "%s: CardRead(\"%s\", CARDID=%d) failed: %s\n", arg0, BoxFName, card, err_name(retval) );
			
			break;
			}
		indexNo++;	// one more processed
		}
	free( items );
	free( Data );
	
	_BoxClose( BoxFHd );
	printf( "</%s>\n", itemsname );
	return retval;
}

/******************************************************************************/
/* the program's main function                                                */

int main( int argc, char ** argv )
{
	DTMERR retval;        /* return value of print_card                         */
	
	char wflag=FALSE;		// -w
	char fflag=FALSE;		// -f
	char tflag=FALSE;		// -t
	char any;
	
	/****************************************************************************/
	/* process arguments                                                        */
	arg0=argv[0];
	if(argv[1] != NULL && argv[2] != NULL && argv[3] == NULL && strcmp(argv[1], "reset") == 0 && strcmp(argv[2], "database") == 0)
		{
		char bfr[256];
		printf("Please close all PIM applications (Calendar, Address Book, ToDo List, Text Editor, EMail) first!\n");
		printf("Really clear all data in database (yes)? ");
		fflush(stdout);
		if(strcmp(fgets(bfr, sizeof(bfr)-2, stdin), "yes\n") == 0)
			{
			if(dtmreset())
				printf("DTM database deleted and reinitialized\nIt is strongly recommended to reboot the Zaurus now!\n");
			else
				printf("Failed.\n");
			}
		else
			printf("Cancelled.\n");
		exit(0);
		}
	eflag=FALSE;
	while(!fflag && argv[1] && argv[1][0] == '-')
		{ // process flags
		if(strcmp(argv[1], "-f") == 0)
			{
			fflag=TRUE;	// suppress usage message and finish 
			break;	// don't skip here
			}
		else if(!wflag && strcmp(argv[1], "-w") == 0)
			wflag=TRUE;
		else if(!wflag && strcmp(argv[1], "-t") == 0)
			tflag=TRUE;
		else if(!wflag && strcmp(argv[1], "-we") == 0)
			wflag=eflag=TRUE;
		else if(strcmp(argv[1], "-a") == 0)
			{ // print ALL boxes
			static char *a[]={ "", "ToDo", "Calendar", "Addressbook", "Memo", /* "Category", "Inbox", "Outbox", "Trash", */ NULL };
			argv=a;	// all
			break;
			}
		else
			break;	// will result in usage:
		argv++;	// has been processed
		}
	if((!wflag && !argv[1]) || (argv[1] && argv[1][0] == '-' && !fflag))
		{ // either -w or applications (not if -f)
		fprintf( stderr, "usage:  %s [-w[e]] [-t] [-a] table | -f path.box ... [<input.xml]\n", arg0 );
		exit(-1);
		}
	if(argv[1] == NULL && eflag)
		{ // -we also needs header
		printf( "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" );
		printf( "<!DOCTYPE dtm PUBLIC \"-//GNU//DTD DTM 1.0//EN\" \"http://www.dstri.de/DTDs/dtm-1.0.dtd\">\n" );
		printf( "<DTM VERSION=\"%s\">\n", VERSION );
		}
	if(wflag && write(stdin))
		{ // write error
		return 1;
		}
	if(ref)
		{
		free(ref);
		ref=NULL;	// reference has been written
		}
	retval=0;	// default
	if((any=(argv[1] != NULL)) && !eflag && !tflag)
		{ // anything to output?
		printf( "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" );
		printf( "<!DOCTYPE dtm PUBLIC \"-//GNU//DTD DTM 1.0//EN\" \"http://www.dstri.de/DTDs/dtm-1.0.dtd\">\n" );
		printf( "<DTM VERSION=\"%s\">\n", VERSION );
		}
	while(argv[1])
		{ // print all specified boxes
		if(strcmp(argv[1], "-f") == 0 && argv[2] != NULL)
			{ // -f file
			BoxFName=argv[2];
			argv++;
			}
		else
			boxname(argv[1]);	// set boxname directly
		if(tflag)
			retval=testbox();
		else
			retval=printbox();
		if(retval)
			break;
		argv++;
		}
	if(!tflag && (any || eflag))
		printf( "</DTM>\n" );
	return retval?1:0;
}
