/*  DATE 12Sep82 0077
Created 27 May 82

sort
This is the external sort routine taken from the book "Software Tools" by
Kernighan and Plauger.  It has been translated from Ratfor to C (an easy
job) by Gary Gilbreath and the same person is responsible for all the various
kluges herein.  In addition, the options -e, -n, -r, and -s have been added.

Written for the Software Toolworks version of C (C/80).
*/

#define LNAM 15
#define TRUE 1
#define FALSE 0
#define LETTER 'a'
#define DIGIT '0'
#define EOF -1
#define EOS 0
#define NUL 0

#define MERGEORDER 6
#define MAXLINE 512
#define MAXFILES 6

#include "printf.h"

/*********** Global variables ***********/
static int nlines;		/* global number of lines var */
static char *linbuf;		/* pointer to sort buffer */
static unsigned int *linptr;	/* pointer to record pointers */
static unsigned int MaxText;	/* max amount of sort test */
static int MaxPtr;		/* max number of records */
static char eor;		/* end-of-record character */
static char revord;		/* sort in reverse order flag */
static char numsort;		/* sort on numbers in first column */
static char blankskip;		/* skip leading blanks */
static char debug;		/* debug option */


sort(innam, outnam, endrec, order, ns, sb, dbug)
char innam[], outnam[], endrec;
char order, ns, sb, dbug;
{
	static char name[LNAM];
	static int infil[MERGEORDER];
	static int i, ifile, ofile, high, lim, low, outfil, t, *atbdos;
	static int junk;
	static unsigned int memleft, begmem;
	

	debug = dbug;		/* set global debug option */
	eor = endrec;		/* set global eor character */
	revord = order;		/* reverse sort option */
	numsort = ns;		/* number sort option */
	blankskip = sb;		/* skip leading blanks option */

	atbdos = 6;		/* point to location of bdos */
/* Point to end of memory.  Allow for ten 292 byte file buffers. */
	begmem = alloc(1) + 2920;	/* point to end of memory */
/* Calculate amount of memory left.  Allow 1500 bytes for stack. */
	memleft = *atbdos - begmem - 1500;
/* Reserve space for line pointers */
	MaxPtr = (memleft/20)/2;	/* assume length of avg line is 30 */
/* Reserve rest of space for sort buffer (must be multiple of 128) */
	MaxText = ((memleft-2*MaxPtr)/128)*128;
/* Set up array pointers */
	linbuf = begmem;
	linptr = linbuf + MaxText + 1;
	if (debug) {		/* if wants to know sizes */
	  printf("BDOS     = %u\n", *atbdos);
	  printf("endprg   = %u\n", begmem - 2920);
	  printf("free mem = %u bytes\n", memleft);
	  printf("memleft  = %u bytes\n", memleft - MaxText - 2*MaxPtr);
	  printf("MaxText  = %u bytes\n", MaxText);
	  printf("MaxPtr   = %d lines\n", MaxPtr);
	  printf("@linbuf = %u\n", linbuf);
	  printf("@linptr = %u\n", linptr);
	}

	nlines = 0;
	high = 0;
	ifile = file(innam, "r");
	do {
		t = gtext(linptr, linbuf, ifile);
		if (nlines != 0) {	/* if at least one line gotten */
			quick(linptr, nlines, linbuf);
			if (t == EOF && high == 0) /* if first sort and all */
				outfil = file(outnam, "w"); /* done in mem */
			else
				outfil = makfil(++high);
			ptext(linptr, nlines, linbuf, outfil);
			fclose(outfil);
		}
	} while (t != EOF);

	if (high <= MERGEORDER && high != 0) { /* if less than MERGEORDER */
		gopen(infil, 1, high);	/* files made and more than 1 temp */
		outfil = file(outnam, "w");
		merge(infil, high, outfil);
		fclose(outfil);
		gremov(infil, 1, high);
	}
	else if (high != 0) {		/* if merge needed */
		for (low = 1; low < high; low += MERGEORDER) {  /* merge */
			lim = min(low + MERGEORDER - 1, high);
			gopen(infil, low, lim);
			outfil = makfil(++high);
			merge(infil, lim - low + 1, outfil);
			fclose(outfil);
			gremov(infil, low, lim);
		}
		gname(high, name);		/* final cleanup */
		outfil = file(name, "r");
		ofile = file(outnam, "w");
		fcopy(ofile, outfil);
		fclose(outfil);
		remove(name);
	}
}


/* fcopy - copy file1 to file2 */
fcopy(file2, file1)
int file2, file1;
{
	static char c;

	while ((c = getc(file1)) != EOF)
		putc(c, file2);
	fclose(file2);
}


/* min - return the smallest of two numbers */

min(num1, num2)
int num1, num2;
{
	return ((num1 < num2) ? num1 : num2);
}


/* gtext - get text lines into linbuf */

gtext(linptr, linbuf, infile)
unsigned int linptr[];
char linbuf[];
int infile;
{
	static int len;
	static unsigned int lbp;

	nlines = 0;
	lbp = 0;
	do {
		len = EOF;
		if (rrec(linbuf + lbp, infile, eor) == EOF)
			break;
		len = reclen(linbuf + lbp, eor);
		linptr[nlines++] = lbp;
		if (debug)
			printf("linptr[%d] = %u\n",nlines-1,linptr[nlines-1]);
		lbp = lbp + len + 1;	/* "1" = room for EOR */
	} while (lbp < MaxText - MAXLINE && nlines < MaxPtr);
	return (len);
}


/* ptext - output text lines from linbuf */

ptext(linptr, nlines, linbuf, outfil)
unsigned int linptr[];
int nlines;
char linbuf[];
int outfil;
{
	static int i;
	static unsigned int j;

	for (i = 0 ; i < nlines ; i++) {
		j = linptr[i];
		wrec(outfil, linbuf + j, eor);
	}
}



/* exchan - exchange linbuf[lp1] with linbuf[lp2] */

exchan(linptr, lp1, lp2)
unsigned int linptr[];
int lp1, lp2;
{
	static unsigned int k;

	k = linptr[lp1];
	linptr[lp1] = linptr[lp2];
	linptr[lp2] = k;
}



/* makfil - make new file for number n */

makfil(n)
int n;
{
	static char name[LNAM];

	gname(n, name);
	return (file(name, "w"));
}


/* gname - make unique name for file id n */

gname(n, name)
int n;
char name[];
{
	static int i, junk;

	strcpy(name, "STEMP");
	i = strlen(name);
	itoa(n, name + i);
}


/* gopen - open group of files low ... lim */

gopen(infiles, low, lim)
int infiles[], lim, low;
{
	static char name[LNAM];
	static int i;

	for (i = 0; i <= lim-low ; i++) {
		gname(low+i, name);
		infiles[i] = file(name, "r");
	}
}


/* gremov - remove group of files low ... lim */

gremov(infiles, low, lim)
int infiles[], lim, low;
{
	static char name[LNAM];
	static int i;

	for (i = 0 ; i <= lim-low ; i++) {
		fclose(infiles[i]);
		gname(low+i, name);
		remove(name);
	}
}


/* merge - merge infiles[0] ... infiles[nfiles-1] onto outfil */

merge(infiles, nfiles, outfil)
int infiles[], nfiles, outfil;
{
	static int i, inf, nf;
	static unsigned int lbp, lp1;

	lbp = 0;
	nf = 0;
	for (i = 0 ; i < nfiles ; i++) /* get one line from each file */
		if (rrec(linbuf + lbp, infiles[i], eor) != EOF) {
			linptr[nf++] = lbp;
			lbp += MAXLINE;		/* room for largest line */
		}
	quick(linptr, nf, linbuf);		/* make initial heap */
	while (nf > 0) {
		lp1 = linptr[0];
		wrec(outfil, linbuf + lp1, eor);
		inf = lp1/MAXLINE;		/* compute file index */
		if (rrec(linbuf + lp1, infiles[inf], eor) == EOF)
			linptr[0] = linptr[--nf];
		reheap(linptr, nf, linbuf);
	}
}


/* reheap - propagate linbuf[linptr[0]] to proper place in heap */

reheap(linptr, nf, linbuf)
unsigned int linptr[];
int nf;
char linbuf[];
{
	static int i, j;

	for (i = 0 ; i+i+1 < nf ; i = j) {
		j = i+i + 1;
		if (j < nf-1)		/* find smaller child */
			if (compar(linptr[j], linptr[j+1], linbuf) > 0)
				j++;
		if (compar(linptr[i], linptr[j], linbuf) <= 0)
			break;		/* proper position found */
		exchan(linptr, i, j);	/* percolate */
	}
}



/* shell - Shell sort for character lines */
/*
shell(linptr, nlines, linbuf)
unsigned int linptr[];
int nlines;
char linbuf[];
{
	static int gap, i, ig, j, k;

	for (gap = nlines/2 ; gap > 0 ; gap /= 2)
		for (j = gap ; j < nlines ; j++)
			for (i = j-gap ; i >= 0 ; i -= gap) {
				ig = i + gap;
				if (compar(linptr[i], linptr[j], linbuf) <= 0)
					break;
				exchan(linptr, i, ig);
			}
}
*/


/* quick - quicksort for character lines */

quick(linptr, nlines, linbuf)
unsigned int linptr[];
int nlines;
char linbuf[];
{
	static unsigned int pivlin;
	static int i, j, p, lv[20], uv[20];

	lv[0] = 0;
	uv[0] = nlines-1;
	p = 0;
	while (p >= 0)
		if (lv[p] >= uv[p])	/* only one element in this subset */
			p--;
		else {
			i = lv[p] - 1;
			j = uv[p];
			pivlin = linptr[j];	/* pivot line */
			while (i < j) {
				for(i++;compar(linptr[i],pivlin,linbuf)<0;i++)
					;
				for(j-- ; j > i ; j--)
					if (compar(linptr[j],pivlin,linbuf)<=0)
						break;
				if (i < j)	/* out of order pair */
					exchan(linptr, i, j);
			}
			j = uv[p];	/* move pivot to position i */
			exchan(linptr, i, j);
			if (i-lv[p] < uv[p]-i) {
				lv[p+1] = lv[p];
				uv[p+1] = i - 1;
				lv[p] = i + 1;
			}
			else {
				lv[p+1] = i + 1;
				uv[p+1] = uv[p];
				uv[p] = i - 1;
			}
			p++;		/* push onto stack */
		}
}


/* compar - compare linbuf[lp1] with linbuf[lp2] */

compar(lp1, lp2, linbuf)
unsigned int lp1, lp2;
char linbuf[];
{
	static unsigned int i, j;
	static char stat;

	i = lp1;
	j = lp2;
	if (blankskip) {	/* if wants to skip leading blanks */
		i = skipblanks(linbuf, i);
		j = skipblanks(linbuf, j);
	}
	if (numsort) {		/* if number sort */
		if ((stat = numcmp(i, j, linbuf)) != 0) /* if not equal */
			return (stat);	/* return result of compare */
		i = pastnum(linbuf, i);	/* move past number */
		j = pastnum(linbuf, j);
	}
	while (linbuf[i] == linbuf[j]) {
		if (linbuf[i] == eor)
			return (0);
		i++;
		j++;
	}
	if (revord)		/* if sorting in reverse order */
		return ((linbuf[i] > linbuf[j]) ? -1 : 1);
	else			/* sorting in normal order */
		return ((linbuf[i] < linbuf[j]) ? -1 : 1);
}


/* skipblanks - return index of first non-blank index in a line. */

skipblanks(s, i)
char s[];
unsigned int i;
{
	static char c;

	while ((c = s[i]) == ' ' || c == '\t')
		i++;
	return (i);
}


/* numcmp - compare numbers at linbuf[i] and linbuf[j] */

numcmp(i, j, linbuf)
unsigned int i, j;
char linbuf[];
{
	static int num1, num2;

	num1 = atoi(linbuf + i);	/* convert ASCII to number */
	num2 = atoi(linbuf + j);
	if (num1 == num2)		/* if equal */
		return (0);
	if (revord)			/* if sorting in reverse order */
		return ((num1 > num2) ? -1 : 1);
	else				/* sorting in normal order */
		return ((num1 < num2) ? -1 : 1);
}


/* pastnum - return pointer to character after last digit. */

pastnum(linbuf, i)
char linbuf[];
unsigned int i;
{
	while (type(linbuf[i++]) == DIGIT) /* loop until non-digit found */
		if (linbuf[i] == eor)
			break;
	return (i);
}


/* rrec - read a record from file ifile with end-of-record character eor. */

rrec(s, ifile, eor)
char *s;
int ifile;
char eor;
{
	static char c;

	while ((c = getc(ifile)) != EOF && c != eor)
		*s++ = c;
	/* read til EOF or end-of-record */
	*s = eor;			/* put on string termininator */
	return (c);
}


/* wrec - write a record to file ofile with end-of-record character eor. */

wrec(ofile, s, eor)
int ofile;
char *s, eor;
{
	static char c;

	while ((c = *s++) != eor)
   		putc(c, ofile);
	putc(eor, ofile);
	return;
}


/* reclen - return length of record terminated by eor. */

reclen(s, eor)
char *s, eor;
{
	static char *p;

	p = s;
	while (*s != eor)
		s++;
	return (s - p);
}
