;------------------------------------------------------------------------------
;DFSI.ASM displays the system info screen
;------------------------------------------------------------------------------

;returns BIOS equipment info (40:10h) in reg16
.RETURN_EQUIP  MACRO     reg16
     push      es
     mov       ax,040h
     mov       es,ax
     mov       reg16,es:[010h]
     pop       es
ENDM

;pass the address in the CMOS data area
;Byte value at that address is retruned in al
.READ_CMOS   MACRO     adr
     cli
     mov       dx,070h
     mov       al,adr
     out       dx,al
     nop
     nop
     inc       dx
     in        al,dx
     sti
ENDM

;Pass the address in the CMOS data area
;Value to write in al
.WRITE_CMOS    MACRO     adr,data
     cli
     mov       dx,070h
     mov       al,adr
     out       dx,al
     inc       dx
     mov       al,data
     nop
     nop
     out       dx,al
     sti
ENDM

.put_MSG     MACRO
     mov       si,offset none
     mov       di,offset si_line1
     call      copy_null
ENDM

  in_sysinfo   db   0
  CPUCODE      dw   0    ;code for CPU and NDP
  stepping386  db   0
  cpuclock     dw   0    ;cpu clock rate x 10 (i.e. div by 10 and insert .)
  bit32        db   0
  xmsver       dw   0
  xmmver       dw   0
  hmaavail     db   0    ;left at 0 if no HMA
                         ;set to 1 if HMA exists but in use
                         ;       2               and free
  dos5hma      db   0
  hmafree      dw   0
  basemem      dw   0    ;total base mem in KB (040h:13h)
  extmem       dw   0    ;KB of extended memory from BIOS call
  bustype      db   0    ;1=XT bus -- does not currently distinguish
                         ;2=ISA
                         ;3=EISA
                         ;4=MCA
  commport_addr     dw   0,0,0,0
  commport_type     db   4,4,4,4
  printport_addr    dw   0,0,0
  ext_kybd_bios     db   0
  siFlags           db   0
                         ;bit 0    Soundblaster
                         ;bit 1    Adlib
                         ;bit 2    MPU-401
                         ;bit 3    Game Port (from read attempt)
                         ;bit 4    Game Port (from equip flag)
                         ;bit 5    Weitek
                         ;bit 6
  SBLASTER_MASK     EQU  01b
  ADLIB_MASK        EQU  10b
  MPU_MASK          EQU  100b
  GAME_MASK         EQU  1000b
  GAMEB_MASK        EQU  10000b
  WEITEK_MASK       EQU  100000b

  sbport            dw   0    ;note: This variable contains garbage if
                              ;sound blaster not found
  ems_active        db   0
  qemm_version      dw   0
  memm_present      db   0

  floppies          db   0    ;high nibble is drive 0
                              ;low nibble is drive 1
                              ;0 = none
                              ;1 = 360K
                              ;2 = 1.2M
                              ;3 = 720K
                              ;4 = 1.44
                              ;5 = 2.88
  num_floppies      db   0

 dfprog_mem    dw   0
 dfdir_mem     dw   0
 dfshell_mem   dw   0

 bios_video    dw   0h        ;video codes from int 10h, function 1A00h
                              ;low = inactive, high = active
 dos_files     dw   0
 dos_buff      dw   0
 dos_env       dw   0
 dos_env_used  dw   0

 temp              db   0
 cdrive        db   0         ;current drive
 restoredrive  db   0         ;current DF drive
                              ;reset when exit sysinfo
  portok    db   0

  none              db   "none",0
  notcmos           db   "BIOS",0

 LARROW   EQU       75
 RARROW   EQU       77
 CLARROW  EQU       115
 CRARROW  EQU       116

;--------------------------------------------------------------------------
; This subroutine displays various system information on the screen
; Returns:  nothing
; Modifies: assume everything
;--------------------------------------------------------------------------

 SYS_INFO      proc near

     mov       in_sysinfo,1        ;flag for error handler
     mov       ah,19h
     int       21h                 ;current drive in al
     mov       restoredrive,al

     mov       al,c_normal
	call		clr_scr

     call      draw_sigrid
     mov       ah,08h
     int       21h

     call      DF_CPUID

     mov       cpucode,ax
     cmp       ah,3           ;80386
     jne       _sysinfo_50
     cmp       mswin_version,0  ;the invalid op code tests for stepping make
                                ;windows very displeased
     jne       _sysinfo_50
     call      step386
     mov       stepping386,al
 _sysinfo_50:
     call      get_freq
     mov       cpuclock,ax

     call      disp_cpu
     call      draw_sigrid
     mov       ah,08h
     int       21h


     call      get_bustype
     mov       bustype,al


     call      get_commports
     call      disp_hardware
     call      draw_sigrid
     mov       ah,08h
     int       21h
     call      get_printports
     call      disp_hardware
     call      draw_sigrid
     mov       ah,08h
     int       21h
     call      get_siflags

     call      disp_hardware
     call      draw_sigrid
     mov       ah,08h
     int       21h

     call      get_global_drive

     call      check_qemm
     or        memm_present,bl
     mov       qemm_version,ax

     call      check_386max
     or        memm_present,bl

     call      check_emm386
     or        memm_present,bl

     call      get_dfmem
     call      get_extmem          ;does a couple of tricks to get total
                                   ;system extended memory

     push      es                  ;get base system memory
     mov       ax,040h
     mov       es,ax
     mov       ax,es:[013h]
     mov       basemem,ax
     pop       es

     call      disp_memory
     call      draw_sigrid
     mov       ah,08h
     int       21h

     call      bios_version
     call      draw_sigrid
     mov       ah,08h
     int       21h
     mov       ax,01A00h
     int       10h
     mov       bios_video,bx
     call      disp_video
     call      draw_sigrid
     mov       ah,08h
     int       21h
     call      get_dosinfo
     call      disp_dos
     call      draw_sigrid
     mov       ah,08h
     int       21h


     call      disp_current_drive

     call      display_siscreen



 _sysinfo_100:
     mov       in_sysinfo,1        ;flag for error handler
     mov       ah,08h
     int       21h
     and       al,al
     jnz        _sysinfo_200        ;not extended key, go ahead
     int       21h                 ;extended key -- continue

                                   ;al = 75 (left arrow) previous log drive
                                   ;al = 77 (right arrow) next logical drive
                                   ;     115 (^left) previous fixed
                                   ;     116 (^right) next fixed
     call      change_displayed_drive
     jc        _sysinfo_100        ;not one of the magic keys
     call      disp_current_drive
     jc        _sysinfo_100
;     call      display_siscreen
;     call      disp_global_drive
     call      draw_sigrid
     jmp       _sysinfo_100

 _sysinfo_200:
     mov  dl,restoredrive          ;restore drive, clear screen, return
	mov	ah,0Eh
	int	21h

     mov       al,c_normal
	call		clr_scr
	call		refresh_dir_disp
	call		put_menu_paths

     mov       in_sysinfo,0

     ret
 SYS_INFO      endp

;--------------------------------------------------------------------------
; change_displayed_drive
; On entry:
;    al contains extended ASCII keycode
;    cdrive contains current drive
;---------------------------------------------------------------------------
     tmp_cdrive     db   0

 change_displayed_drive  proc near

     mov       bl,cdrive
     mov       tmp_cdrive,bl

     mov       bl,al               ;save keycode
     mov       al,cdrive           ;get the current drive (within sysinfo)
     add       al,'A'              ;convert to letter
     mov       di,offset logical_drives
     mov       cx,logical_count
     repne     scasb               ;search for
     dec       di
                                   ;di now points to current drive in table
     cmp       bl,LARROW
     je        _cdd_20
     cmp       bl,CLARROW
     je        _cdd_20
     jmp       short _cdd_100
 _cdd_20:
     dec       di
     cmp       di,offset logical_drives
     jae       _cdd_50             ;still within the table
     add       di,logical_count    ;wrap around
 _cdd_50:
     cmp       bl,LARROW
     jne       _cdd_60             ;look at all logical drives
     cmp       logical_count,2     ;if only 2 logical drives, are never going
                                   ;to hit a hard drive
     je        _cdd_err
     cmp       byte ptr [di],'B'
     jbe       _cdd_20             ;is a floppy -- skip
 _cdd_60:
     mov       dl,byte ptr [di]
     sub       dl,'A'
     mov       cdrive,dl
     mov       ah,0Eh
     int       21h
     clc
     ret
 _cdd_100:
     cmp       bl,RARROW
     je        _cdd_120
     cmp       bl,CRARROW
     je        _cdd_120
     jmp       short _cdd_err
 _cdd_120:
     inc       di
     cmp       byte ptr[di],0       ;means we're past end
     jne       _cdd_150
     sub       di,logical_count         ;wrap around
 _cdd_150:
     cmp       bl,RARROW
     jne       _cdd_160                 ;look at all logical drives
     cmp       logical_count,2     ;if only 2 logical drives, are never going
                                   ;to hit a hard drive
     je        _cdd_err
     cmp       byte ptr [di],'B'
     jbe       _cdd_120
 _cdd_160:
     jmp       _cdd_60
 _cdd_err:
     stc
     ret
 change_displayed_drive  endp

 serial    STRUC
          info_level     dw   ?
          serial_num     dd   ?
          svolume        db   11 dup (?)
          fsystem        db   8 dup (?)
 serial    ENDS

 serial_info serial <>
 fatid     db   0
 fatid1    db "    Fixed",0
 fatid2    db "Removable",0

 ;---------------------------------------------------------------------------
 ;Displays current drive parameters
 ;Current drive can be changed with the arrow keys, etc.
 ;On entry to this function, the current drive has been set to the drive we
 ;   want
 ;---------------------------------------------------------------------------

 disp_current_drive proc near
JUMPS
     ;clear out changeable area

     mov       al,space
     mov       di,offset si_line10 + 47
     mov       cx,7
 _dcd_10:
     push      cx di
     mov       cx,15
     rep       stosb
     pop       di cx
     add       di,SI_LENGTH
     loop      _dcd_10


     mov       ah,19h
     int       21h                 ;current drive in al
     mov       cdrive,al           ;we use this with the lookup table
                                   ;A=0, B=1, etc.

     push      ax

     push      ds
     mov       dl,al
     inc       dl                  ;0=default in this case, not A
     mov       ah,1Ch
     int       21h                 ;get drive allocation info
     jc        _dcd_err
     mov       bl,byte ptr ds:[bx] ;store FAT ID byte
     pop       ds
     mov       fatid,bl
     mov       dx,cx               ;move sector size to dx
     xor       ah,ah               ;al=sectors per cluster, zero ah
     mul       dx                  ;get bytes per cluster
     mov       cx,6
     mov       di,offset DCLUSTER_LOC
     call      cformat

     pop       ax
     add       al,'A'
     mov       di,offset DLETTER_LOC
     stosb
     mov       di,offset DTYPE_LOC
     cmp       fatid,0F8h
     jne       _dcd_20
     mov       si,offset fatid1
     jmp       short _dcd_100
 _dcd_20:
     mov       si,offset fatid2
 _dcd_100:
     call      copy_null

     ;Get the serial info using undocumented DOS call

     mov       ax,DOS_VERSION
     cmp       al,4
     jb        _dcd_150            ;function only DOS 4 and above

     mov       ax,6900h
     mov       bl,0
     mov       dx,offset serial_info
     int       21h
     jc        _dcd_200

     ;convert the DWORD into hex digits

     mov       ax,word ptr [serial_info.serial_num+2]
     mov       bl,16
     mov       si,offset DSERIAL_LOC
     call      word_to_asce
     mov       al,'-'
     xchg      di,si
     stosb
     xchg      di,si
     mov       ax,word ptr [serial_info.serial_num]
     call      word_to_asce
 _dcd_150:
     call      read_volume
     mov       si,offset temp_volume
     mov       di,offset DVOL_LOC
     mov       cx,11
     rep       movsb
 _dcd_200:
     mov       ah,36h         ;get free disk space
     xor       dl,dl
     int       21h
     push      ax bx cx dx    ;save to recalculate w/ total clusters

     mul       cx             ;sectors/cluster * bytes/sector
     mul       bx             ;* available clusters
                              ;result now in DX:AX
     mov       cx,11
     mov       di,offset DFREE_LOC
     call      cformat

     pop       dx cx bx ax
     mov       bx,dx          ;copy total clusters
     mul       cx
     mul       bx

     mov       cx,11
     mov       di,offset DTOTAL_LOC
     call      cformat
     clc
     ret
 _dcd_err:
     pop       ds ax
     ;reset to the previous default drive
     mov       dl,tmp_cdrive
     mov       cdrive,dl
     mov       ah,0Eh
     int       21h
 _dcd_err2:
     stc

     ret
NOJUMPS
 disp_current_drive endp


 get_dosinfo        proc near

     mov       dos_files,0
     push      es
     mov       ah,52h
     int       21h            ;es:bx points to list of lists
     ;pointer to list of DOS file tables at 04h (DWORD)
     mov       ax,es:[bx+4]
     mov       dx,es:[bx+6]
     cmp       ax,0FFFFh
     jne       _gdos_50
     cmp       dx,0FFFFh
     jne        _gdos_50      ;error in finding info in list of lists
     pop       es
     mov       bx,032h
     mov       ax,es:[bx]
     mov       dos_files,ax   ;file handle count from PSP
     jmp       short _gdos_110

 _gdos_50:
     mov       es,dx          ;load segment
     mov       bx,ax

     mov       ax,es:[bx+4]
     add       dos_files,ax

     mov       dx,es:[bx+2]
     mov       ax,es:[bx]

     cmp       ax,0FFFFh
     jne        _gdos_50

 _gdos_100:
     pop       es
 _gdos_110:                   ;end of files determination
                              ;dos_files now contains info

     mov       ax,dos_version
     cmp       al,4
     jb        _gdos_200      ;for now, we only determine buffers if
                              ;we can get them from the list of lists
                              ;which means DOS 4 or later
     push      es
     mov       ah,52h
     int       21h

     mov       ax,es:[bx+03Fh]
     mov       dos_buff,ax
     pop       es
 _gdos_200:
     ;determine total environment space and amount used

     push      es
 _gdos_205:
     mov       ax,es:[16h]
     and       ax,ax
     jz        _gdos_210
     mov       bx,es
     cmp       ax,bx
     je        _gdos_210
     mov       es,ax
     jmp       _gdos_205
 _gdos_210:
     mov       ax,es:[02Ch]
     and       ax,ax
     jz        _gdos_220
     mov       bx,dos_version
     cmp       bh,19               ;check for DOS 3.2x
     jbe       _gdos_230
     cmp       bh,30
     jae       _gdos_230
 _gdos_220:
     mov       bx,es               ;get envseg another way
     mov       ax,bx
     dec       bx
     mov       es,bx
     add       ax,es:[3]
     inc       ax
 _gdos_230:
     dec       ax
     mov       es,ax
     mov       ax,es:[3]
     shl       ax,4
     mov       dos_env,ax
 _gdos_240:
     mov       ax,es
     inc       ax
     mov       es,ax
     xor       bx,bx
 _gdos_245:
     cmp       word ptr es:[bx],0
     je        _gdos_250           ;end of used section found
     inc       bx
     jmp       _gdos_245
 _gdos_250:
     inc       bx
     inc       bx
     mov       dos_env_used,bx

     pop       es


     ret
 get_dosinfo        endp

 logical_count      dw   0
 drive0             db   0
 drive1             db   0

;---------------------------------------------------------------------------
; Gets global (i.e. not stuff you access with the arrow keys) drive info to
; store
;---------------------------------------------------------------------------

 get_global_drive   proc near

     mov       num_floppies,0



     mov       si,offset logical_drives      ;count them
     call      str_len
     mov       logical_count,cx

     call      test_for_cmos  ;returns carry set if CMOS not found
     jc        _ggd_200

     .READ_CMOS 010h          ;read diskette drive byte
     mov       floppies,al
     mov       ah,al          ;count them
     and       al,11110000b
     jz        _ggd_60
     inc       num_floppies
 _ggd_60:
     and       ah,1111b
     jz        _ggd_70
     inc       num_floppies
 _ggd_70:
     jmp       short _ggd_300
 _ggd_200:                    ;need to get our info w/o CMOS

     .RETURN_EQUIP  ax
     inc       num_floppies   ;BIOS flag doesn't know about 0 floppies
     test      ax,080h        ;test bit 7
     jz        _ggd_300       ;not set
     inc       num_floppies   ;set means 2 drives
 _ggd_300:
     call      test_for_cmos  ;returns carry set if CMOS not found
     jc        _ggd_400

     .READ_CMOS     012h
     mov       ah,al
     and       al,11110000b
     shr       al,4           ;al now contains type of Drive 0
     cmp       al,0Fh
     jne       _ggd_320
     .READ_CMOS 19h           ;get extended drive type
 _ggd_320:
     mov       drive0,al
     and       ah,1111b
     cmp       ah,0Fh
     jne       _ggd_340
     .READ_CMOS 1Ah
     mov       ah,al
 _ggd_340:
     mov       drive1,ah
 _ggd_400:
  _ggd_ret:
     ret
 get_global_drive   endp

;-----------------------------------------------------------------------------
; Fills the address logical drives with the logical drives present
; in the system.  uses byte cdrive to store default drive so can restore
;-----------------------------------------------------------------------------
 logical_drives     db 27 dup(0)

 set_logical_drives proc near

     mov       ah,19h
     int       21h
     mov       cdrive,al        ;store current default drive so we can restore

     mov       cx,26          ;cycle through all logical drives
     mov       dl,0           ;start w/ 'A' (drive 0)
     mov       di,offset logical_drives
 _sld_25:
     mov       ah,0Eh         ;set default disk drive
     int       21h

     mov       ah,19h         ;get default drive
     int       21h

     cmp       al,dl          ;was the drive stored?
     jne       _sld_50

     add       al,'A'         ;if yes, convert and store
     stosb
 _sld_50:
     inc       dl             ;advance to next drive
     loop      _sld_25

     mov       dl,cdrive      ;restore existing default
     mov       ah,0Eh
     int       21h
     ret

 set_logical_drives endp

;-----------------------------------------------------------------------------
; Tests to see if CMOS exists on the system.  Returns carry clear if yes,
; carry set if no.
;-----------------------------------------------------------------------------

 test_for_CMOS proc near
     .READ_CMOS  06h
     mov       temp,al        ;keeps value to return later
     .WRITE_CMOS 06h,0AAh
     .READ_CMOS 06h
     cmp       al,0AAh
     jne       _tfc_fail
     .WRITE_CMOS 06h,055h
     .READ_CMOS 06h
     cmp       al,055h
     jne       _tfc_fail
     mov       ah,0C0h
     push      es
     int       15h
     pop       es
     jnc       _tfc_success
     push      es
     mov       ax,0FFFFh
     mov       es,ax
     mov       bx,0Eh
     mov       al,byte ptr es:[bx]
     pop       es
     cmp       al,0FDh
     jae       _tfc_fail

 _tfc_success:
     clc
     jmp short _tfc_ret
 _tfc_fail:
     stc
 _tfc_ret:
     pushf
     .WRITE_CMOS 06h,temp
     popf
     ret
 test_for_CMOS      endp

 DISPLAY_SISCREEN   proc near

     mov       al,c_normal
	call		clr_scr

     mov       al,c_intense
     mov       cx, 0302h      ;row column upper left
     mov       dx, 0519h      ;row column lower right
     call      clr_window

     mov       al,c_intense
     mov       cx, 0702h      ;row column upper left
     mov       dx, 1119h      ;row column lower right
     call      clr_window

     mov       al,c_intense
     mov       cx, 1302h      ;row column upper left
     mov       dx, 1419h      ;row column lower right
     call      clr_window

     mov       al,c_intense
     mov       cx, 1602h      ;row column upper left
     mov       dx, 1719h      ;row column lower right
     call      clr_window

     mov       al,c_intense
     mov       cx, 031Bh      ;row column upper left
     mov       dx, 092Dh      ;row column lower right
     call      clr_window

     mov       al,c_intense
     mov       cx, 0B1Bh      ;row column upper left
     mov       dx, 172Dh      ;row column lower right
     call      clr_window

     mov       al,c_intense
     mov       cx, 032Fh      ;row column upper left
     mov       dx, 114Ch      ;row column lower right
     call      clr_window

     mov       al,c_intense
     mov       cx, 142Fh      ;row column upper left
     mov       dx, 154Ch      ;row column lower right
     call      clr_window

     call      disp_cpu
     call      disp_hardware
     call      disp_DOS
     call      disp_memory
     call      disp_video
     call      disp_global_drive


     call      draw_sigrid
;     mov       al,c_normal
;     xor       cx,cx
;     mov       dx,044Fh
;     call      clr_window
;     mov       al,c_normal
;     mov       cx,1700h
;     mov       dx,184Fh
;     call      clr_window


     ret
 DISPLAY_SISCREEN   endp


 si_line0  db ' Directory Freedom 4.60   Copyright 1992 Gordon Haff and Bit Masons Consulting',0
 si_line1  db '        System Information Screen     Press <esc> or <space> to return        ',0
 si_line2  db ' CPUDOSDisks Ŀ',0
 si_line3  db '  CPU:                    DOS version:       Logical Drives               ',0
 si_line4  db '  Running at ??    MHz                                                    ',0
 si_line5  db '  NDP:                    Environ            CMOS Information             ',0
 si_line6  db ' MemoryĴ Files     Buf         A:                        ',0
 si_line7  db '        KB  Total   Free  Verify    Break       B:                        ',0
 si_line8  db '  Convl                   Win       DV          Type  HD1:      HD2:      ',0
 si_line9  db '  EMS                                                                     ',0
 si_line10 db '  Extnd                  HardwareĴ               Drive letter   ',0
 si_line11 db '  XMS                     Bus                              Volume Label   ',0
 si_line12 db '  XMM driver              Mouse                            Serial #       ',0
 si_line13 db '  HMA none        K free  101-key Supprtd                  Type           ',0
 si_line14 db '                            Kybd  Present                  Cluster size   ',0
 si_line15 db '  DF Program     KB Conv                                   Total space    ',0
 si_line16 db '     Directry    KB Conv  LPT1  not present                Free space     ',0
 si_line17 db '     for shell      KB    LPT2  not present                               ',0
 si_line18 db ' BIOSĴ LPT3  not present  L/R cursor keys cycle disks  ',0
 si_line19 db '          (C)                               VideoĴ',0
 si_line20 db '                          COM1  not present  Active:                      ',0
 si_line21 db '  SoundĴ COM2  not present  Inactive:                    ',0
 si_line22 db '  SBlaster                COM3  not present ',0
 si_line23 db '  Adlib                   COM4  not present                                ',0
 si_line24 db '                                ',0

 SI_LENGTH     EQU  79
 CPU_LOC       EQU  si_line3 + 8
 NDP_LOC       EQU  si_line5 + 8
 STEP_LOC      EQU  si_line3 + 16
 FREQ_LOC      EQU  si_line4 + 14
 SBLAST_LOC    EQU  si_line22 + 13
 SBPORT_LOC    EQU  si_line22 + 17
 ADLIB_LOC     EQU  si_line23 + 10
; MPU_LOC       EQU  si_line23 + 24
 BUS_LOC       EQU  si_line11 + 41
 MOUSE_LOC     EQU  si_line12 + 44
 WEITEK_LOC    EQU  si_line4 + 17
 KYBD_BIOS_LOC EQU  si_line13 + 44
 KYBD_PRES_LOC EQU  si_line14 + 44
; GAME_LOC      EQU  si_line14 + 42
 PRINTER_LOC   EQU  si_line16 + 33
 COMM_LOC      EQU  si_line20 + 33
 DOS_LOC       EQU  si_line3 + 41
 DATE_LOC      EQU  si_line4 + 28
 TIME_LOC      EQU  si_line4 + 38
 BREAK_LOC     EQU  si_line7 + 44
 VERIFY_LOC    EQU  si_line7 + 35
 MEMMAN_LOC    EQU  si_line9 + 28
 XMSV_LOC      EQU  si_line11 + 7
 XMMV_LOC      EQU  si_line12 + 14
 XMSF_LOC      EQU  si_line11 + 19
 HMA_LOC       EQU  si_line13 + 7
 HMAF_LOC      EQU  si_line13 + 17
 EMSVER_LOC    EQU  si_line9 + 7
 DFPROG_LOC    EQU  si_line15 + 15
 DFDIR_LOC     EQU  si_line16 + 15
 DFSHELL_LOC   EQU  si_line17 + 17
 CONV_LOC      EQU  si_line8 + 13
 EMS_LOC       EQU  si_line9 + 13
 EXT_LOC       EQU  si_line10 + 13
 BIOS_LOC      EQU  si_line19 + 2
 BIOS2_LOC     EQU  si_line20 + 2
 VIDEOA_LOC    EQU  si_line20 + 58
 VIDEOB_LOC    EQU  si_line21 + 58
 NOTCMOS_LOC   EQU  si_line5 + 48
 MSWIN_LOC     EQU  si_line8 + 32
 DVIEW_LOC     EQU  si_line8 + 41
 DOS_FILES_LOC EQU  si_line6 + 34
 DOS_BUFF_LOC  EQU  si_line6 + 42
 DOS_ENV_LOC   EQU  si_line5 + 36
 FLOP1_LOC     EQU  si_line6 + 54
 FLOP2_LOC     EQU  si_line7 + 54
 HD1_LOC       EQU  si_line8 + 62
 HD2_LOC       EQU  si_line8 + 72
 LOGDRV_LOC    EQU  si_line4 + 49

 DLETTER_LOC   EQU  si_line10 + 59
 DVOL_LOC      EQU  si_line11 + 49
 DSERIAL_LOC   EQU  si_line12 + 51
 DTYPE_LOC     EQU  si_line13 + 51
 DCLUSTER_LOC  EQU  si_line14 + 54
 DTOTAL_LOC    EQU  si_line15 + 49
 DFREE_LOC     EQU  si_line16 + 49


 DRAW_SIGRID        proc near

	mov		cx,25
	mov		di,0 CROW 0
     mov       si,offset si_line0
_grid2:
	push		cx
	call		str_len
	.put_cx_chars_reg
	add		si,cx
	inc		si
	add		di,ROW_LENGTH
	pop		cx
     loop      _grid2

     ret
 DRAW_SIGRID        endp

;---------------------------------------------------------------------------
; This routine gets various system info up front (primarily related to XMS
; and EMS memory most of which we need anyway for the memory allocation
; routines
;----------------------------------------------------------------------------
 SETUP_SYSINFO      PROC NEAR

     mov       ax,4300h
     int       2fh
     cmp       al,80h
     jne       _ssinfo50                ;al=80h if XMS driver present

 _ssinfo20:
     push      es
     mov       ax,4310h
     int       2fh
     mov       word ptr xmsaddr,bx
     mov       word ptr xmsaddr+2,es    ;save far pointer to entry point
     pop       es

     mov       ah,8
     call      xmsaddr
     mov       xmstotal,dx
     mov       xmslargest,ax

     or        highmem_flag,01b

     mov       ah,0
     call      xmsaddr
     mov       xmsver,ax
     mov       xmmver,bx
     mov       hmaavail,dl
     cmp       dx,1
     jne       _ssinfo50                ;dx = 0 no HMA

     mov       ax,dos_version
     cmp       al,5
     jb        _ssinfo30                ;don't look for below DOS 5
     mov       ax,04A01h                ;hma in use by DOS 5?
     push      es
     int       2fh
     pop       es
     and       bx,bx                    ;bx = 0.  DOS 5 isn't using.
     jz        _ssinfo30
     mov       dos5hma,1
     mov       cl,10
     mov       ax,bx                    ;convert HMA free bytes to KB
     shr       ax,cl
     mov       hmafree,ax               ;Kbytes free

 _ssinfo30:
     mov       ah,1                     ;attempt to allocate hma
     mov       dx,0FFFFh
     call      xmsaddr
     cmp       ax,1                     ;successful?
     jne       _ssinfo50                ;allocation failed, assume in use
     mov       hmaavail,2
                                        ;now free it
     mov       ah,2
     call      xmsaddr



 _ssinfo50:

 ;first we test for presence of expanded memory using DOS Open and IOCTL
 ;functions (per Chap 2 Extending DOS)

     mov       dx,offset emmname
     mov       ax,3D00h            ;open
     int       21h
     jc        _ssinfo100
                                   ;open successful, make sure isn't a file
     mov       bx,ax               ;bx=handle from open
     mov       ax,4400h            ;IOCTL get device info
     int       21h
     jc        _ssinfo100

     and       dx,80h              ;bit 7=1 if char device
     jz        _ssinfo100

     mov       ax,4407h            ;EMM present, make sure available
     int       21h
     jc        _ssinfo100

     mov       ah,3Eh              ;close
     int       21h
     jc        _ssinfo100

     mov       ah,40h              ;get EMS system status
     int       67h
     or        ah,ah
     jnz       _ssinfo100

     mov       ems_active,1         ;used later when looking at EMS managers
     mov       ah,46h              ;check EMM version
     int       67h
     or        ah,ah
     jnz       _ssinfo100

     mov       emsversion,al
     cmp       al,32h              ;make sure EMS 3.2+
     jb       _ssinfo100

     or        highmem_flag,010b   ;we have the proper ems

     mov       ah,42h              ;get number of EMS pages
     int       67h
     or        ah,ah
     jnz       _ssinfo100
     mov       tpages,dx
     mov       apages,bx


 _ssinfo100:
     ret

 SETUP_SYSINFO      ENDP



;------------------------------------------------------------------------
; Gets DF Memory utilization for display
;------------------------------------------------------------------------

 get_dfmem          proc near
     mov       used_mem,0
     cmp       mem_use,0
     jne       _gdfm_50
     mov       ax,dbs_start
     shr       ax,4
     jmp       short _gdfm_60
 _gdfm_50:
     mov       ax,diralloc
 _gdfm_60:
     add       used_mem,ax         ;program conventional memory usage
     mov       cl,6
     shr       ax,cl
     mov       dfprog_mem,ax       ;store for later display
                                   ;now we do the directory memory usage
     mov       ax,dbs_end
     sub       ax,dbs_start
     push      ax                  ;save memory used
     mov       cl,10
     shr       ax,cl               ;convert to KB
     mov       dfdir_mem,ax        ;save for later display
                                   ;mem_use 1 = XMS
                                   ;        2 = EMS
                                   ;        4 = Conv
     pop       ax
     ;now we figure how much we have for the shell
     mov       bx,used_mem
     mov       ax,avail_mem
     sub       ax,bx
     mov       cl,6
     shr       ax,cl
     cmp       mem_use,2
     je        _gdfm_100
     cmp       mem_use,1
     je        _gdfm_100

                                   ;so long as directory doesn't need conl
                                   ;memory on swap, go ahead
     sub       ax,dfdir_mem        ;otherwise, subtract out directory
 _gdfm_100:
     mov       dfshell_mem,ax

     ret
 get_dfmem          endp

;------------------------------------------------------------------------
; Checks for presence of EMM386 memory manager.  This is another
; quick and dirty.  Returns y/n in bl
;------------------------------------------------------------------------

 check_emm386       proc near

     cmp       ems_active,1
     jne       _cemm386_ret
     mov       ax,0FFA5h
     int       67h
     cmp       ax,0845Ah
     jne       _cemm386_ret
     mov       bl,3
     ret
 _cemm386_ret:
     mov       bl,0
     ret
 check_emm386       endp

;------------------------------------------------------------------------
; Checks for presence of 386^MAX memory manager.  To save a big structure
; and other things required to absolutely confirm presence (and to get
; the version number) we just do a quick look and return y/n in bl
;------------------------------------------------------------------------
 sig386Max     db   '386MAX$$',0

 check_386max       proc near

     mov       ax,03D00h
     mov       dx,offset sig386MAX
     inc       dx
     int       21h
     jc        _c386m_ret
     mov       bl,2
     ret
 _c386m_ret:
     mov       bl,0
     ret
 check_386max       endp



;------------------------------------------------------------------------
; Checks for presence of QEMM memory manager.  Returns the following
; results in ax and bx:
; bx = 0 if not present, 1 if present
; ah/al = major and minor versions in ASCII form
;------------------------------------------------------------------------
     QEMMid    db   0
     qemmaddr   dd   ?

 check_qemm         proc near

     push      es
     mov       portok,0
     mov       QEMMid,0D2h

 _cqemm_50:
     xor       ax,ax
     mov       ah,QEMMid
     mov       bx,05144h
     mov       cx,04D45h
     mov       dx,04D30h
     int       2Fh
     cmp       al,0FFh
     jne       _cqemm_100
     cmp       bx,04D45h
     jne       _cqemm_100
     cmp       cx,04D44h
     jne       _cqemm_100
     cmp       dx,05652h
     jne       _cqemm_100
     mov       portok,1
     jmp       short _cqemm_200
 _cqemm_100:
     cmp       QEMMid,0FFh
     jae       _cqemm_120
     inc       QEMMid
     jmp       short _cqemm_130
 _cqemm_120:
     mov       QEMMid,0C0h
 _cqemm_130:
     cmp       QEMMid,0D2h
     jne       _cqemm_50
 _cqemm_200:
     xor       bx,bx
     mov       bl,portok
     and       bx,bx
     jz        _cqemm_ret          ;QEMM not found

     mov       ah,QEMMid
     mov       al,1
     mov       bx,05145h
     mov       cx,04D4Dh
     mov       dx,03432h
     int       2Fh                 ;alters es!
     cmp       bx,04F4Bh           ;OK?
     je        _cqemm_300
     xor       bx,bx
     jmp       _cqemm_ret
 _cqemm_300:
     mov       ah,3
     mov       word ptr qemmaddr+2,es
     mov       word ptr qemmaddr,di
     call      qemmaddr
     mov       ax,bx
     call      unbcd
     xchg      ah,al
     call      unbcd
     xchg      ah,al

     and       bl,bl
     jz        _cqemm_ret
     mov       bl,1

 _cqemm_ret:
     pop       es
     ret
 check_qemm         endp

;---------------------------------------------------------------------------
; Tries to get the extended memory available in the system.
; If DOS 4 or greater is present, tries to obtain through the List of Lists.
; If this fails, just get it through the BIOS (which may well say 0)
;
; List of Lists method from PC Techniques Dec/Jan 92. Hax 92 by
;     Rich Sadowsky
;---------------------------------------------------------------------------
 get_extmem         proc near

     mov       ax,dos_version
     cmp       al,4
     jb        _gext_100           ;can only do this with DOS4 or greater

     push      es
     mov       ah,52h
     int       21h                 ;get pointer (es:bx) to list of lists
     add       bx,45h
     mov       ax,es:[bx]
     pop       es
     cmp       ax,0FFFFh           ;call failure.  Possibly including under
                                   ;OS/2 compatibility box
     je        _gext_100
     jmp       short _gext_200
 _gext_100:
     mov       ah,088h
     int       15h
     jc        _gext_ret
 _gext_200:
     mov       extmem,ax
 _gext_ret:
     ret
 get_extmem         endp

;------------------------------------------------------------------------
; Sets various bits in siFlags as a function of whether various pieces of
; equipment are present in the system.  Returns Sound Blaster port in
; BX if a Sound Blaster board in present
;------------------------------------------------------------------------
     sbfound   db   0
     midifound db   0

JUMPS
 GET_SIFLAGS        proc near

;     cmp	bypass_si,1
;     je	        _gsi_ret

     cmp       mswin_version,0
     jne       _gsi_300            ;MSWIN 3.1+ will give error if DOS
                                   ;apps try to use Sound Blaster

     ;Adlib board
     ;I have no idea what the purpose of the following black magic is.
     ;See Infoplus Page 8

     mov       dx,0388h
     in        al,dx
     mov       bl,al     ;bl = xbyte2
     mov       al,0BDh
     out       dx,al
     in        al,dx
     in        al,dx
     in        al,dx
     in        al,dx
     inc       dx
     in        al,dx
     mov       bh,al     ;bh = xbyte3
     mov       al,0
     out       dx,al
     mov       cx,36
     dec       dx
 _gsi_100:
     in        al,dx
     loop      _gsi_100

     and       al,7
     push      ax
     mov       al,bl
     out       dx,al
     inc       dx
     mov       al,bh
     out       dx,al

     pop       ax

     cmp       al,6
     jne       _gsi_150
     or        siFlags,ADLIB_MASK
 _gsi_150:                         ;end of Adlib detection

     ;Now we do the Sound Blaster thing

     mov       temp,0              ;used to flag 1st pass through repeat
     push      bp
     mov       sbfound,0
     mov       cx,6                ;loop counter
     mov       sbport,0200h
     xor       dx,dx               ;xword2
 _gsi_160:
     cmp       sbfound,1
     je        _gsi_200
     xor       bp,bp               ;xword1
     mov       dx,sbport
     add       dx,010h
     mov       sbport,dx
     add       dx,0Ch
     mov       portok,0
 _gsi_165:
     cmp       bp,0201h  ;do while xword1(bp) < 0201h and not sbfound
     jae       _gsi_170
     cmp       portok,0
     jne       _gsi_170
 _gsi_166:
     in        al,dx
     and       al,080h
     and       al,al
     jnz       _gsi_167
     mov       portok,1
 _gsi_167:
     inc       bp
     jmp       _gsi_165
 _gsi_170:
     cmp       portok,0
     je        _gsi_177      ;port not ok -- loop to next port

     in        al,dx
     mov       bl,al          ;bl = xbyte3
     mov       al,0D3h
     out       dx,al
     push      cx             ;save main loop counter
     mov       cx,01000h      ;setup time delay loop
 _gsi_171:
     nop
     loop      _gsi_171
     pop       cx
     mov       dx,sbport
     add       dx,6
     mov       al,1
     out       dx,al
     in        al,dx
     in        al,dx
     in        al,dx
     in        al,dx
     mov       al,0
     out       dx,al
     mov       dx,sbport
     add       dx,0Eh
     xor       bh,bh
 _gsi_173:                    ;repeat until xbyte2 = 10 or portok
     cmp       temp,1
     jne       _gsi_175
     cmp       bh,010h        ;bh = xbyte2
     je        _gsi_180
     cmp       portok,0
     jne       _gsi_180
 _gsi_175:
     mov       temp,1
     mov       portok,0
     mov       bp,0
 _gsi_175A:
     cmp       bp,0201h
     jae       _gsi_176A
     cmp       portok,0
     jne       _gsi_176A

     in        al,dx          ;do while bp < 0201h and not portok
     and       al,080h
     cmp       al,080h
     jne       _gsi_176
     mov       portok,1
 _gsi_176:
     inc       bp
     jmp       _gsi_175A
 _gsi_176A:
     cmp       portok,1
     jne       _gsi_177

     push      dx
     mov       dx,sbport
     add       dx,0Ah
     in        al,dx
     pop       dx
     cmp       al,0AAh
     jne       _gsi_177
     mov       sbfound,1
 _gsi_177:
     cmp       sbfound,1
     je        _gsi_200
     jmp       _gsi_190
 _gsi_178:
     jmp       _gsi_173
 _gsi_180:
     mov       al,bl
     out       dx,al
 _gsi_190:
     loop      _gsi_160

 _gsi_200:
     pop       bp
     cmp       sbfound,1
     jne       _gsi_210
     or        siFlags,SBLASTER_MASK
 _gsi_210:
 _gsi_300:
 _gsi_400:
     .RETURN_EQUIP  ax
     test      ah,10000b
     jz        _gsi_420
     or        siFlags,GAMEB_MASK
 _gsi_420:
     cmp       bit32,1       ;is a 32-bit processor?
     jne       _gsi_500      ;Weitek test only works with 32-bit
     call      weitek
     test      bx,01b
     jz        _gsi_500      ;no weitek
     or        siFlags,WEITEK_MASK
 _gsi_500:
     sti
 _gsi_ret:
     ret
 GET_SIFLAGS        endp
NOJUMPS


;------------------------------------------------------------------------
; Fills in the base addresses of the printer ports
;  printport_addr   (3) word
;------------------------------------------------------------------------
 GET_PRINTPORTS     proc near
     .return_equip  ax
     mov       cl,14
     shr       ax,cl     ;# of parallel printer adapters is in bits 14-15
     and       ax,011b   ;mask out all other bits
     mov       cx,ax     ;cx now has number of printer ports

     mov       bx,offset printport_addr
     and       cx,cx
     jz        _gprint_ret
     and       cx,011b        ;only look at up to three
     mov       dx,08h         ;printer addresses at offset 8h in BIOS data area
 _gprint_100:
     mov       ax,040h
     push      es
     mov       es,ax
     xchg      bx,dx
     mov       ax,es:[bx]
     xchg      bx,dx
     mov       [bx],ax
     inc       bx
     inc       bx
     inc       dx
     inc       dx
     pop       es
     loop      _gprint_100
 _gprint_ret:
     ret
 GET_PRINTPORTS     endp

;-------------------------------------------------------------------------
; Get the base addresses and UART type of the comm ports
; Fills in two arrays:
;  commport_addr    (4) word
;  commport_type    (4) byte
;                        0:  8250
;                        1:  16450
;                        2:  16550
;                        3:  16550A
;                        4:  ????
;-------------------------------------------------------------------------
 GET_COMMPORTS      proc near

JUMPS
     .return_equip  ax
     mov       cl,9
     shr       ax,cl     ;# of comm ports is in bits 9-11
     and       ax,0111b  ;mask out all other bits
     mov       cx,ax     ;cx now has number of com ports

     mov       bx,offset commport_addr
     and       cx,cx
     jz        _gcomm200
     dec       cx        ;change to 0-3 base
     and       cx,011b   ;never be greater than 3 (4 comm ports)
     inc       cx
     xor       dx,dx     ;zero out dx -- is positive counter from 0
 _gcomm100:
     mov       ax,040h   ;BIOS segment
     push      es
     mov       es,ax
     xchg      bx,dx
     mov       ax,es:[bx]  ;1st 4 words in 040h are commport addr
     xchg      bx,dx
     mov       [bx],ax     ;store commport addr in array

     inc       ax
     inc       ax

     push      bx dx cx
     mov       dx,ax       ;dx now contains port number
     in        al,dx
     mov       bl,al
     mov       al,0C1h
     out       dx,al
     in        al,dx
     mov       bh,al      ;per Infoplus bl = xbyte2, bh = xbyte3
     mov       al,bl
     out       dx,al      ;return port to normal

     and       bh,0C0h
     mov       cl,6
     shr       bh,cl

     cmp       bh,3
     jne       _gcomm150
     jmp       short _gcomm180
 _gcomm150:
     cmp       bh,2
     jne       _gcomm160
     jmp       short _gcomm180
 _gcomm160:
     cmp       bh,3
     jbe       _gcomm170
     mov       bh,4
     jmp       short _gcomm180
 _gcomm170:
     cmp       bh,1
     jne       _gcomm171
     jmp       short _gcomm180
 _gcomm171:              ;the tricky part -- distinguish between 16450 & 8250
     add       dx,5      ;dx now = port addr + 7
     in        al,dx
     mov       bl,al
     mov       al,0FAh
     out       dx,al
     nop                ;might need a couple of cycles here
     nop
     in        al,dx
     cmp       al,0FAh
     jne       _gcomm_is8250
     mov       al,0AFh
     out       dx,al
     nop
     nop
     in        al,dx
     cmp       al,0AFh
     jne       _gcomm_is8250
     mov       al,bl
     out       dx,al
     mov       bh,1            ;is a 16450
     jmp       short _gcomm180

 _gcomm_is8250:
     mov       bh,0
 _gcomm180:
     ;at this point bh contains the UART code
     mov       ax,bx
     pop       cx dx
     mov       bx,offset commport_type
     push      dx
     shr       dx,1      ;divide by 2 for byte rather than word
     add       bx,dx
     mov       [bx],ah
     pop       dx
     pop       bx
     inc       bx
     inc       bx
     inc       dx
     inc       dx
     pop       es
     loop      _gcomm100   ;loop until done

_gcomm200:
     ret
NOJUMPS
 GET_COMMPORTS      endp

;-------------------------------------------------------------------------
; Get the type of system bus.  Returns value in al per:
;   1=XT bus
;   2=ISA
;   3=EISA
;   4=MCA
;-------------------------------------------------------------------------
 GET_BUSTYPE        proc near

     push      es
     mov       ah,0C0h
     int       15h            ;es:bx now points to system config area

     ;Pass 7 -- bx+5 rather than bx+4?
     mov       al,es:[bx+5]   ;stores submodel byte
     pop       es
     test      al,02h
     jz        _gb_no_mca     ;per Infoplus.  If I interpreted correctly a set
                              ;bit 2 indicates MCA bus
     mov       al,4
     ret
 _gb_no_mca:
     ;now onto MCA.  Use the access EISA system infomation BIOS call
     mov       ax,0D800h
     mov       cl,0           ;slot number
     int       15h
     jnc       _gb_is_eisa    ;function worked so assume is eisa

     cmp       ah,86h         ;Invalid BIOS routine call
     jne        _gb_is_eisa   ;Failed but for reason other than BIOS
                              ;call failure

     ;XT bus detection logic would go here
;     mov       al,1
;     jmp       short _gbus_ret
 _gb_100:
     mov       al,2
     ret

 _gb_is_eisa:
               ;for time being assume is ISA.  Probably a way to distinguish
               ;from an 8-bit XT bus, but don't know what it is
     mov       al,3

 _gbus_ret:
     ret
 GET_BUSTYPE        endp

;----------------------------------------------------------------------------

;--------------------------------------------------------------------------
; DF_CPUID.  Identifies type of CPU
; Adapted from Robert L. Hummel's CPUID.ASM in The Processor and Coprocessor
;    page 75.  See source for full comments
;
; 80188/86 and NEC stuff from CHIPS.ASM by the following
;       Copr. 1987      Pat Shea - Psi! (that Copr. is on there cuz my
;                                        lawyer sez I should, but feel
;                                        free to hack away!!!    pats.)
;
;
; Returns AH
;
;    01   8086 or 8088
;    05   80186/188
;    06   NEC V20/V30
;    02   80286
;    03   80386DX or SX
;    04   80486DX or SX
;
; Returns AL
;
;    00   No NDP
;    01   8087
;    02   80287
;    03   80387DX or SX
;    04   80486DX or SX
;
; Note: A return code of 0400 indicates a 80486SX w/o an 80487SX
;------------------------------------------------------------------------------
 NDP_STATUS    DW   -1
;--------------------------------------------------------------------------

 DF_CPUID     proc near

     mov  dx,0100h
     mov  word ptr [ndp_status],-1


     pushf
     pop  ax

     and  ah,0Fh

     push ax
     popf

     pushf
     pop  ax

     and  ah,0F0h
     cmp  ah,0F0h        ;if set, CPU is an 8086 or 8088

     jne   _cpuid_100     ;a 286 or better

; here's we try to figger out whether it's an 80188/80186, an 8088/8086
;   or an NEC V20/V30 - 'couple of slick tricks from Clif Purkiser.....

dig:    mov    AX, 0ffffh     ; load up AX
        mov    CL, 33         ; HERE's the FIRST TRICK.... this will
                              ;   shift everything 33 times if it's
                              ;   8088/8086, or once for a 80188/80186!
        shl    AX, CL         ; on a shift of 33, all bits get zeroed
        jz     digmor         ;   out so if anything is left ON it's
                              ;   gotta be an 80188/80186
        mov    dx,0500h       ; save 0500h in DX cuz it's an 80188/80186
        jmp    SHORT cpuid_2  ;   and bail out

digmor: xor    AL,AL          ; clean out AL to set ZF
        mov    AL,40h         ; ANOTHER TRICK.... mul on an NEC duz NOT
        mul    AL             ;   effect the zero flag BUT on an Intel
        jz     gotNEC         ;   8088/8086, the zero flag gets thrown
        mov    dx,0100h       ; 0100 into DX cuz it's an Intel 8088/8086
        jmp    SHORT cpuid_2  ;   and bail out

gotNEC: mov    DX,0600h       ; it's an NEC V20/V30 so save 600 in DX


 _cpuid_100:
     inc  dh

     pushf
     pop  ax

     or   ah,0F0h

     push ax
     popf

     pushf
     pop  ax

     and  ah,0F0h        ;if all clear, must be a 286

     jz   cpuid_2

.386

     mov  bit32,1        ;indicate for other functions have a 32-bit

     inc  dh

     mov  ecx,esp
     and  esp,not 3

     pushfd
     pop  eax

     mov  ebx,eax

     xor  eax,00040000h

     push eax
     popfd

     pushfd
     pop  eax

     xor  eax,ebx        ;if bit not changed, must be 386

     jz   cpuid_1

;----------------------------------------------------------------------
;CPU is 486 or later.  Restore flags.
;----------------------------------------------------------------------

     inc  dh
     push ebx
     popfd

;-----------------------------------------------------------------------
; Restore the original stack pointer
;-----------------------------------------------------------------------

 cpuid_1:

     mov  esp,ecx

.8086

 cpuid_2:

     fninit

     fnstsw   word ptr [ndp_status]

     cmp       byte ptr [ndp_status],0       ;if !0, no NDP
     jne       cpuid_3


     fnstcw    word ptr [ndp_status]
     and       word ptr [ndp_status],103Fh
     cmp       word ptr [ndp_status],3Fh
     jne       cpuid_3

     mov       dl,dh
     cmp       dh,3
     jne       cpuid_3

     fld1
     fldz
     fdiv

     fld       st
     fchs
     fcompp

     fstsw     [ndp_status]
     mov       ax,word ptr [ndp_status]
     sahf

     jne       cpuid_3

     dec       dl

 cpuid_3:

     mov       ax,dx
     ret

 DF_CPUID     endp

;--------------------------------------------------------------------------
; 386STEP identifies the stepping level of a 386 chip as B0 or earlier,
; B1, or D0 or higher.  Thanks to Bob Moote and Richard Smith of Phar Lap
; Software for their advice and assistance in preparing this code.
;
; Copyright (c) 1992 Jeff Prosise
; First published in PC Magazine, February 11, 1992
;
;For our purposes here, we've stripped out the 386 and 486 detection code.
; It is assumed that the calling routine will make certain to only call
; this routine if a 386 is detected
; Returns 0 for B0 or earlier   in al,
;         1     B1
;         2     D0 or later
;
;--------------------------------------------------------------------------

 stepmsg             db      "Stepping level $"
.386P

 step386       proc near
               push      es
;
; Test for a stepping level of B0 or earlier by executing an XBTS instruction
; and trapping invalid opcode exceptions.
;
                mov     ax,3506h                ;Get interrupt 06h vector
                int     21h                     ;  in ES:BX
                mov     ax,2506h                ;Point interrupt 06h to
                mov     dx,offset invalid_op    ;  internal address
                int     21h
                mov     dx,0                    ;Set DX to 0
                mov     cx,1                    ;Initialize CX for XBTS
                db      0Fh,0A6h,00h            ;Execute XBTS (Step B1 or
                                                ;  later will branch to
                                                ;  label INVALID_OP)
                mov     dx,1                    ;Set DX to 1
               jmp       short iop_a
 invalid_op:
               add      sp,6                   ;invalid op codes put on stack?
                                               ;didn't affect orig PC mag
                                               ;code because it was standalone
                                               ;program so stack pointer at
                                               ;end didn't need to match at
                                               ;start.  added w/ df integratiom
 iop_a:
                push    ds                      ;Save DS on the stack
                push    dx                      ;Save DX also
                mov     dx,bx                   ;Restore the original
                mov     ax,es                   ;  interrupt 06h vector
                mov     ds,ax
                mov     ax,2506h
                int     21h
                pop     dx                      ;Restore DX and DS
                pop     ds
                or      dx,dx                   ;Branch if DX is still set
                jz      b1                      ;  to 0
                mov     al,0                    ;Load bl with step b0 code
                jmp     short step386_ret       ;  and exit if it's not
;
; Test for stepping level B1 chips by seeing if ECX is properly decremented
; after a REP INSB instruction followed by a POP.
;
b1:             cld                             ;Clear direction flag
                mov     ax,cs                   ;Point ES:EDI to scratch
                mov     es,ax                   ;  buffer
                mov     edi,offset stepmsg
                mov     dx,80h                  ;Load I/O port address
                mov     ecx,1                   ;Initialize ECX
                push    ax                      ;Push AX onto the stack
                rep     insb                    ;Read string from I/O port
                pop     ax                      ;Pop AX off the stack
                or      ecx,ecx                 ;Branch if ECX is equal to 0
                jz      d0
                mov     al,1                    ;Load al with step b1
                jmp     short step386_ret       ;Jump to exit
;
; By process of elimination, the chip must be step D0 or higher.
;

d0:             mov     al,2                    ;load al with step d0
 step386_ret:

               pop       es
               ret
.8086
 step386       endp

;------------------------------------------------------------------------------
; This checks to see if the BIOS reports a Weitek math coprocessor. This should
; only be called if a 386 or 486 is found.
; NOTE!! This may not work with all computers!!
; From InfoPlus

fnoWeitek       equ     0
fWeitek         equ     1
fWeitek_real    equ     81h

weitek  proc    near
.386
        xor     eax,eax                 ;zero everything
        int     11h                     ;do equipment check
        test    eax,01000000h           ;check bit 24, set if Weitek present
        je      no_weitek
        mov     bx,fWeitek
        test    eax,0800000h            ;check bit 23, set if Weitek can be
        je      weitek_done             ; addressed in real mode
        mov     bx,fWeitek_real
        jmp     short weitek_done
no_weitek:
        mov     bx,fnoWeitek
weitek_done:
        ret
.8086
weitek  endp


;--------------------------------------------------------------------------
; Tries to determine if 101/102 keyboard present and supported.
; Looks at Bit 4 of 0040:0096.
; From Hax #33 PC Techniques 10/90
; BIOS support part from INFOPLUS (added 4.60)
; On entry: nothing
; Modifies: AX
; Returns: Al=0 if no extended keyboard, Al=1 if extended keyboard
;          Ah=0 if no BIOS support for extended kybd, Ah=1 if support
;---------------------------------------------------------------------------

 check_keyboard	proc near

     push      bx
     push      es
	mov		ax,040h
	mov		es,ax
	mov		bx,96h
	mov		al,byte ptr es:[bx]
     pop       es
	test		al,10h
	jnz		_ck_yes
	xor		ax,ax
;     ret
     jmp       short _ck_bios
 _ck_yes:
	mov		ax,1
 _ck_bios:
     mov       bx,ax
     mov       ah,02h
     int       16h       ;return shift flag status
     mov       cl,al
     xor       al,0FFh
     mov       ah,012h
     int       16h       ;return extended shift flags
     cmp       al,cl
     jne       _ck_bios_no
     mov       bh,1
     jmp       short _ck_ret
 _ck_bios_no:
     xor       bh,bh
 _ck_ret:
     mov       ax,bx
     pop       bx
     ret

 check_keyboard	endp

;Thanks to
;{ WhichCPU - Yet another CPU-Identifier
;  1992 by Michael Holin
;                                           this is Public Domain
;
;  Thanks to
;
;  Michael Tischer and many others,         for the base-cpu-id-routine
;  Ted Forgeron & Pat Shea,                 for the base-npu-id-routine
;  Hans-Ullrich Siehl,                      for his RapidCAD
;  c't Magazin,                             for many interesting articles
;  Andrew Rossmann,                         for his SYSID+
;                                           (maybe soon comes a version, which
;                                            detects 386sx and RapidCAD  ;-)
;}
;for DXorSX and supporting code

.386
DXorSX  proc far                    ;Mem-Access via DWord-Read
        assume  cs:_TEXT, es:nothing
;        push bp
;        mov bp,sp
        push ds
        push es
        push si
        push di
        push cx
        pushf
        mov ax,0040h
        mov ds,ax
        mov bx,00F0h
        xor ecx,ecx
        xor esi,esi
        mov cx,word ptr ds:bx           ;Anzahl
        mov si,word ptr ds:bx + 2       ;Offset
        mov ax,word ptr ds:bx + 4       ;Segment
        mov ds,ax
        cld
        cli
        mov al,34h
        out 43h,al
        jmp $+2
        mov al,0
        out 40h,al
        jmp $+2
        out 40h,al
        rep
        lodsd

        mov al,0
        out 43h,al
        sti
        jmp $+2
        in al,40h
        xchg ah,al
        jmp $+2
        in al,40h
        xchg ah,al
        neg ax             ;return time for doing the loop
        popf
        pop cx
        pop di
        pop si
        pop es
        pop ds
;        pop ds
;        mov sp,bp
;        pop bp
        ret
DXorSX endp

.8086

 cpu      db   "80x86",0
 ndp      db   "80x87",0
 cpu_nec  db   "NEC V20/V30",0
 cpu_486sx db   "80486SX",0
 cpu_old  db   "8086/88",0
 cpu_nold db   "8087",0
 cpu_odd  db   "80188/86",0
 cpu_step db   "Step:",0
 cpu_slevels db "B0-","B1 ","D0+"
 weitekd   db   "Weitek",0
 sx_suffix db   "SX",0
 dx_suffix db   "DX",0


 disp_cpu      proc near

     mov       ax,cpucode
     cmp       ah,1
     jne       _dcpu_10
     mov       si,offset cpu_old
     jmp       short _dc_ndp
 _dcpu_10:
     cmp       ah,5
     jne       _dcpu_20
     mov       si,offset cpu_odd
     jmp       short _dc_ndp
 _dcpu_20:
     cmp       ah,6
     jne       _dcpu_25
     mov       si,offset cpu_nec
     jmp       short _dc_ndp
 _dcpu_25:
     cmp       ax,0400h
     jne       _dcpu_30
     mov       si,offset cpu_486sx
     jmp       short _dc_ndp
 _dcpu_30:
     or        ah,30h          ;convert to ASCII
     mov       byte ptr [cpu+2],ah
     mov       si,offset cpu
 _dc_ndp:
     mov       di,offset CPU_LOC
     call      copy_null

     cmp       al,0
     jne       _dcpu_50
     mov       si,offset none
     jmp       short _dc_100
 _dcpu_50:
     cmp       al,1
     jne       _dcpu_60
     mov       si,offset cpu_nold
     jmp       short _dc_100
 _dcpu_60:
     or        al,30h
     mov       byte ptr [ndp+2],al
     mov       si,offset ndp

 _dc_100:
     mov       di,offset NDP_LOC
     call      copy_null

     cmp       ah,'3'
     jne       _dc_200             ;not 386, no step info or sx/dx detection

     ;display SX or DX suffix for 386
     mov       word ptr [040:0F0h],02000h
     mov       word ptr [040:0F2h],1
     mov       word ptr [040:0F4h],0
     call      dxorsx
     call      dxorsx
     push      ax
     mov       word ptr[040:0F2h],2
     call      dxorsx
     pop       bx
     add       ax,90
     mov       di,offset CPU_LOC + 6
     cmp       ax,bx
     jb        _dc_120
     mov       si,offset dx_suffix
     call      copy_null
     jmp       short _dc_150            ;do step check
 _dc_120:
     mov       si,offset sx_suffix
     call      copy_null
     jmp       short _dc_200

 _dc_150:
     cmp       mswin_version,0     ;can't do step checks under Windows
     jne       _dc_200

     mov       si,offset cpu_step
     mov       di,offset STEP_LOC+1
     call      copy_null

;     inc       di
     mov       cl,stepping386
     mov       ch,3                ;length of elements
     mov       si,offset cpu_slevels
     call      put_indexed


 _dc_200:
     mov       si,offset FREQ_LOC
     mov       ax,cpuclock
     and       ax,ax
     jz        _dc_300             ;estimated frequency not stored

     mov       bl,10               ;decimal radix
     call      word_to_asce        ;write out to screen
     mov       al,byte ptr [si-1]
     mov       byte ptr [si],al
     mov       byte ptr [si-1],'.' ;move over last digit and insert decimal
                                   ;since clock x 10 is what's stored
 _dc_300:
     test      siFlags,WEITEK_MASK ;write "Weitek" if one is present
     jz        _dc_800
     mov       si,offset Weitekd
     mov       di,offset WEITEK_LOC
     call      copy_null

 _dc_800:
 _dc_ret:
     ret
 disp_cpu      endp


 busses   db   "  XT"," ISA","EISA"," MCA"
 comports db   "  8250"," 16450"," 16550","16550A","   ???"
 memmans  db   "QEMM   ","386^MAX","EMM386 "
 hmause   db   "none","used","free","DOS5"


 disp_hardware  proc near

     mov       al,siFlags
     mov       dx,ax
     and       al,SBLASTER_MASK
     mov       di,offset SBLAST_LOC
     call      put_yn
     cmp       al,'N'
     je        _dhw_20

     mov       si,offset SBPORT_LOC
     mov       ax,sbport
     mov       bl,16
     call      word_to_asce
     mov       byte ptr [si],'h'
 _dhw_20:
     mov       ax,dx
     and       al,ADLIB_MASK
     mov       di,offset ADLIB_LOC
     call      put_yn

;     mov       ax,dx
;     and       al,MPU_MASK
;     mov       di,offset MPU_LOC
;     call      put_yn

;     mov       ax,dx
;     and       al,GAME_MASK
;     mov       di,offset GAME_LOC
;     call      put_yn
;     mov       al,'/'
;     stosb
;     mov       ax,dx
;     and       al,GAMEB_MASK
;     call      put_yn

     mov       si,offset busses
     mov       cl,BusType
     dec       cl                  ;Bus Type stored off base of 1
     mov       ch,4
     mov       di,offset BUS_LOC
     call      put_indexed

     mov       di,offset MOUSE_LOC
     mov       al,is_mouse
     call      put_yn
;     and       al,al
;     jnz       _dhw_25
;     mov       byte ptr [di+3],'N'
;     jmp       short _dhw_30
; _dhw_25:
;     mov       ax,mouse_version
;     call      put_version
; _dhw_30:

     mov       di,offset KYBD_BIOS_LOC
     mov       al,ext_kybd_bios
     call      put_yn

     mov       di,offset KYBD_PRES_LOC
     mov       al,ext_kybd_flag
     call      put_yn

     mov       di,offset printport_addr      ;show printer ports
     mov       cx,3
     mov       bl,16
     mov       dx,11
     mov       si,offset PRINTER_LOC
 _dhw_100:
     mov       ax,[di]
     and       ax,ax
     jz        _dhw_120
     push      si
     call      put_spaces
     add       si,8
     call      word_to_asce
     mov       byte ptr[si],'h'
     pop       si
 _dhw_120:
     inc       di
     inc       di
     add       si,SI_LENGTH
     loop      _dhw_100
                                             ;show comm ports
     mov       di,offset commport_addr
     mov       cx,4
     mov       bl,16
     mov       dx,11
     mov       si,offset COMM_LOC
 _dhw_200:
     mov       ax,[di]
     and       ax,ax
     jz        _dhw_250
     push      si
     call      put_spaces
     inc       si
     call      word_to_asce
     mov       byte ptr [si],'h'

     push      di cx
     mov       di,offset commport_type
     add       di,4
     sub       di,cx
     mov       cl,byte ptr [di]              ;point to index for types
     mov       ch,6
     mov       di,si
     inc       di
     inc       di
     mov       si,offset comports
     call      put_indexed
     pop       cx di
     pop       si
 _dhw_250:
     inc       di
     inc       di
     add       si,SI_LENGTH
     loop      _dhw_200
     ret
 disp_hardware endp



 disp_DOS      proc near
     mov       di,offset dos_loc
     mov       ax,dos_version
     xchg      ah,al          ;Microsoft stores MAJOR in low
     call      put_version

     call      store_sys_dt   ;store system date/time
     call      parse_date_time ;store in display format
     mov       si,offset DATE_LOC
     mov       dx,s_year           ;setup registers
	mov		ax,s_md             ;with stored info
	call		asc_date            ;call date writing function
     mov       si,offset TIME_LOC
     mov       dx,s_hm
	mov		bx,1
	call		asc_time

     mov       di,offset BREAK_LOC
     mov       al,break_stat
     call      put_yn

     mov       di,offset VERIFY_LOC
     mov       al,verify_stat
     call      put_yn

     mov       cl,memm_present
     and       cl,cl
     jz        _ddos_100                ;no tested memory manager present
     dec       cl                       ;to get to zero base
     mov       ch,7
     mov       si,offset memmans
     mov       di,offset MEMMAN_LOC
     call      put_indexed
     mov       di,offset MEMMAN_LOC
     mov       byte ptr [di+11],'Y'

     cmp       cl,0                     ;is it QEMM (remember reduced by 1)
     jne       _ddos_100
     add       di,8
     mov       ax,qemm_version          ;if yes, write version number
     call      put_version

 _ddos_100:

     mov       di,offset MSWIN_LOC
     mov       ax,mswin_version
     and       ax,ax
     jnz       _ddos_150
     inc       di
     call      put_yn                   ;puts N
     jmp       short _ddos_160
 _ddos_150:
     xchg      ah,al                    ;'cause MS does things backwards
     call      put_version
 _ddos_160:
     mov       di,offset DVIEW_LOC
     mov       ax,dv_version
     and       ax,ax
     jnz       _ddos_170
     inc       di
     call      put_yn
     jmp       short _ddos_180
 _ddos_170:
     call      put_version
 _ddos_180:

     mov       di,offset DOS_FILES_LOC
     xor       dx,dx
     mov       ax,dos_files
     mov       cx,3
     call      format

     mov       di,offset DOS_BUFF_LOC
     xor       dx,dx
     mov       ax,dos_buff
     mov       cx,3
     call      format

     mov       di,offset DOS_ENV_LOC
     xor       dx,dx
     mov       ax,dos_env_used
     mov       cx,4
     call      format
     mov       di,offset DOS_ENV_LOC + 4
     mov       byte ptr [di],'/'
     xor       dx,dx
     mov       ax,dos_env
     mov       cx,4
     mov       di,offset DOS_ENV_LOC + 5
     call      format


    ret
 disp_DOS      endp

 dfdir_options db "Conv"," XMS"," EMS","Disk","Conv"

 disp_memory   proc near


     mov       di,offset XMSV_LOC
     test      highmem_flag,01b
     jz        _dmem_40
     mov       ax,xmsver
     call      put_version

     mov       di,offset XMMV_LOC
     mov       ax,xmmver
     call      put_version

     xor       dx,dx
     mov       di,offset XMSF_LOC
     mov       cx,6
     mov       ax,xmslargest
     call      cformat

     jmp       short _dmem_50
 _dmem_40:
     mov       si,offset none
     call      copy_null
 _dmem_50:
     mov       si,offset hmause
     mov       di,offset HMA_LOC
     mov       cl,hmaavail
     mov       ch,4
     call      put_indexed

     mov       di,offset HMAF_LOC
     mov       ax,hmafree
     xor       dx,dx
     mov       cx,2
     call      format

     cmp       ems_active,0
     je        _dfmem_100               ;no active ems

     mov       si,offset EMSVER_LOC
     xor       ax,ax
     mov       al,emsversion
     mov       bl,16
               ;they store the number as a single hex number.
               ;e.g. version 4.0 is stored as 40h.  Isn't that special?
     call      word_to_asce
     mov       al,byte ptr [si-1]
     mov       byte ptr [si-1],'.'
     mov       byte ptr [si],al

 _dfmem_100:

     xor       dx,dx
     mov       ax,tpages           ;total pages EMS memory (pages)
     mov       cl,4
     shl       ax,cl               ;convert to KB
     mov       di,offset EMS_LOC-1
     mov       cx,6
     call      cformat

     xor       dx,dx
     mov       ax,apages           ;avail pages EMS memory (pages)
     mov       cl,4
     shl       ax,cl               ;convert to KB
     mov       di,offset EMS_LOC-1
     add       di,7
     mov       cx,6
     call      cformat

     xor       dx,dx
     mov       ax,avail_mem        ;available memory in paragraphs
     mov       cl,6
     shr       ax,cl               ;convert to KB
     mov       di,offset CONV_LOC
     add       di,7
     mov       cx,5
     call      cformat

     xor       dx,dx
     mov       ax,basemem        ;base memory in KB
     mov       di,offset CONV_LOC
     mov       cx,5
     call      cformat

     xor       dx,dx
     mov       ax,extmem
     mov       di,offset EXT_LOC-1
     mov       cx,6
     call      cformat


     mov       di,offset DFPROG_LOC
     mov       ax,dfprog_mem
     mov       cx,3
     xor       dx,dx
     call      cformat

     mov       di,offset DFDIR_LOC
     push      di
     mov       ax,dfdir_mem
     mov       cx,3
     xor       dx,dx
     call      cformat
     pop       di
     mov       cl,mem_use
     add       di,6
     mov       si,offset dfdir_options
     mov       ch,4
     call      put_indexed
 _dfmem_150:
     mov       di,offset DFSHELL_LOC
     mov       ax,dfshell_mem
     mov       cx,3
     xor       dx,dx
     call      cformat



     ret
 disp_memory   endp

;-----------------------------------------------------------------------------
; Searches for BIOS date and copyright info and puts in screen
;-----------------------------------------------------------------------------
 copr_array    db   "COPR.",0,"Copr.",0,"(C)",0,"(c)",0,"Copyright",0

 bios_version       proc near

     push      ds
     mov       ax,0F000h      ;point to BIOS ROM
     mov       ds,ax
     mov       si,0FFF5h      ;"normal" BIOS date offset
     mov       di,offset BIOS_LOC
     mov       cx,8
     call      movasciib

     pop       ds
     mov       cx,5           ;5 elements in copyright array
     mov       dx,offset copr_array     ;point to first element
 _bv_70:
     call      str_search
     jnc       _bv_100        ;search succeeded
     loop      _bv_70
     jmp       short _bv_200
 _bv_100:
     push      ds
     mov       ax,0F000h      ;point to BIOS ROM
     mov       ds,ax
     mov       si,dx
     mov       di,offset BIOS_LOC + 12
     mov       cx,12
     call      movasciib
     mov       cx,24
     mov       di,offset BIOS2_LOC
     call      movasciib
     pop       ds

 _bv_200:
     ret


 bios_version       endp


;---------------------------------------------------------;
; This subroutine searches through ROM for the copyright. ;
; On entry: DX points to text to match (ASCIIZ)
; On return: carry failed, dx points to next string
;            no carry succeeded, dx points to match
;---------------------------------------------------------;

 STR_SEARCH        proc near
               MOV    BX,0                   ;Start at offset zero.
               push   cx
               push   bp
               push   ds
               mov    si,dx
               call   str_len
               mov    ax,0F000h              ;point to ROM BIOS data segment
               mov    ds,ax
               mov    bp,cx
 SS_NEXT:      mov    cx,bp
               MOV    SI,BX                  ;Retrieve starting offset.
               MOV    DI,DX                  ;DX has offset of text to match.
               REPZ   CMPSB
               JZ     SS_MATCH               ;If zero, we got a match.
               INC    BX                     ;Else, point to next offset.
               JNZ    SS_NEXT                ;If back to zero, we are done.
               add    dx,bp
               inc    dx
               pop    ds
               pop    bp
               pop    cx
               STC                           ;Indicate no match with carry.
               RET
 SS_MATCH:
               mov    dx,si
               pop    ds
               pop    bp
               pop    cx
               CLC                           ;Indicate match with no carry.
               RET

 STR_Search        endp



     video_array    db "No display         "
                    db "MDA + 5151 (mono)  "
                    db "CGA + color        "
                    db "                   "
                    db "EGA + color        "
                    db "EGA + mono         "
                    db "PGA                "
                    db "VGA + analog mono  "
                    db "VGA + analog color "
                    db "                   "
                    db "MCGA + digital clr "
                    db "MCGA + analog mono "
                    db "MCGA + analog color"
     VELEMENT_LENGTH     EQU 19
     vid_unknown    db "Unknown",0

     video_array2   db "MDA "," CGA"," EGA","MCGA"," VGA"
     VELEMENT2_LENGTH EQU 4

;----------------------------------------------------------------------------
; Puts the appropriate array elements from the video bios call to the screen
;----------------------------------------------------------------------------
 disp_video         proc near
     mov       bx,bios_video
     mov       di,offset VIDEOA_LOC
     cmp       bl,0Ch
     ja        _dvideo_50
     mov       ch,VELEMENT_LENGTH
     mov       si,offset video_array
     mov       cl,bl
     call      put_indexed
     jmp       short _dvideo_100
 _dvideo_50:
;     mov       si,offset vid_unknown
;     call      copy_null
     call      get_adapter
     mov       cl,al
     mov       si,offset video_array2
     mov       ch,VELEMENT2_LENGTH
     call      put_indexed
     jmp       short _dvideo_ret
 _dvideo_100:
     mov       di,offset VIDEOB_LOC
     cmp       bh,0Ch
     ja        _dvideo_150
     mov       si,offset video_array
     mov       cl,bh
     call      put_indexed
     jmp       short _dvideo_ret
 _dvideo_150:
     mov       si,offset vid_unknown
     call      copy_null
 _dvideo_ret:
     ret
 disp_video         endp

 floppy_types  db   "none     "
               db   "360KB  5"
               db   "1.2MB  5"
               db   "720KB  3"
               db   "1.44MB 3"
               db   "2.88MB 3"
               db   "unknown  "

 disp_global_drive  proc near

     call      test_for_cmos
     jnc       _dgd_20
     mov       di,offset NOTCMOS_LOC
     mov       si,offset notcmos
     call      copy_null
     mov       si,offset HD1_LOC-11      ;blank out HD types
     mov       dx,20
     call      put_spaces

     mov       di,offset FLOP1_LOC
     mov       al,num_floppies
     call      put_yn
     mov       di,offset FLOP2_LOC
     dec       al
     call      put_yn
     jmp       short _dgd_50
 _dgd_20:
     mov       di,offset FLOP1_LOC
     mov       al,floppies
     and       al,11110000b
     shr       al,4
     cmp       al,5
     jbe       _dgd_30
     mov       al,6                ;unknown type
 _dgd_30:
     mov       cl,al
     mov       ch,9
     mov       si,offset floppy_types
     call      put_indexed
     mov       cl,floppies
     and       cl,1111b
     cmp       cl,5
     jbe       _dgd_40
     mov       cl,6
 _dgd_40:
     mov       ch,9
     mov       di,offset FLOP2_LOC
     mov       si,offset floppy_types
     call      put_indexed

     mov       di,offset HD1_LOC
     xor       dx,dx
     xor       ax,ax
     mov       cx,3
     mov       al,drive0
     call      format

     mov       di,offset HD2_LOC
     xor       dx,dx
     xor       ax,ax
     mov       cx,3
     mov       al,drive1
     call      format

 _dgd_50:

     mov       di,offset LOGDRV_LOC
     mov       si,offset logical_drives
     mov       cx,logical_count
     rep       movsb



     ret
 disp_global_drive  endp
;-----------------------------------------------------------------------------
; Puts null terminated string pointed to by si onto screen with current
; attribute
;-----------------------------------------------------------------------------
 PUT_LEN_CHARS proc near
     call str_len
     call put_cx_chars
     ret
 PUT_LEN_CHARS endp

 COPY_NULL     proc near
     call str_len
     rep  movsb
     ret
 COPY_NULL     endp
;-----------------------------------------------------------------------------
; Puts cl entry (0 base) of array with element length ch to location pointed
; to by di.  On entry, si points to first element of array
;-----------------------------------------------------------------------------
 PUT_INDEXED   proc near
     push      ax cx bx
     mov       ax,cx
     xor       cx,cx
     xor       bx,bx
     mov       bl,ah     ;bx now contains element length
     mov       cl,al
     and       cl,cl
     jz        _pidx_200 ;skip loop if zero
 _pidx_100:
     add       si,bx
     loop      _pidx_100
 _pidx_200:
     mov       cl,ah
     rep       movsb
     pop       bx cx ax
     ret
 PUT_INDEXED   endp

;-----------------------------------------------------------------------------
; Puts a Y or N to di as a function of the al register value
;-----------------------------------------------------------------------------
 PUT_YN        proc near
     cmp       al,0
     jne       _pyn_100
     mov       al,'N'
     jmp       short _pyn_200
 _pyn_100:
     mov       al,'Y'
 _pyn_200:
     stosb
     ret
 PUT_YN        endp

;------------------------------------------------------------------------------
; PUT_VERSION
; Takes a software version number of the form ah=major version and al=minor
; version and displays to di.  Inserts leading 0 for minor version as
; required.
;------------------------------------------------------------------------------
 PUT_VERSION   proc near
     mov       bx,ax
     xor       ax,ax
     mov       al,bh
     mov       si,di
     push      bx
     mov       bx,10
     call      word_to_asce
     pop       bx
     mov       byte ptr [si],'.'
     inc       si
     mov       word ptr [si],'00'
     xor       ax,ax
     mov       al,bl
     cmp       al,10
     jae       _pv_10
     inc       si
 _pv_10:
     push      bx
     mov       bx,10
     call      word_to_asce
     pop       bx
     ret
 PUT_VERSION  endp

;-----------------------------------------------------------------------------
; on input, si points to spaces to blank
; dx has number of spaces
;------------------------------------------------------------------------------
 put_spaces    proc near
     push      di si cx ax
     mov       cx,dx
     mov       di,si
     mov       al,20h
     rep       stosb
     pop       ax cx si di
     ret
 put_spaces    endp


;-----------------------------------------------------------------------------
; Works just like movsb but ascii chars only
;-----------------------------------------------------------------------------
 movasciib     proc near

 _mascb_100:
     lodsb                    ;don't want to print any garbage
     cmp       al,32          ;(date's not always there) so rep movsb
                             ;isn't a great idea
     jb        _mascb_200
     cmp       al,'z'
     ja        _mascb_200
     stosb
     loop      _mascb_100
 _mascb_200:
     ret

 movasciib     endp