; Adapted From
;
;                            CPUTYPE
;
;    Display information about CPU.  Demonstrates use of a
;    number of individual detection subroutines.  Command line
;    options include:
;       -       Prevent test which require the use of protected
;               mode instructions, which might not be handled
;               correctly V86 mode.
;       +       Show raw timing values for cache and CPU speed.
;
;    Due to a bug in Windows, only the CPU ID tests can be run.
;    If Windows is running, all other tests are supressed.
;    The problem occurs in the execution of the RDMSR instruction
;    on a non-IBM chip (which is not supported on other chips).
;    Even though the bad opcode interrupt is properly intercepted
;    by this program, Windows kills the program anyway!
;
;    (c) Copyright 1994  Frank van Gilluwe  All Rights Reserved.


; Include file for Undocumented PC programs and Undocumented
;  instruction macros.

;  UNDOCUMENTED INSTRUCTION MACROS 
;
; See book for details on each instruction. A few instructions
;  are documented like CPUID, but most assemblers do not allow
;  these instructions, and are included here for convenience.
;
; WARNING:  Many of these instructions are limited to a
;  specific processor and/or vendor.  Using a macro/instruction
;  on the wrong processor will likely hang or cause a bad
;  opcode fault. Be real sure the instruction is allowed
;  for the active processor!

POPCS   macro                      ; Pop word from stack into CS
        db      0Fh                ;   (8088/8086 only)
        endm


; LOADALL Note: not all assemblers allow conditions based
;  on the current cpu type.  Both MASM 6.0 and later, and
;  TASM 3.0 and later support the cpu type condition.  If
;  yours does not support @Cpu, you may need to build two
;  different named macros, like LOADALL2 and LOADALL3 for
;  each CPU case.

LOADALL macro                      ; Load all registers into CPU
    IF @Cpu AND 8                  ;  test for 386 or higher
        db      0Fh, 07h           ; loadall for 386+
    ELSE
        db      0Fh, 05h           ; loadall for 286
    ENDIF
        endm


RDTSC   macro                      ; Read Time Stamp Counter
        db      0Fh, 31h           ;  TSC into edx:eax
        endm


RSM     macro                      ; Resume from system management
        db      0Fh, 0AAh          ;   mode
        endm


SETALC  macro                      ; Set AL with carry
        db      0D6h
        endm


ICEBP   macro                      ; In-Circuit-Emulator Breakpoint
        db      0F1h               ;   single byte interrupt 1
        endm


;  COMMON MACROS 

OUTMSG  macro   location           ; output text to screen
        mov     dx, offset location
        mov     ah, 9
        int     21h
        endm


MSGNO   macro   loc_for_no         ; insert "No " at loc_for_no
        mov     loc_for_no, "N"
        mov     loc_for_no+1, "o"
        mov     loc_for_no+2, " "
        endm


MSGYES  macro   loc_for_yes        ; insert "Yes " at loc_for_yes
        mov     loc_for_yes, "Y"
        mov     loc_for_yes+1, "e"
        mov     loc_for_yes+2, "s"
        endm

IODELAY macro
        jmp     short $+2          ; short delay between I/O
        jmp     short $+2          ;  to same device
        endm

;---------------------------------------------------------------------------
; Start CPU ID rotuiens
;---------------------------------------------------------------------------

cputype proc    near
     assume cs:_text, ds:_text, es:_text

;start:
        xor     bx,bx
        TESTFLAG siset,01b
        jz      no_options         ;skip V86 test bit NOT set
        mov     bl, '-'
no_options:
        mov     [cpu_cmd], bl     ; save option

; get cpu type and cpumode, and save results


        call    cpuvalue           ; get CPU type 0 to 5 in al
        mov     [cpu_val], al      ; save
        mov     [cpu_info], ah     ; save flags
        call    cpumode            ; get current mode: V86, real,
                                   ;   protected into ax
        mov     [cpu_prot], al     ; save type of protection
        mov     [cpu_priv], ah     ; save privledge level

; --- determine CPU & FPU type and output message text

begin_cpu:
        mov     al, [cpu_val]      ; get previous saved cpu value
        mov     ah, [cpu_info]     ; get previous saved flags
        mov     di, offset fpun
        test    ah, 2              ; V20/V30 ?
        jnz     cpui_getfpu        ; jump if so
        mov     cx, offset cpunend - offset cpuname
        mul     cl                 ; ax=al*cl
        mov     si, offset cpuname
        add     si, ax
        mov     di, offset cpun
        cld
cpui_loop1:
        movsb                      ; xfer CPU name
        cmp     byte ptr [si], ' ' ; end of CPU name ?
        je      cpui_getfpu        ; exit if so
        loop    cpui_loop1

cpui_getfpu:
        call    fputype            ; detect FPU, 0 to 5 in al
        mov     [fpu_val], al      ; save

        cmp     [cpu_cmd], "-"    ; option to prevent test ?
        je      cpui_skp0          ; jump to skip test
        call    fpuloc             ; see if FPU inside CPU
        mov     [fpu_info], al     ; save

cpui_skp0:
        mov     al, [fpu_val]
        cmp     al, 4              ; Pentium with FPU ?
        ja      cpui_skp3          ; jump if so
        jb      cpui_skp1          ; jump if not 80486 with math


        cmp     [cpu_cmd], "-"    ; option to prevent test ?
        jne     cpui_skp0a         ; jump if fpu_info valid
        mov     word ptr [di-2], '  '  ; blank out "SX"
        sub     di, 2
        mov     si, offset fpuin2
        jmp     cpui_skp4          ; just indicate system has FPU

cpui_skp0a:
        test    [fpu_info], 1      ; inside chip ?
        jnz     cpui_skp2
        mov     al, 3              ; must be 80486SX with 80387
cpui_skp1:
        mov     cx, offset fpunend - offset fpuname
        mul     cl                 ; ax = al * cl
        add     ax, offset fpuname
        mov     si, ax
        cld
        rep     movsb              ; xfer FPU name
        jmp     cpui_skp5

cpui_skp2:
        mov     byte ptr [di-2], 'D' ; it's a 80486DX
cpui_skp3:
        mov     si, offset fpuin
cpui_skp4:
        mov     cx, offset fpuine - offset fpuin
        cld
        rep     movsb              ; xfer "FPU inside" msg

cpui_skp5:
;        OUTMSG  cpumsg             ; display result
        mov    di,offset cpu_loc
        mov    si,offset cpun
        call   copy_null

; Now check if Windows is running, and exit if so.  See notes
;  at top about Windows problem in attempting to find the CPU
;  vendor, which is required for other tests.

        mov     ax, 1600h
        int     2Fh                ; Windows check
        cmp     al, 0
        je      cpui_find_vendor   ; not running enhanced
        cmp     al, 80h
        je      cpui_find_vendor   ; not running enhanced
;don't run other tests
        mov     si,offset windows1
        mov     di,offset cache_loc - 11
        mov     cx,12
        mov     al,' '
        rep     stosb
        mov     di,offset STEP_LOC + 3
        call    copy_null
        mov     si,offset windows2
        mov     di,offset CACHE_LOC - 5
        call    copy_null

        jmp     cpui_exit

; --- determine CPU Vendor

cpui_find_vendor:
        call    cpuvendor          ; get vendor number in al
        mov     [cpu_mfg], al      ; save for later

; --- determine CPU Speed

        call    cpuspeed           ; determine internal CPU speed

cpui_skp6:
        mov     di, offset cpusval
        xor     bl, bl
        call    decw               ; convert to decimal

; --- now adjust [cpu_mfg] if 386 at 40 Mhz (AMD)
;                          if 386 under 25 MHz (Intel)

        cmp     [cpu_val], 3       ; 386 ?
        jne     cpui_skp8          ; jump if not
        cmp     [cpu_mfg], 3       ; Now set to Intel or AMD?
        jne     cpui_skp8          ; jump if not
        mov     bl, 4              ; assume Intel
        cmp     ax, 21             ; 16 or 20 Mhz ?
        jbe     cpui_skp7          ; jump if so (very likely Intel)
        cmp     ax, 39             ; below 39 Mhz ?
        jb      cpui_skp8          ; if so, Intel or AMD
        cmp     ax, 41             ; above 41 Mhz ?
        ja      cpui_skp8          ; unclear what it is
        mov     bl, 5
cpui_skp7:
        mov     [cpu_mfg], bl      ; set to AMD or Intel, not both

; output vendors and speed strings

cpui_skp8:
;        OUTMSG  cpuvend
;        mov     di,offset vendor_loc
;        mov     si,offset cpuvend
;        call    copy_null

        mov     al, [cpu_mfg]
        cmp     al, 1              ; from CPUID string ?
        jne     cpui_skp9          ; jump if not
        mov     cx, 12
        mov     dx, offset idstring
        jmp     cpui_skp10

cpui_skp9:
        mov     cx, offset vendend - offset vendors
        mul     cl                 ; ax = index into text strings
        add     ax, offset vendors
        mov     dx, ax
cpui_skp10:
;        mov     ah, 9
;        int     21h                ; output vendor name
;        OUTMSG  crlf               ; linefeed
        mov     di,offset vendor_loc
        mov     si,dx
        call    copy_null

;        OUTMSG  cpuspd             ; display processor speed
         mov   di,offset MHZ_LOC
         mov   si,offset cpuspd
         call  copy_null


; --- find pre-fetch q size

        cmp     [cpu_val], 5       ; Pentium ?
        je      cpui_get_step      ; skip pre-fetch queue test if so


; --- try for CPU version id (for 386 and above)

cpui_get_step:
        call    cpustep            ; check for CPU step & output


; --- display CPU mode (for 286 and above)

        mov     al, [cpu_prot]     ; get protected/real status
        mov     ah, [cpu_priv]     ; get privilege level
        call    cpum_display       ; display results


; --- check internal cache information

        cmp     [cpu_cmd], "-"    ; prevent cache & undoc tests?
        jne     cpui_skp10a
        jmp     cpui_exit          ; jump if so

cpui_skp10a:
        call    cpu_cache          ; find internal CPU cache type
        call    cpu_cache_display  ; display results
        cmp     al, 2              ; cache present and enabled ?
        jb      cpui_exit          ; no cache or cache disabled

; --- find data cache size by timing reads for blocks of memory

        cmp     [cpu_val], 5       ; Pentium ?
        jb      cpui_skp11
;        OUTMSG  cSizeData          ; display "data cache size"
        mov    si,offset dcachesize_msg
        mov    di,offset SCACHE_LOC
        call   copy_null
        jmp     cpui_skp12

cpui_skp11:
        mov    si,offset intcachesize_msg
        mov    di,offset SCACHE_LOC
        call   copy_null
cpui_skp12:
        call    cache_d_size       ; get size of data cache in ax
        cmp     ax, 0              ; size zero ?
        je      cpui_skp13
        cmp     ax, 32             ; 32K or more ?
        jae     cpui_skp13
        mov     di, offset cSize1
        xor     bl, bl
        call    decw
        inc     di
        mov     word ptr [di], 'BK' ; insert KB
;        OUTMSG  cSize1
        mov    si,offset cSize1
        mov    di,offset SCACHE_LOC+17
        call   copy_null
        jmp     cpui_skp14

cpui_skp13:
        mov si,offset cSizeNone          ; "external cache masks size"
        mov    di,offset SCACHE_LOC
        call   copy_null
cpui_skp14:

; instruction cache on pentium message (not measured)

        cmp     [cpu_val], 5       ; Pentium ?
        jb      cpui_exit
;        OUTMSG  cSizeCode

; All done

cpui_exit:
         ret
;        mov     ah,4Ch
;        int     21h                ; exit with al return code
;      int 20h
;EXITCODE
cputype endp



;
; all of the subroutines are located in CPUTYPE2


; SUBROUTINES FOR CPUTYPE

;
;    CPU IDENTIFICATION SUBROUTINE
;       Identify the CPU type, from 8088 to the Pentium.  Works
;       even if the 386 or later CPU is in V86 mode.  Note that
;       interrupts are enabled at exit, even if they were
;       disabled on entry.  If it is necessary to run this
;       routine with interrupts disabled, just remove all CLI
;       and STI instructions, so long as interrupts are
;       always disabled before running.
;
;       Called with:    nothing
;
;       Returns:        al = CPU type
;                             0 if 8088/8086 or V20/V30
;                             1 if 80186/80188
;                             2 if 80286
;                             3 if 80386
;                             4 if 80486
;                             5 if Pentium
;                       ah =  bit 0 = 0 if CPUID unavailable
;                                     1 if CPUID ok
;                             bit 1 = 0 if not V20/V30
;                                     1 if NEC V20/V30
;
;       Regs used:      ax, bx (all)
;                       eax, ebx (386 or later)
;
;       Subs called:    hook_int6, restore_int6, bad_op_handler

.8086   ; all instructions 8088/8086 unless overridden later

cpuvalue proc    near
        push    cx
        push    dx
        push    ds
        push    es

; 8088/8086 test - Use rotate quirk - All later CPUs mask the CL
;   register with 0Fh, when shifting a byte by cl bits.  This
;   test loads CL with a large value (20h) and shifts the AX
;   register right.  With the 8088, any bits in AX are shifted
;   out, and becomes 0.  On all higher level processors, the
;   CL value of 20h is anded with 0Fh, before the shift.  This
;   means the effective number of shifts is 0, so AX is
;   unaffected.

        mov     cl, 20h            ; load high CL value
        mov     ax, 1              ; load a non-zero value in AX
        shr     ax, cl             ; do the shift
        cmp     ax, 0              ; if zero, then 8088/86
        jne     up186              ; jump if not 8088/86

; V20/V30 test - It is now either a V20/V30 or a 8088.  I'll use
;   another undocumented trick to find out which.  On the 8088,
;   0Fh performs a POP CS.  On the V20/V30, it is the start of
;   a number of multi-byte instructions.  With the byte string
;   0Fh, 14h, C3h the CPU will perform the following:
;               8088/8086               V20/V30
;             pop     cs              set1   bl, cl
;             adc     al, 0C3h

        xor     al, al             ; clear al and carry flag
        push    cs
        db      0Fh, 14h, 0C3h     ; instructions (see above)
        cmp     al, 0C3h           ; if al is C3h then 8088/8086
        jne     upV20
        mov     ax, 0              ; set 8088/8086 flag
        jmp     uP_Exit

upV20:
        pop     ax                 ; correct for lack of pop cs
        mov     ax, 200h           ; set V20/V30 flag
        jmp     uP_Exit

; 80186/80188 test - Check what is pushed onto the stack with a
;   PUSH SP instruction.  The 80186 updates the stack pointer
;   before the value of SP is pushed onto the stack.  With all
;   higher level processors, the current value of SP is pushed
;   onto the stack, and then the stack pointer is updated.

up186:
        mov     bx, sp             ; save the current stack ptr
        push    sp                 ; do test
        pop     ax                 ; get the pushed value
        cmp     ax, bx             ; did SP change ?
        je      up286              ; if not, it's a 286+
        mov     ax, 1              ; set 80186 flag
        jmp     uP_Exit

; 80286 test A - We'll look at the top four bits of the EFLAGS
;   register.  On a 286, these bits are always zero.  Later
;   CPUs allow these bits to be changed.  During this test,
;   We'll disable interrupts to ensure interrupts do not change
;   the flags.

up286:
        cli                        ; disable interrupts
        pushf                      ; save the current flags

        pushf                      ; push flags onto stack
        pop     ax                 ; now pop flags from stack
        or      ax, 0F000h         ; try and set bits 12-15 hi
        push    ax
        popf                       ; set new flags
        pushf
        pop     ax                 ; see if upper bits are 0

        popf                       ; restore flags to original
        sti                        ; enable interrupts
        test    ax, 0F000h         ; were any upper bits 1 ?
        jnz     up386              ; if so, not a 286

; 80286 test B - If the system was in V86 mode, (386 or higher)
;   the POPF instruction causes a protection fault, and the
;   protected mode software must emulate the action of POPF. If
;   the protected mode software screws up, as occurs with a
;   rarely encountered bug in Windows 3.1 enhanced mode, the
;   prior test may look like a 286, but it's really a higher
;   level processor. We'll check if the protected mode bit is
;   on.  If not, it's guaranteed to be a 286.

.286P                              ; allow a 286 instruction
        smsw    ax                 ; get machine status word
        test    ax, 1              ; in protected mode ?
        jz      is286              ; jump if not (must be 286)

; 80286 test C - It's very likely a 386 or greater, but it is
;   not guaranteed yet.  There is a small possibility the system
;   could be in 286 protected mode so we'll do one last test. We
;   will try out a 386 unique instruction, after vectoring the
;   bad-opcode interrupt vector (int 6) to ourselves.

        call    hook_int6          ; do it!
;        mov     [badoff], offset upbad_op  ; where to go if bad
        mov     [badoff], offset proc_upbad_op  ; where to go if bad
.386
        xchg    eax, eax           ; 32 bit nop (bad on 286)

        call    restore_int6       ; restore vector
        jmp     up386              ; only gets here if 386
                                   ;  or greater!

; Interrupt vector 6 (bad opcode) comes here if system is a
;   80286 (assuming the 286 protected mode interrupt 6 handler
;   will execute the bad-opcode interrupt).
proc_upbad_op proc near
upbad_op:
        call    restore_int6
is286:
        mov     ax, 2              ; set 80286 flag
        jmp     uP_Exit

; 80386 test - Bit 18 in EFLAGS is not settable on a 386, but is
;   changeable on the 486 and later CPUs.  Bit 18 is used to
;   flag alignment faults. During this test, we'll disable
;   interrupts to ensure no interrupt will change any flags.

.386                               ; allow 386 instructions

up386:
        cli                        ; disable interrupts
        pushfd                     ; push flags to look at
        pop     eax                ; get eflags
        mov     ebx, eax           ; save for later
        xor     eax, 40000h        ; toggle bit 18
        push    eax
        popfd                      ; load modified eflags to CPU
        pushfd                     ; push eflags to look at
        pop     eax                ; get current eflags
        push    ebx                ; push original onto stack
        popfd                      ; restore original flags
        sti                        ; enable interrupts
        xor     eax, ebx           ; check if bit changed
        jnz     up486              ; changed, so 486 or later
        mov     ax, 3              ; set 80386 flag
        jmp     uP_Exit

; 80486 test - Bit 21 in EFLAGS is not settable on a 486, but is
;   changeable on the Pentium CPU.  If bit 21 is changeable, it
;   indicates the CPU supports the CPUID instruction.  It's
;   amazing it's only taken 10 years to implement the CPUID
;   instruction, which should have been included from the start!
;   During this test, we'll disable interrupts to ensure no
;   interrupt will change any flags.

up486:
        cli                        ; disable interrupts
        pushfd                     ; push flags to look at
        pop     eax                ; get eflags
        mov     ebx, eax           ; save for later
        xor     eax, 200000h       ; toggle bit 21
        push    eax
        popfd                      ; load modified eflags to CPU
        pushfd                     ; push eflags to look at
        pop     eax                ; get current eflags
        push    ebx                ; push original onto stack
        popfd                      ; restore original flags
        sti                        ; enable interrupts
        xor     eax, ebx           ; check if bit changed
        jnz     upPentium          ; changed, it's a Pentium
        mov     ax, 4              ; set 80486 flag
        jmp     uP_Exit

; Pentium - It's possible the CPUID instruction may appear on
;   other CPU chips, so run the CPUID instruction to see what
;   CPU type it indicates.  The CPUID returns a family number
;   0 to 5 for the processor type.  As of this date, only the
;   Pentium supports the CPUID instruction and it is assigned
;   type 5.

upPentium:
        push    ecx                ; CPUID changes eax to edx
        push    edx
        mov     eax, 1             ; get family info function
.586
        CPUID                      ;
.386
        and     eax, 0F00h         ; find family info
        shr     eax, 8             ; move to al
        mov     ah, 1              ; set flag that CPUID ok
        pop     edx
        pop     ecx

up_Exit:
        pop     es
        pop     ds
        pop     dx
        pop     cx
        ret
proc_upbad_op endp
cpuvalue endp
.8086                              ; return to 8086 instructions

;
;    CPU VENDOR IDENTIFICATION SUBROUTINE
;       Determines the CPU manufacturer by checking a number
;       of unique aspects of each vendors chips.
;
;       Called with:    ds:[cpu_val] set with CPU type
;                       ds:[cpu_info] set with additional CPU info
;
;       Returns:        al = Vendor number
;                              0 = Unknown, 8088 to 80286
;                              1 = vendor in CPUID string
;                              2 = NEC V20/V30
;                              3 = Intel or AMD, not IBM or Cyrix
;                              4 = Intel
;                              5 = AMD, only one to make 40Mhz 386
;                              6 = IBM, only 386/486 with RDMSR
;                              7 = Cyrix or TI, no UMOV support
;                       transfers CPUID string to ds:[idstring]
;
;       Regs used:      ax, bx, ecx, dx
;
;       Subs called:    hook_int6, restore_int6, bad_op_handler
;                       hook_intD, restore_intD

.386                               ; allow 386 instructions
cpuvendor  proc   near
        mov     al, 2              ; assume NEC
        test    [cpu_info], 2      ; V20/V30 ? (only NEC makes this)
        jnz     cpuv_exit          ; jump if so
        mov     al, 0              ; assume unknown vendor
        cmp     [cpu_val], 2       ; get cpu number
        jbe     cpuv_exit          ; jump if 286 or less
        test    [cpu_info], 1      ; CPUID valid ?
        jz      cpuv_skp1          ; jump if not

        mov     eax, 0             ; get vendor string function
.586
        CPUID                      ;   for CPUID into ebx, ecx, edx
.386
        mov     di, offset idstring
        mov     eax, ebx
        call    xfer_bytes         ; xfer 4 text bytes from eax
        mov     eax, edx
        call    xfer_bytes         ; xfer 4 text bytes from eax
        mov     eax, ecx
        call    xfer_bytes         ; xfer 4 text bytes from eax
        mov     al, 1
        jmp     cpuv_exit

; We'll check if it's a Cyrix/TI CPU.  Cyrix does not need
;  the UMOV instruction, and is not supported.  (The pentium
;  does not support UMOV either, but the Pentium vendor has
;  already been detected, since it supports CPUID).

cpuv_skp1:
        call    hook_int6          ; hook the bad-opcode int
        call    hook_intD          ; hook general protection fault
;        mov     [badoff], offset cpuv_badop  ; where to go if bad
        mov     [badoff], offset proc_cpuv_badop  ; where to go if bad
        mov     al, 05Ah
        mov     bh, 0A5h
        clc                        ; clear carry
        db      0Fh, 10h, 0F8h     ; umov al, bh
        db      90h, 90h
        jc      cpuv_badop         ; carry should not be set
        cmp     al, bh
        jne     cpuv_badop         ; al should = bh
        jmp     cpuv_try_IBM       ; UMOV ok, likely Intel/AMD/IBM

; UMOV not supported on 386+ CPU, so must be Cyrix

proc_cpuv_badop proc near
cpuv_badop:
        mov     al, 7
        jmp     cpuv_restore

; See if it's an IBM CPU.  Only IBM's chips support the Read
;   Machine Specific Register (RDMSR) with ecx=1000h

cpuv_try_IBM:
;        mov     [badoff], offset cpuv_badop2 ; where to go if bad
        mov     [badoff], offset proc_cpuv_badop2 ; where to go if bad
        mov     ecx, 1000h
.586P
        RDMSR                      ; read machine specific reg
.386
        db      90h, 90h           ; safety NOPs
        mov     al, 6              ; RDMSR works, must be IBM!
        jmp     cpuv_restore

proc_cpuv_badop2 proc near
cpuv_badop2:
        mov     al, 3              ; not IBM, likely Intel or AMD

cpuv_restore:
        call    restore_intD       ; restore the int D handler
        call    restore_int6       ; restore int 6 vector
cpuv_exit:
        ret
proc_cpuv_badop2 endp
proc_cpuv_badop endp
cpuvendor endp
.8086                              ; return to 8086 instructions


;
;    CPU SPEED DETERMINATION SUBROUTINE
;       Determines the CPU speed by accurately measuring
;       a short loop.
;
;       Called with:    ds:[cpu_val] set with CPU type
;                       ds:[cpu_mfg] set with CPU vendor
;
;       Returns:        ax = Speed in Mhz
;                       bx = raw timing value (not too useful)
;
;       Regs used:      ax, bx, cx

cpuspeed proc   near
        push    dx
        push    si
        mov     al, [cpu_val]      ; get cpu number
        xor     ah, ah
        shl     ax, 1
        shl     ax, 1              ; times 4
        mov     si, offset ibmspd  ; table of values
        cmp     [cpu_mfg], 6       ; IBM ?
        je      cpus_skp1          ; jump if so
        mov     si, offset cyrixspd  ; table of values
        cmp     [cpu_mfg], 7         ; Cyrix ?
        mov     si, offset intelspd  ; use Intel table

; now setup timer 2 to time instruction execution

cpus_skp1:
        add     si, ax             ; point SI to value for CPU
        mov     al, 0B0h           ; Timer 2 command, mode 0
        out     43h, al            ; send command
        IODELAY
        mov     al, 0FFh           ; counter value FFFF
        out     42h, al            ; send lsb to counter
        IODELAY
        out     42h, al            ; send msb to counter
        IODELAY

; all interrupts, including NMI, are shut off to prevent any
; interrupts from affecting the timing

        cli                        ; disable interrupts
        mov     al, 80h
        out     70h, al            ; disable NMI
        IODELAY
        in      al, 61h            ; read the current contents
        IODELAY
        or      al, 1              ; set gate bit on
        out     61h, al            ; turn on timer (begins timing)
        xor     dx, dx
        mov     bx, 1
        mov     ax, [si]

; this loop executes a bunch of slow divide instructions

cpus_loop1:
        mov     cx, 10h            ; loop value
cpus_loop2:
        div     bx                 ; ax = dx:ax/1  dx=rem
        div     bx                 ; (lots of cycles per inst)
        div     bx
        div     bx
        div     bx
        div     bx
        div     bx
        div     bx
        div     bx
        div     bx
        div     bx
        div     bx
        div     bx
        div     bx
        loop    cpus_loop2
        dec     ax
        jnz     cpus_loop1         ; loop x times for the CPU

; when the loop completes, the timer is stopped, and interrupts
; are re-enabled.

        in      al, 61h            ; read the current contents
        IODELAY
        and     al, 0FEh           ; set gate bit off
        out     61h, al            ; disable the counter
        xor     al, al
        out     70h, al            ; enable NMI
        sti                        ; enable interrupts

; now the timer contents are read, and the duration of
; instruction execution is determined

        mov     al, 80h            ; latch output command
        out     43h, al            ; send command
        IODELAY
        in      al, 42h            ; get lsb of counter
        IODELAY
        mov     dl, al
        in      al, 42h            ; get msb of counter
        mov     dh, al             ; dx = counter value
        mov     ax, 0FFFFh         ; starting value
        sub     ax, dx             ; ax = duration count
        mov     cx, ax
        mov     bx, ax             ; save for exit
        mov     ax, cx
        cmp     word ptr [si+2], 0 ; no factor adjust ?
        je      cpus_skp2          ; exit with value

; now compensate for each CPU type and vendor, since every
; type executes instructions with different timings
; (i.e. a divide on a 8088 takes from 144 to 162 clocks
; depending on the values, while a Pentium takes 25 clocks)

        mov     ax, [si+2]         ; get factor
        xor     dx, dx
        shl     ax, 1
        rcl     dx, 1
        shl     ax, 1
        rcl     dx, 1              ; factor * 4
        div     cx                 ; factor adjust (ax=dx:ax/cx)

; return the CPU speed in MHz in AX

cpus_skp2:
        pop     si
        pop     dx
        ret
cpuspeed endp


;
;    FLOATING-POINT DETECTION SUBROUTINE
;       Determines if the math coprocessor is present by
;       checking the FPU's status and control words.  Also
;       detects if a 80386 system has a 80287 or 80387 FPU.
;
;       Called with:    ds = cs (to handle local varible fpu_temp)
;                       ds:[cpu_val] set with CPU type
;
;       Returns:        al = FPU type
;                             0 if no FPU
;                             1 if 8087
;                             2 if 80287
;                             3 if 80387
;                             4 if 80486 with FPU
;                             5 if Pentium with FPU
;
;       Regs used:      ax

fpu_temp dw     0                  ; temp word for detector

.386
.387                               ; allow math instructions

fputype proc    near
        fninit                     ; initialize FPU (reset)
        mov     [fpu_temp], 1234h  ; set any non zero value
        fnstsw  [fpu_temp]         ; get status word from FPU
        and     [fpu_temp], 0FFh   ; only look at bottom 8 bits
        jnz     fput_not_found     ; if non-zero, no FPU
        fnstcw  [fpu_temp]         ; get control word from FPU
        and     [fpu_temp], 103Fh  ; strip unneeded bits
        cmp     [fpu_temp], 3Fh    ; are the proper bits set ?
        je      fput_present       ; jump if so, FPU present

fput_not_found:
        mov     ax, 0              ; FPU not present
        jmp     fput_Exit

; FPU was found, so see which type, 8087, 80287, or 80387

fput_present:
        mov     ax, 1              ; assume 8087
        cmp     [cpu_val], 2       ; CPU below 286 ?
        jb      fput_Exit          ; if so, must be 8087
        mov     ax, 2              ; assume 80287
        je      fput_Exit          ; if 80286, then FPU is 80287
        xor     ah, ah
        mov     al, [cpu_val]      ; get cpu, 80386 or higher
        cmp     al, 3              ; 80386 ?
        jne     fput_Exit          ; jump if 486 or higher

; 80386 could have a 80287 or 80387 FPU.  To find out, check if
;   -infinity is equal to +infinity.  If not, it's an 80387.

        fld1                       ; push +1 onto stack
        fldz                       ; push +0 onto the stack
        fdiv                       ; 1/0 = infinity
        fld     st                 ; load +infinity onto stack
        fchs                       ; now -infinity on stack
        fcompp                     ; compare + and - infinity
        fstsw   [fpu_temp]         ; status of compare in temp
        test    [fpu_temp], 4000h  ; equal ? (test zero bit)
        jnz     fput_Exit          ; jump if not (387, al= 3)
        mov     al, 2              ; +/- infinity equal, 80287
fput_Exit:
        ret
fputype endp
.8086


;
;    FLOATING POINT ON CHIP SUBROUTINE
;       For the 80486/Pentium, a check is made to see if the FPU
;       is on the CPU chip or is separate.  The Extension Type
;       bit 4 in CR0 is checked to see if it can be changed.
;       If it can't be changed, the CPU has the FPU inside the
;       chip.  If the bit can be changed, it's outside the CPU.
;
;                   ********** IMPORTANT **********
;         Changing the CR0 register while in V86 mode causes
;         a CPU fault, that should be handled transparently by
;         the protected mode software.  See book for details.
;
;       Called with:    ds:[fpu_val] set with FPU type
;
;       Returns:        al = 0 if FPU not on CPU or no FPU
;                            1 if FPU inside CPU
;
;       Regs used:      ax

.386P

fpuloc proc     near
        xor     al, al             ; assume FPU not in CPU
        cmp     [fpu_val], 4       ; 486 or higher CPU with FPU?
        jb      fpul_exit          ; exit if not
        push    bx
        push    eax
        mov     eax, cr0
        mov     bx, ax             ; save lower portion
        and     eax, 0FFFFFFEFh    ; attempt to set 16-bit mode
        mov     cr0, eax
        mov     eax, cr0
        xchg    ax, bx
        mov     cr0, eax           ; restore cr0 to original
        pop     eax
        mov     al, bl
        shr     al, 4              ; return ET bit in bit 0
        pop     bx
fpul_exit:
        ret
fpuloc  endp
.8086


;
;    CPU STEP
;       If the CPUID instruction is supported, get info from
;       it, otherwise, check if BIOS supports function that
;       has the CPU step information.  Message displayed
;       depending on results.
;
;       Call with:      ds:[cpu_val] set with CPU
;                       ds:[cpu_info] set with CPUID flag
;
;       Returns:        display step information
;
;       Regs used:      eax, ebx, ecx, edx, di
;
;       Subs called:    hex2ascii, xferbytes, hook_int6,
;                       restore_int6, bad_op_handler

.386P

cpustep proc    near
        cmp     [cpu_val], 3       ; 386 or later ?
        jae     cpust_skp1
        jmp     cpus_no_BIOS_rev

cpust_skp1:
        test    [cpu_info], 1      ; CPUID supported ?
        jz      cpus_noid          ; jump if not
        mov     eax, 0             ; get vendor string function
.586
        CPUID                      ;   for CPUID
.386P
        mov     di, offset idtext
        mov     eax, ebx
        call    xfer_bytes         ; xfer 4 text bytes from eax
        mov     eax, edx
        call    xfer_bytes         ; xfer 4 text bytes from eax
        mov     eax, ecx
        call    xfer_bytes         ; xfer 4 text bytes from eax
;        mov     di,offset rev_loc
;        mov     si,offset idtext
;        call    copy_null
;        OUTMSG  idtxtmg            ; display vendor string

        mov     eax, 1             ; get stepping information
.586
        CPUID                      ;   for CPUID in al
.386P
        call    hex2ascii          ; convert al to ascii in bx
        mov     word ptr modval, bx  ; xfer ascii bytes
        mov     si, offset cpustepmsg
        mov     di, offset STEP_LOC
        call    copy_null

        cmp     [cpu_val],5        ;SHOULDN'T be necessary to explicitly
                                   ;check for a Pentium here, but there
                                   ;were preculiarities on Don's 486SX laptop
        jne     cpus_out_msg
        call    testndp            ; look for Pentium FDIV bug
        jz      cpus_out_msg
        mov     si,offset Penthasbug
        mov     di,offset STEP_LOC + 13
        call    copy_null
        jmp     cpus_out_msg       ; output message

; CPUID not supported, so let's try the BIOS function for the ID.
;   Unfortunately the BIOS function is not supported by most
;   manufacturer's BIOSs.

cpus_noid:
        mov     ax, 0C910h         ; BIOS get chip revision
        int     15h                ;   returned in cx
        jc      cpus_chk_A_step    ; carry if unsuccessful
        jcxz    cpus_chk_A_step    ; 0=not supported
        mov     al, ch
        mov     di, offset bvernum
        call    hex2ascii
        mov     [di], bx
        add     di, 2
        mov     al, cl
        call    hex2ascii
        mov     [di], bx
        add     di, 4
        mov     si, offset bvt     ; get text for step
        mov     dx, offset bver    ; output message


cpus_loop:
        cmp     word ptr [si], 0FFFFh ; at end ?
        je      cpus_out_msg       ; jump if so
        cmp     [si], ax           ; version match ?
        je      cpus_bios_match    ; jump if so
        add     si, offset bvte - offset bvt
        jmp     cpus_loop

cpus_bios_match:
        mov     cx, offset bvte - offset bvt -1
        cld
        rep     movsb              ; xfer CPU & step text
        mov     si,dx
        mov     di,offset step_loc
        call    copy_null
        jmp     cpus_out_msg

; Ok - neither the CPUID or the BIOS supports getting the CPU
;   step, so a few checks may determine if a 386 or 486 is
;   revsion A or if it is a later revision.

cpus_chk_A_step:
        call    hook_int6          ; hook the bad-opcode int
        cmp     [cpu_val], 3       ; 386 only ?
        jne     cpus_rev_486       ; jump if above
;        mov     [badoff], offset cpus_no_rev ; for bad opcode
        mov     [badoff], offset proc_cpus_no_rev ; for bad opcode
        mov     ax, 1
        mov     bx, 1
        mov     cl, 1
        mov     dx, 1
        db      0Fh, 0A6h, 0DAh    ; xbts  bx, dx, ax, al
        nop                        ;  (only valid on 386 A step)
        nop
;        OUTMSG  stepa              ; use Step A message
        mov     si,offset stepa
        mov     di,offset STEP_LOC
        call    copy_null
        jmp     cpus_restore

cpus_rev_486:
        cmp     [cpu_val], 4       ; 486 only ?
        jne     cpus_no_rev        ; jump if not
        mov     [badoff], offset cpus_no_rev ; for bad opcode
        db      0Fh, 0A6h, 0DAh    ; cmpxchg   bx, dx
        nop                        ;  (only valid on 486 A step)
        nop
        mov     si,offset stepa
        mov     di,offset STEP_LOC
        call    copy_null
;        OUTMSG  stepa              ; use Step A message
        jmp     cpus_restore

proc_cpus_no_rev proc near
cpus_no_rev:
        mov     si,offset novermg
        mov     di,offset STEP_LOC
        call    copy_null
;        OUTMSG  novermg            ; no support in BIOS message
cpus_restore:                      ; restore interrupt vector 6
        call    restore_int6       ; restore the int 6 handler
        jmp     cpus_exit

cpus_no_BIOS_rev:
        mov     si,offset novermg
        mov     di,offset STEP_LOC
        call    copy_null
cpus_out_msg:
cpus_exit:
        ret
proc_cpus_no_rev endp
cpustep endp


;
;    TRANSFER BYTES SUBROUTINE
;       Transfer the four bytes in EAX to [di]
;
;       Call with:      eax = source bytes
;                       ds:[di] = destination
;
;       Returns:        4 bytes transferred
;                       di = original di+4
;
;       Regs used:      eax

.386                               ; allow 386 instructions

xfer_bytes proc    near
        push    cx
        mov     cx, 4
xferb_loop:
        mov     [di], al           ; xfer text byte
        inc     di
        ror     eax, 8             ; rotate to next byte
        loop    xferb_loop
        pop     cx
        ret
xfer_bytes endp
.8086


;
;    CPU MODE
;       Check if the 286 or later CPU is in real, protected or
;       V86 mode.  It is assumed that if the 80386 or later
;       processor is in protected mode, we must be in V86 mode.
;
;       Call with:      ds:[cpu_val] set
;
;       Returns:        al = 0 protected mode not supported
;                            1 if real mode
;                            2 if protected mode
;                            3 if V86 mode
;                       ah = privilege level 0 to 3
;
;       Regs used:      ax

.386P                              ; allow 286/386 instructions

cpumode proc    near
        push    cx
        xor     cx, cx             ; assume no protected mode
        cmp     [cpu_val], 2       ; 286 CPU or later ?
        jb      cpum_Exit          ; jump if not
        mov     cx, 1              ; assume real mode flag
        smsw    ax                 ; get machine status word
        test    ax, 1              ; in protected mode ?
        jz      cpum_Exit          ; jump if not (real mode)

cpu_not_real:
        mov     cl, 2              ; protected mode
        pushf
        pop     ax                 ; get flags
        and     ax, 3000h          ; get I/O privilege level
        shl     ax,12
        mov     ch, al             ; save privilege
        cmp     [cpu_val], 2       ; if 286, then protected
        je      cpum_Exit          ; jump if so

; On 386 or later, we have to assume V86 mode.  Note that the
;  next four lines of code (commented out) might seem the
;  correct way to detect V86 mode.  It will not work, since the
;  PUSHFD instruction clears the VM bit before placing it on the
;  stack.  This is undocumented on the 386 and 486, but
;  documented on the Pentium.

;        pushfd                     ; save flags on stack
;        pop     eax                ; get extended flags
;        test    eax, 20000h        ; V86 mode ?
;        jz      cpum_out_mode      ; jump if not

        mov     cl, 3              ; return V86 mode

cpum_Exit:
        mov     ax, cx             ; return status
        pop     cx
        ret
cpumode endp
.8086


;
;    CPU MODE DISPLAY
;       Display results from CPU MODE routine
;
;       Call with:      ax = real/protected mode status from
;                            CPUMODE
;
;       Returns:        display mode information
;
;       Regs used:      ax, bx, dx
;
;       Subs called:    hex2ascii

cpum_display  proc    near
        cmp     al, 0              ; protected mode support?
        je      cpumd_Exit         ; exit if not possible
        mov     si, offset mode_R  ; assume real mode
        cmp     al, 1              ; real ?
        je      cpumd_outmsg       ; jump if so
        mov     si, offset mode_V  ; assume V86
        cmp     al, 3              ; V86 ?
        je      cpumd_outmsg       ; jump if so
        mov     si, offset mode_P  ; must be protected
        mov     al, ah
        call    hex2ascii
        mov     [mode_Pa], bh      ; save privilege level
cpumd_outmsg:
;        mov     ah, 9
;        int     21h                ; display mode message
        mov    di,offset MODE_LOC
        call   copy_null
cpumd_Exit:
        ret
cpum_display endp


;
;    CPU CACHE
;       Check if the later CPU cache is enabled and cache
;       flags.  Cyrix, IBM, and Intel all handle the cache
;       information a little differently.
;
;       Call with:      ds:[cpu_val] set to CPU
;                       ds:[cpu_mfg] set to manufacturer
;
;       Returns:        al = 0 if no cache
;                            1 if cache disabled
;                            2 if enabled, no-write through
;                            3 if enabled, write through
;
;       Regs used:      eax, bx

.486P                              ; allow 486 instructions
cpu_cache proc  near
        xor     bl, bl             ; assume no cache
        cmp     [cpu_val], 3       ; cache only in 486+, and some 386
        jb      cpuc_Exit          ; jump if none
        mov     bh, [cpu_mfg]
        cmp     bh, 6              ; IBM ?
        je      cpuc_ibm           ; jump if so
        cmp     bh, 7              ; Cyrix ?
        je      cpuc_cyrix         ; jump if so

; Use Intel method of detection (486 and later only)

        cmp     [cpu_val], 4       ; cache only in 486+
        jb      cpuc_Exit          ; jump if none
        mov     eax, cr0           ; get control register
        inc     bl                 ; has an internal cache
        test    eax, 40000000h     ; cache disabled ?
        jnz     cpuc_Exit          ; jump if so
        inc     bl                 ; return cache is enabled
        test    eax, 20000000h     ; write through ?
        jnz     cpuc_Exit          ; jump if not
        inc     bl                 ; write-through enabled
        jmp     cpuc_exit

; IBM method of cache enable detection

cpuc_ibm:
        mov     ecx, 1000h         ; get register 1000h
.586P
        RDMSR                      ; read machine specific reg
.486P
        mov     bl, 1              ; has a cache, assume disabled
        test    eax, 80h           ; cache enabled ?
        jz      cpuc_exit          ; jump if not
        mov     bl, 3              ; cache with write-through
        jmp     cpuc_exit

; Cyrix is similar to Intel, but if enabled, always has write-through

cpuc_cyrix:
        mov     eax, cr0           ; get control register
        mov     bl, 1              ; has an internal cache
        test    eax, 40000000h     ; cache disabled ?
        jnz     cpuc_Exit          ; jump if so
        mov     bl, 3              ; write-through enabled

cpuc_Exit:
        mov     al, bl             ; return status in al
        ret
cpu_cache endp
.8086

;
;    CPU CACHE DISPLAY
;       Display status from CPU_CACHE
;
;       Call with:      al = value from CPU_CACHE
;
;       Returns:        display cache information
;
;       Regs used:      dx

cpu_cache_display proc  near
        push    ax
        mov     si, offset cache_X ; assume no cache
        cmp     al, 0
        je      cpuc_msgout        ; no cache
        mov     si, offset cache_D ; assume cache disabled
        cmp     al, 1
        je      cpuc_msgout        ; disabled
        mov     si, offset cache_N ; assume no write-through
        cmp     al, 2
        je      cpuc_msgout        ; no write-through
        mov     si, offset cache_W ; must be write-through
cpuc_msgout:
        mov     di,offset CACHE_LOC
        call    copy_null
        pop     ax
        ret
cpu_cache_display endp

;
;    FIND DATA CACHE SIZE
;       Determine the internal CPU data cache size.  Results
;       will not be valid on a CPU without an internal cache.
;
;       Call with:      nothing
;
;       Returns:        ax = cache size * 1KB
;
;       Regs used:      ax, bx, cx, si, di
;
;       Subs called:    read_n_time

cache_d_size proc  near
        mov     di, offset timings ; array of timings
        mov     bh, 7              ; get 7 sets of values
        mov     bl, 128            ; number of times to re-read
        mov     cx, 256            ; start at .5K (256 words)

chd_loop:
        call    read_n_time        ; read cx words @ ds:0,
                                   ;   returns duration in ax

        mov     [di], ax           ; save value
        add     di, 2
        shr     bl, 1              ; number of group reads/2
        shl     cx, 1              ; number of bytes * 2
        dec     bh
        jnz     chd_loop

        mov     bp, 1              ; cache size 1=1KB
        mov     dx, bp
        mov     di, offset timings
        mov     cx, 6              ; 7 timings
        xor     bx, bx             ; first time - minimal value

chd_loop3:
        mov     ax, [di]
        sub     ax, [di+2]         ; difference between times
        jns     chd_not_neg
        not     ax                 ; get positive difference
chd_not_neg:
        cmp     ax, bx             ; which is larger ?
        jb      chd_skp2
        mov     bx, ax             ; save new large value
        mov     dx, bp             ; save cache size
chd_skp2:
        shl     bp, 1
        add     di, 2
        loop    chd_loop3
        shr     dx, 1              ; adjust back

; now check that there was sufficient difference to matter
;   (i.e. maximum difference should be greater that 5% of first
;   timing value)

        mov     cx, dx             ; save size
        mov     ax, bx             ; bx
        mov     bx, [timings]      ; get 1st timing value
        xor     dx, dx
        div     bx                 ; ax = ax/bx, dx=remainder
        or      ax, ax
        jnz     chd_ok             ; if ax > 0, then valid
        cmp     dx, 5              ; if > 5% ok
        ja      chd_ok             ; jump if so
        xor     cx, cx             ; return zero
chd_ok:
        mov     ax, cx
        ret
cache_d_size endp


;
;    READ AND TIME
;       Read CX words from memory, starting at ds:0.   Re-read
;       the data BL times. Time the duration required for
;       block of reads using hardware timer 2.  Interrupts are
;       disabled during the test to improve results.
;
;       Call with:      cx = number of words to read (0FFFh max)
;                       bl = number of times to repeat read
;
;       Returns:        ax = duration of reads (ax*838nS = time)
;
;       Regs used:      ax

read_n_time  proc    near
        push    dx
        push    si
        mov     al, 0B0h           ; Timer 2 command, mode 0
        out     43h, al            ; send command
        IODELAY
        mov     al, 0FFh           ; counter value FFFF
        out     42h, al            ; send lsb to counter
        IODELAY
        out     42h, al            ; send msb to counter
        IODELAY
        cli                        ; disable interrupts
        in      al, 61h            ; read the current contents
        IODELAY
        or      al, 1              ; set gate bit on
        out     61h, al            ; activate counter
        cld
        push    bx

read_again:
        push    cx
        xor     si, si             ; start from 0
        rep     lodsw              ; read block
        pop     cx
        dec     bl                 ; number of times to cycle
        jnz     read_again

        pop     bx
        in      al, 61h            ; read the current contents
        IODELAY
        and     al, 0FEh           ; set gate bit off
        out     61h, al            ; disable the counter
        sti                        ; enable interrupts

        mov     al, 80h            ; latch output command
        out     43h, al            ; send command
        IODELAY
        in      al, 42h            ; get lsb of counter
        IODELAY
        mov     dl, al
        in      al, 42h            ; get msb of counter
        mov     dh, al             ; dx = counter value
        mov     ax, 0FFFFh         ; starting value
        sub     ax, dx             ; ax = duration count
        pop     si
        pop     dx
        ret
read_n_time endp


;
;    HOOK INTERRUPT 6
;       Save the old interrupt 6 vector and replace it with
;       a new vector to the bad_op_handler.  Vectors are handled
;       directly without using DOS.
;
;       Called with:    nothing
;
;       Returns:        vector hooked
;                       old vector stored at
;                         ds:[old_int6_seg]
;                         ds:[old_int6_off]
;
;       Regs used:      none

hook_int6 proc    near
        push    ax
        push    cx
        push    es
        xor     ax, ax
        mov     es, ax
        cli                        ; disable interrupts
        mov     ax, es:[6*4]       ; get offset of int 6
        mov     cx, es:[6*4+2]     ; get segment
        mov     es:[6*4], offset bad_op_handler
;        mov     word ptr es:[6*4+2], seg bad_op_handler
        mov     word ptr es:[6*4+2], cs
        sti                        ; enable interrupts
        mov     [old_int6_seg], cx ; save original vector
        mov     [old_int6_off], ax
        pop     es
        pop     cx
        pop     ax
        ret
hook_int6 endp


;
;    RESTORE INTERRUPT 6
;       Restore the previously saved old interrupt 6 vector.
;       Vectors handled directly without using DOS.
;
;       Called with:    old vector stored at
;                         ds:[old_int6_seg]
;                         ds:[old_int6_off]
;
;       Returns:        vector restored
;
;       Regs used:      none

restore_int6 proc    near
        push    ax
        push    cx
        push    dx
        mov     cx, [old_int6_seg] ; get original vector
        mov     dx, [old_int6_off]
        push    es
        xor     ax, ax
        mov     es, ax
        cli                        ; disable interrupts
        mov     es:[6*4], dx       ; restore original int 6
        mov     es:[6*4+2], cx
        sti                        ; enable interrupts
        pop     es
        pop     dx
        pop     cx
        pop     ax
        ret
restore_int6 endp


;
;    BAD OFFSET INTERRUPT HANDLER
;       If a bad opcode occurs (80286 or later) will come here.
;       The saved BADOFF offset is used to goto the routine
;       previously stored in BADOFF.
;
;       In a few cases, it is also used for double faults. A few
;       instructions (RDMSR & WRMSR) can issue a double fault if
;       not supported, so well come here as well.
;
;       Called with:    cs:[badoff] previously set
;
;       Returns:        returns to address stored in badoff


bad_op_handler proc far
        push    ax
        push    bp
        mov     ax, cs:[badoff]
        mov     bp, sp
        mov     ss:[bp+4], ax      ; insert new return offset
        pop     bp
        pop     ax
        iret
bad_op_handler endp

;
;    HOOK INTERRUPT D
;       Save the current interrupt mask state, and mask off
;       interrupt D (General Protection Fault).  Also save
;       the old interrupt D vector and replace it with  a new
;       vector to bad_op_handler.  Vectors are handled
;       directly without using DOS.
;
;       NOTE: This is only effective if in Real Mode.  If in
;       V86 mode, the memory manager will not pass the fault
;       into this routine.
;
;       Called with:    nothing
;
;       Returns:        vector hooked
;                       old vector stored at
;                         ds:[old_intD_seg]
;                         ds:[old_intD_off]
;                       interrupt mask stored at
;                         ds:[int_mask]
;
;       Regs used:      none

hook_intD proc    near
        push    ax
        push    cx
        push    es
        xor     ax, ax
        mov     es, ax
        cli                        ; disable interrupts

        in      al, 21h            ; get current interrupt mask
        mov     [int_mask], al     ; save
        IODELAY
        or      al, 20h            ; set bit to disable hardware
        out     21h, al            ;  IRQ 5 (interrupt D)
                                   ; Now change the vector
        mov     ax, es:[0Dh*4]     ; get offset of int D
        mov     cx, es:[0Dh*4+2]   ; get segment
        mov     es:[0Dh*4], offset bad_op_handler
        mov     word ptr es:[0Dh*4+2], cs
        sti                        ; enable interrupts
        mov     [old_intD_seg], cx ; save original vector
        mov     [old_intD_off], ax
        pop     es
        pop     cx
        pop     ax
        ret
hook_intD endp


;
;    RESTORE INTERRUPT D
;       Restore the previously saved old interrupt D vector
;       and restore the interrupt mask to it's prior state
;       Vectors handled directly without using DOS.
;
;       Called with:    old vector stored at
;                         ds:[old_intD_seg]
;                         ds:[old_intD_off]
;                       interrupt mask stored at
;                         ds:[int_mask]
;
;       Returns:        vector restored
;
;       Regs used:      none

restore_intD proc    near
        push    ax
        push    cx
        push    dx
        mov     cx, [old_intD_seg] ; get original vector
        mov     dx, [old_intD_off]
        push    es
        xor     ax, ax
        mov     es, ax
        cli                        ; disable interrupts
        mov     es:[0Dh*4], dx     ; restore original int D
        mov     es:[0Dh*4+2], cx
                                   ; Now restore IRQ 5 mask
        in      al, 21h            ; get current interrupt mask
        mov     ah, [int_mask]     ; get previous mask
        IODELAY
        or      ah, 0DFh           ; set all but IRQ 5 bit
        and     al, ah             ; insert old IRQ 5 state
        out     21h, al            ;  IRQ 5 (interrupt D)

        sti                        ; enable interrupts
        pop     es
        pop     dx
        pop     cx
        pop     ax
        ret
restore_intD endp


;
;    DECW
;       Convert the hex number in ax into decimal 1 to 5 ascii
;       characters and insert into [di].  Increment di ptr.  The
;       leading zeros are suppressed.
;
;       Called with:    ax = input hex number
;                       di = pointer where to store characters
;                       bl = 0 for left justification
;                            1 for no justification
;
;       Returns:        word converted to ascii at [di]
;
;       Regs used:      bx
;
;       Subs called:    hex2ascii

decw    proc    near
        push    ax
        push    cx
        push    dx
        cmp     ax, 0              ; check for zero
        jne     decskip0           ; jump if not
        mov     al, 4              ; if justify, make ax = 0
        mul     bl                 ;    no justify, ax = 4
        add     di, ax             ; move pointer
        mov     byte ptr [di], '0' ; put up ascii zero
        jmp     decskip15          ; done !

decskip0:
        xor     cl, cl             ; temp flag, 0=suppression on
        mov     ch, bl             ; save flag (0=left justify)
        xor     dx, dx             ; zero
        mov     bx, 10000
        div     bx                 ; (labelcnt)/10000
        cmp     al, 0              ; 10000's ?
        je      decskip1           ; jump if zero
        inc     cl                 ; no longer zero suppression
        call    hex2ascii          ; convert to ascii
        mov     byte ptr [di], bh  ; put 1000's digit in
        jmp     decskip2
decskip1:
        cmp     ch, 0              ; left justify ?
        je      decskip3           ; jump if so
decskip2:
        inc     di
decskip3:
        mov     ax, dx             ; get remainder
        xor     dx, dx             ; zero
        mov     bx, 1000
        div     bx                 ; (labelcnt)/1000
        cmp     cl, 0              ; zero suppression active ?
        jne     decskip3a          ; jump if not
        cmp     al, 0              ; 1000's ?
        je      decskip4           ; jump if zero
decskip3a:
        inc     cl                 ; no longer zero suppression
        call    hex2ascii          ; convert to ascii
        mov     byte ptr [di], bh  ; put 1000's digit in
        jmp     decskip5
decskip4:
        cmp     ch, 0              ; left justify ?
        je      decskip6           ; jump if so
decskip5:
        inc     di
decskip6:
        mov     ax, dx             ; get remainder
        xor     dx, dx             ; zero
        mov     bx, 100
        div     bx                 ; (remainder in dx)/100
        cmp     cl, 0              ; zero suppression active ?
        jne     decskip7           ; jump if not
        cmp     al, 0              ; zero ?
        je      decskip8           ; suppress zero
decskip7:
        inc     cl                 ; no longer zero suppression
        call    hex2ascii          ; convert to ascii
        mov     byte ptr [di], bh  ; put 100's digit in
        jmp     decskip9
decskip8:
        cmp     ch, 0              ; left justify ?
        je      decskip10          ; jump if so
decskip9:
        inc     di
decskip10:
        mov     ax, dx             ; get remainder
        xor     dx, dx             ; zero
        mov     bx, 10
        div     bx                 ; (remainder in dx)/10
        cmp     cl, 0              ; zero suppression active ?
        jne     decskip11          ; jump if not
        cmp     al, 0              ; zero ?
        je      decskip12          ; suppress zero
decskip11:
        call    hex2ascii          ; convert to ascii
        mov     byte ptr [di], bh  ; put 10's digit in
        jmp     decskip13
decskip12:
        cmp     ch, 0              ; left justify ?
        je      decskip14          ; jump if so
decskip13:
        inc     di
decskip14:
        mov     al, dl             ; get 1's digit (remainder)
        call    hex2ascii          ; convert to ascii
        mov     byte ptr [di], bh  ; put 1's digit in output
decskip15:
        inc     di
        pop     dx
        pop     cx
        pop     ax
        ret
decw    endp


;
;   HEX2ASCII
;       Convert the hex number in al into two ascii characters
;
;       Called with:    al = input hex number
;
;       Returns         bx = converted digits in ascii
;
;       Regs Used:      al, bx

hex2ascii       proc    near
        mov     bl, al
        and     al, 0fh
        add     al, 90h
        daa
        adc     al, 40h
        daa
        mov     bh, al

        mov     al, bl             ; upper nibble
        shr     al, 1
        shr     al, 1
        shr     al, 1
        shr     al, 1
        and     al, 0fh
        add     al, 90h
        daa
        adc     al, 40h
        daa
        mov     bl, al             ; bx has two ascii bytes
        ret
hex2ascii       endp
.8086

;
;    SYSTEM TYPE DETECTION SUBROUTINE
;       Determine the type of system the software is running
;       on.
;
;       Called with:    nothing
;
;       Returns:        al = System type
;                             0 if PC (8088 based)
;                             1 if XT (8088 based)
;                             2 if PC convertible (8088 based)
;                             3 if PC jr (8088 based)
;                             4 other pre-80286 based machine
;                             8 if XT (80286 based)
;                            10h if AT or ISA
;                            20h if EISA
;                            40h if MCA
;
;       Regs used:      ax, bx
;                       eax, ebx (386 or later)
;
;       Subs called:    cpuvalue
assume cs:_text, ds:_text, es:_text
sysvalue proc    near
        push    cx
        push    dx
        push    es

        call    cpuvalue   ; get the cpu type in al
        mov     cl, al             ; save cpu number (0 to 5)

; Avoid directly reading BIOS ROM, because a few memory managers
; like 386MAX alter bytes at the end of the BIOS.

        push    cx                 ; save cpu number on stack
        mov     ah, 0C0h
        int     15h                ; get BIOS config data es:bx
        pop     cx
        jc      sys_skp1           ; jump if no config support
                                   ;   (old BIOS)
        mov     dl, es:[bx+2]      ; get model byte
        mov     dh, es:[bx+3]      ; get submodel byte

        mov     al, 40h            ; assume MCA
        test    byte ptr es:[bx+5], 2
        jnz     sys_Exit           ; exit if MCA
        jmp     sys_skp2

; we only get here on older PCs in which a memory manager
;  can not be run

sys_skp1:                          ; ok, get BIOS model directly
        mov     ax, 0F000h
        mov     es, ax             ; point into system BIOS
        mov     dx, es:[0FFFEh]    ; get model & submodel byte

; now use the model and submodel bytes im DX to determine machine

sys_skp2:
        xor     al, al             ; assume PC (al=0)
        cmp     dl, 0FFh
        je      sys_Exit           ; jump if PC
        inc     al                 ; assume XT (al=1)
        cmp     dl, 0FEh
        je      sys_Exit           ; jump if XT
        cmp     dl, 0FBh
        je      sys_Exit           ; jump if XT
        inc     al                 ; assume PC convertible (al=2)
        cmp     dl, 0F9h
        je      sys_Exit           ; jump if convertible
        inc     al                 ; assume PCjr (al=3)
        cmp     dl, 0FDh
        je      sys_Exit           ; jump if PCjr
        inc     al                 ; assume other pre-286 (al=4)
        cmp     cl, 2              ; cl=CPU type - pre-286 ?
        jb      sys_Exit           ; jump so
        ja      sys_skp3           ; jump if 386 or above

; possible a 286 XT - use the model and submodel bytes to
;  determine

        mov     al, 8              ; assumption for 286XT
        cmp     dx, 02FCh          ; model code for 286XT ?
        je      sys_exit           ; jump if so

; check if EISA system by looking for the "EISA" string at
;  address F000:FFD9

sys_skp3:
        mov     ax, 0F000h
        mov     es, ax
        mov     al, 10h            ; assume a standard AT/ISA
        cmp     word ptr es:[0FFD9h], 'IE'
        jne     sys_exit           ; jump if not EISA
        cmp     word ptr es:[0FFDBh], 'AS'
        jne     sys_exit           ; jump if not EISA
        mov     al, 20h            ; EISA machine
sys_Exit:
        pop     es
        pop     dx
        pop     cx
        ret
sysvalue endp




miscundocdisp  proc near

        call    sysvalue           ; get system value in al
        mov     cx, offset sysend - offset sysmsg
        mov     si, offset sysmsg
syst_loop:
        cmp     byte ptr [si], 0FFh  ; no match ?
        je      syst_skp1          ; exit if so
        cmp     al, [si]           ; system type match ?
        je      syst_skp1          ; exit if so
        add     si, cx
        jmp     syst_loop          ; loop until match or end

syst_skp1:
        inc     si
        dec     cx
        mov     di, offset systypm ; where to insert type text
        cld
        rep     movsb              ; xfer message
        mov     di,offset BUS_LOC
        mov     si,offset systyp             ; display system type
        call    copy_null

        call    sertype
        ret

miscundocdisp endp



;
;                           SERIAL TYPE
;
;
;   Reports current serial port I/O addresses, checks the port
;   has a UART attached, and determines the basic chip type.
;
;   (c) Copyright 1994  Frank van Gilluwe  All Rights Reserved.
;
;Modified for DF 5.00 to fill in the commport arrays rather than
;print out the results
;-------------------------------------------------------------------------
; Get the base addresses and UART type of the comm ports
; Fills in two arrays:
;  commport_addr    (4) word
;  commport_type    (4) byte
;                        0:  8250
;                        1:  16450
;                        2:  16550
;                             ...


sertype proc near
        mov     cx,0               ;serial port number - 1
ser_loop:

     mov       ax,040h   ;BIOS segment
     push      es
     mov       es,ax
     mov       bx,cx
     shl       bx,1
     mov       dx,es:[bx]  ;1st 4 words in 040h are commport addr

     mov       si,offset commport_addr
     add       si,bx
     mov       [si],dx     ;store commport addr in array
     pop       es

     cmp     dx, 0              ; port number present ?
     je      ser_next           ; jump if not

     cmp       mswin_version,0
     je        ser_100             ;MSWIN 3.1+ causes unreliable test results
     mov       bx,0
     jmp       short ser_200

ser_100:
        push    cx
        call    testport           ; test that serial port is
                                   ;  there and chip type
        pop     cx
ser_200:
        mov     dx,bx
                                   ;0 = none, 1=8250 etc.
        mov     bx,offset commport_type
        add     bx,cx
        mov     [bx],dl



ser_next:
        inc     cx
        cmp     cx, 4              ; are we done yet?
        jb      ser_loop           ; jump if so

ser_exit:
        ret
sertype endp


;
;    Testport
;       Check if the UART is responding and if so, test to
;       find out which chip it is equivalent to.
;
;       Called with:    dx = serial I/O port
;
;       Returns:        bl = chip type
;                            0 = no response, not a UART
;                            1 = 8250 (no scratch register)
;                            2 = 8250A or 16450
;                            3 = 16C1450
;                            4 = 16550 with defective FIFO
;                            5 = 16550AF/C/CF with FIFO
;                            6 = 16C1550 with FIFO
;                            7 = 16552 dual UART with FIFO
;                            8 = 82510 with FIFO

testport proc   near


        mov     bl, 0              ; return value if no-response
        mov     di, dx             ; save I/O port
        cli                        ; disable interrupts
        call    DLAB1              ; Divisor Latch Access bit=1
        mov     ah, 5Ah            ; test value 1
        call    IOtest             ; write value, read it and compare
        jne     test_exit2         ; exit if not valid
        mov     ah, 0A5h           ; test value 2
        call    IOtest             ; write value, read it and compare
        jne     test_exit2         ; exit if not valid
        sti                        ; enable interrupts

; now check if the scratch pad register is working (missing on 8250)

        inc     bl                 ; return 1 if 8250
        add     dx, 7              ; point to scratch pad register
        mov     ah, 5Ah            ; test value 1
        call    IOtest             ; write value, read it and compare
        jne     test_exit2         ; exit if no scatch register
        mov     ah, 0A5h           ; test value 2
        call    IOtest             ; write value, read it and compare
        je      scratch_ok
test_exit2:
        jmp     test_exit          ; exit if no scratch register

scratch_ok:
        mov     dx, di             ; restore base I/O port

; check for 1655x and 82510 FIFO versions

        cli                        ; disable interrupts
        add     dx, 2              ; interrupt ID register
        in      al, dx
        IODELAY
        and     al, 0C0h
        cmp     al, 0C0h           ; is 16550 FIFO already enabled ?
        je      is_1655x           ; jump if so

        mov     ah, 1              ; try to enable FIFO on 1655x
        out     dx, al             ;  (if 82510 also set bank 0)
        IODELAY
        in      al, dx             ; read Interrupt Identification
        IODELAY
        mov     bh, al             ; save readback value
        mov     al, 0              ; Disable FIFO on 16550 or
        out     dx, al             ;   select bank 0 on 82510
        IODELAY
        and     bh, 0C0h
        cmp     bh, 0C0h           ; was FIFO enabled on 16550 ?
        je      is_1655x           ; jump if so
        cmp     bh, 40h            ; FIFO UART in some PS/2s ?
        je      is_16550C
        cmp     bh, 80h            ; 16550 with defective FIFO ?
        je      is_16550

; not 16550, so now check for possible 82510 by switching to bank
;  3 (which is only possible on a 82510)

        mov     al, 60h            ; select bank 3
        out     dx, al
        IODELAY
        in      al, dx             ; read back from port
        IODELAY
        mov     bh, al             ; save for later

        mov     al, 0              ; if 83510, return to
        out     dx, al             ;  bank 0
        IODELAY
        and     bh, 60h            ; now, did it goto bank 3?
        cmp     bh, 60h
        je      is_82510           ; jump if so (is 82510)

; no FIFO, so UART is 8250A or 16450 variant - Check if power down
;   bit functions on this UART, in the modem control register

no_FIFO:
        mov     dx, di
        add     dx, 4              ; select modem control register
        mov     al, 80h            ; see if power down mode allowable
        out     dx, al
        IODELAY
        in      al, dx             ; get modem register
        IODELAY
        mov     ah, al             ; save result
        mov     al, 0
        out     dx, al             ; turn power back on
        test    ah, 80h            ; did power down bit get set ?
        jnz     is_16C1450         ; jump if so

is_8250A:
        mov     bl, 2              ; set to 8250A/16450 UART
        jmp     test_exit

is_16C1450:
        mov     bl, 3              ; set to 16C1450 UART
        jmp     test_exit

is_16550:
        mov     bl, 4              ; set to 16550 with defective FIFO
        jmp     test_exit

; FIFO detected, and must be 16550 series

is_1655x:
        mov     dx, di
        call    DLAB1
        add     dx, 2
        mov     ah, 7
        call    IOtest             ; write value, read it
        mov     bh, al             ; save result for later
        mov     al, 0              ; reset register select
        out     dx, al
        cmp     bh, 7              ; did it work ?
        je      is_16552           ; if compare ok, is 16552

; Check if power down bit functions on this UART, in the modem
;   control register (only on 16C1550)

        mov     dx, di
        add     dx, 4              ; select modem control register
        mov     al, 80h            ; see if power down mode allowable
        out     dx, al
        IODELAY
        in      al, dx             ; get modem register
        IODELAY
        mov     ah, al             ; save result
        mov     al, 0
        out     dx, al             ; turn power back on
        test    ah, 80h            ; did power down bit get set ?
        jnz     is_16C1550         ; jump if so

is_16550C:
        mov     bl, 5              ; set to 16550AF/C/CF UART
        jmp     test_exit

is_16C1550:
        mov     bl, 6              ; set to 16C1550
        jmp     test_exit

is_16552:                          ; set to 16552 UART
        mov     bl, 7
        jmp     test_exit

is_82510:
        mov     bl, 8              ; set to 82510 UART

test_exit:
        call    DLAB0              ; reset DLAB to 0
        sti                        ; enable interrupts in case off
        ret
testport endp


;
;    DLAB1
;       Set the Divisor Latch Access Bit to 1 without
;       changing other line control bits
;
;       Called with:    dx = serial I/O port
;
;       Regs Used:      al

DLAB1   proc    near
        push    dx
        add     dx, 3              ; point to Line Control Reg
        in      al, dx             ; get current state
        IODELAY
        or      al, 80h            ; set DLAB to 1
        out     dx, al
        pop     dx
        ret
DLAB1   endp


;
;    DLAB0
;       Set the Divisor Latch Access Bit to 0 without
;       changing other line control bits
;
;       Called with:    dx = serial I/O port
;
;       Regs Used:      al

DLAB0   proc    near
        push    dx
        add     dx, 3              ; point to Line Control Reg
        in      al, dx             ; get current state
        IODELAY
        and     al, 7Fh            ; set DLAB to 0
        out     dx, al
        pop     dx
        ret
DLAB0   endp


;
;    IOtest
;       Write value al to port, then read back value and compare
;       to original value.
;
;       Called with:    ah = value to use for test
;                       dx = I/O port to test
;
;       Returns:        al = value read from port
;                       zero flag = 1 if register ok

IOtest  proc    near
        mov     al, ah             ; value to test
        out     dx, al             ; write value
        IODELAY
        in      al, dx             ; read back value
        IODELAY
        cmp     al, ah             ; compare if the same
        ret
IOtest  endp

;
;    DISKETTE DRIVE TYPE DETECTION SUBROUTINE
;       Determine the type of diskette drive
;
;       Called with:    dl = drive to check (0=a:, 1=b:, etc.)
;
;       Returns:        al = drive type
;                               0 = no drive
;                               1 = 360KB
;                               2 = 1.2MB
;                               3 = 720KB
;                               4 = 1.44MB
;                               5 = 2.88MB
;                               6 = 320KB  (obsolete)
;                               7 = unknown type
;
;       Regs used:      ax

dsktype proc    near
        push    bx
        push    cx
        push    dx
        push    es

        int     11h                ; equipment check
        test    ax, 1              ; any drives ?
        jz      dskt_none          ; jump if not
        and     al, 0C0h           ; get # of drives bits
        mov     cl, 2
        rol     al, cl             ; convert to # of drives
        cmp     al, dl             ; is drive available ?
        jb      dskt_none          ; jump if not

; first we'll try getting the drive parameters from the BIOS

        mov     ah, 8
        int     13h                ; get drive parameters
        jc      dskt_skp1          ; jump if failed
        mov     al, bl             ; get drive type
        cmp     al, 5              ; unknown type if > 5
        jbe     dskt_done
        mov     al, 7              ; set to unknown type
        jmp     dskt_done

; Older systems can't supply drive info, so use diskette table.
;   First the drive is reset.  This forces systems that change
;   the diskette parameter pointer for different drive types
;   to update the pointer to the correct drive type. Interrupt
;   vector 1Eh now points to the correct diskette parameter
;   table.  Only 360KB and 1.2MB drives should get here.

dskt_skp1:
        xor     ah, ah
        int     13h                ; reset drive
        xor     ax, ax
        mov     es, ax
        mov     bx, es:[1Eh*4]     ; get offset of table
        mov     ax, es:[1Eh*4+2]   ; get segment of table
        mov     es, ax             ; es:bx points to table
        mov     ah, es:[bx+4]      ; get sectors per track
        mov     al, 1              ; assume 360K
        cmp     ah, 9              ; 9 sectors per track ?
        je      dskt_done          ; jump if so
        inc     al
        cmp     ah, 15             ; 15 sectors per track
        je      dskt_done          ; jump if so
        mov     al, 6              ; unknown type
        cmp     ah, 8              ; 8 sectors per track
        je      dskt_done          ; jump if so
        inc     al                 ; al=7, unknown type
        jmp     dskt_done

dskt_none:
        xor     al, al
dskt_done:
        pop     es
        pop     dx
        pop     cx
        pop     bx
        ret
dsktype endp

;
;    HARD DISK DRIVE TYPE DETECTION SUBROUTINE
;       Determine if a hard drive is attached, its likely type,
;       and the capacity of that drive.
;
;       Called with:    ds = cs
;                       dl = drive to check, 80h=drive 0, etc.
;
;       Returns:        al = drive type
;                               0 = no drive or no controller
;                               1 = XT type controller
;                               2 = AT type controller
;                               3 = SCSI type controller
;                               4 = PS/2 ESDI type controller
;                               5 = unknown controller type
;                       cx = total number of cylinders (includes
;                              the diagnostic cylinder)
;                       dh = total number of heads (1 to 64)
;                       dl = number of 512 byte sectors
;       Regs used:      ax

; local data for routine

hd_parm_seg     dw      0               ; pointer to disk
hd_parm_off     dw      0               ;  parameter table
hd_cylinder     dw      0               ; temp # from table
ESDIbuf         db      1024 dup (0)    ; ESDI info buffer

hdsktype proc   near
        push    bx
        push    es
        xor     bp, bp             ; used for temp drive type
        cmp     dl, 80h            ; at least 80h ?
        jb      hdsk_skp1          ; exit if not

; first we will get the number of drives attached

        push    dx
        mov     dl, 80h            ; ask for drive 0
        mov     ah, 8
        int     13h                ; read disk drive parameters
        pop     ax
        add     dl, 7Fh
        cmp     al, dl             ; drive for this number ?
        jbe     hdsk_skp2
hdsk_skp1:
        jmp     hdsk_none          ; jump if out of range

; now determine the controller/disk type

hdsk_skp2:
        inc     bp                 ; bp=1 assume XT type drive
        mov     dl, al             ; dl = drive number
        push    dx
        mov     ah, 15h
        int     13h
        pop     dx
        cmp     ah, 1              ; invalid request status ?
        je      hdsk_skp4          ; if so, XT type & exit
        inc     bp                 ; set type to AT
        cmp     ah, 3              ; confirm valid hard disk
        je      hdsk_skp3
        jmp     hdsk_none          ; exit if not

; let's check if it's a drive which does not use the drive
;   parameter table like SCSI and some ESDI drives.

hdsk_skp3:
        inc     bp                 ; assume SCSI
        cmp     dl, 82h            ; if 82h to 87h, assume SCSI
        jb      hdsk_skp5
hdsk_skp4:
        jmp     hdsk_info          ; jump if so

hdsk_skp5:
        mov     bx, 4*41h          ; assume ptr to vector 41h
        cmp     dl, 80h            ; drive 0 ?
        je      hdsk_skp6          ; jump if so
        mov     bx, 4*46h          ; pointer to vector 46h
hdsk_skp6:
        xor     ax, ax
        mov     es, ax
        mov     si, es:[bx]        ; offset of parameter table
        mov     ax, es:[bx+2]      ; segment of parameter table
        mov     [hd_parm_seg], ax  ; save pointer
        mov     [hd_parm_off], si
        mov     es, ax             ; es:si ptr to table

; we now have a pointer to the disk parameter table, but there
;   are two types!  Check if word at offset 3 is 0 or -1, then
;   use normal table (cylinder at offset 0).  Otherwise use
;   PS/2 style, where cylinder is at offset 19h.

        mov     ax, es:[si+3]      ; get word to check
        cmp     ax, 0
        je      hdsk_skp7
        cmp     ax, -1
        je      hdsk_skp7
        add     si, 19h            ; adjust to PS/2 style
hdsk_skp7:
        mov     ax, es:[si]        ; get cylinders from table
        mov     [hd_cylinder], ax  ; save
        cmp     ax, 0              ; invalid # of cylinders ?
        jne     hdsk_skp8          ; jump if ok (non-zero)
        jmp     hdsk_info          ; we will assume SCSI

hdsk_skp8:
        dec     bp                 ; assume AT type
        push    dx
        mov     ah, 8
        int     13h                ; get disk parameters
        call    hdconvert          ; convert into useful values
        pop     dx
        cmp     ax, [hd_cylinder]  ; are they the same ?
        jne     hdsk_skp9
        jmp     hdsk_info          ; if so, likely AT type

; Not likely AT type, since cylinders do not match up!
;  First we'll try an older Future Domain SCSI test

; Future Domain SCSI test - Interrupt 13h, function 18h will
;  return invalid command with DL >= 80h, if no Future Domain
;  SCSI card is present

hdsk_skp9:
        inc     bp                 ; assume SCSI
        push    dx
        mov     ah, 18h
        int     13h                ; Future Domain SCSI ?
        pop     dx
        jc      hdsk_skp10         ; jump if not
        cmp     ax, 4321h          ; confirmation number
        jne     hdsk_skp10
        jmp     hdsk_info

; Now check if possible PS/2 ESDI drive

hdsk_skp10:
        push    cs
        pop     es
        push    dx
        mov     ax, 1B0Ah
        mov     bx, offset ESDIbuf
        int     13h                ; Get ESDI config
        pop     dx
        jc      hdsk_skp11         ; jump if not PS/2 ESDI
        cmp     bx, offset ESDIbuf
        jne     hdsk_skp11
        mov     ax, cs
        mov     bx, es
        cmp     ax, bx             ; is CS = ES ?
        jne     hdsk_skp11         ; jump if not
        inc     bp
        jmp     hdsk_info

; General SCSI test (not drive specific)

hdsk_skp11:
        xor     ax, ax
        mov     es, ax
        cmp     word ptr es:[4*4Fh+2], 0  ; vector valid ?
        je      hdsk_skp12         ; jump if not
        push    dx
        mov     ax, 8200h
        mov     cx, 8765h
        mov     dx, 0CBA9h
        int     4Fh                ; check for SCSI CAM
        cmp     dx, 5678h
        pop     dx
        jne     hdsk_skp12         ; jump if not
        cmp     ah, 0
        jne     hdsk_skp12         ; jump if not
        cmp     cx, 9ABCh
        jne     hdsk_skp12         ; jump if not
        jmp     hdsk_info

hdsk_skp12:
        inc     bp                 ; Indicate unknown
                                   ;  controller type

; now get the disk parameter for the specified drive

hdsk_info:
        mov     ah, 8
        int     13h                ; read disk drive parameters
        call    hdconvert          ; convert to useful value
        mov     dl, cl
        mov     cx, ax             ; put in proper registers

        mov     ax, bp             ; get drive type
        jmp     hdsk_exit



hdsk_none:
        xor     al, al             ; no drive with this number
        xor     cx, cx
        xor     dx, dx
hdsk_exit:
        pop     es
        pop     bx
        ret
hdsktype endp


;
;    HARD DISK CONVERT
;
;       The cylinder number is in cx and part of dh.  This
;       routine properly combines them back into a 12 bit
;       cylinder number in AX. The number is increased by
;       one for the diagnostic cylinder, and by one more, since
;       the number is zero based. The upper two bits in CL are
;       cleared to get the sector number.   The upper two bits
;       of DH are cleared to get the head number.  One is added
;       to the head number, since it is zero based.
;
;       Called with:    ch = lower 8 bits of cylinder number
;                       cl upper two bits are
;                             bits 9 & 8 of cylinder
;                          lower six bits is sector number
;                       dh upper two bits are
;                             bits 11 & 10 of cylinder
;                          lower six bits are head number
;
;       Returns:        ax = total combined cylinders
;                       cl = sector number
;                       dh = total heads
;
;       Regs used:      cx, dh

hdconvert proc   near
        push    dx
        push    cx
        mov     ax, cx             ; get fragmented cylinder #
        rol     al, 1
        rol     al, 1
        and     al, 3
        xchg    al, ah             ; cx = # of cylinders
        and     dh, 0C0h           ; get undoc cylinder bits
        mov     cl, 4
        shl     dh, cl             ; shift to 10 & 9th bits
        or      ah, dh
        add     ax, 2              ; ax = real max cylinders
        pop     cx
        and     cx, 3Fh            ; mask for sector number
        pop     dx
        and     dh, 3Fh            ; mask for heads
        inc     dh                 ; adjust for head count
        ret
hdconvert endp